/*
 * A featured example program for the APEG add-on library, using OpenGL
 */

#include <string.h>
#include <stdio.h>

#include <allegro.h>
#include <alleggl.h>

#ifdef _allegrogl_included_alleggl_h

#include <GL/glu.h>

#include "apeg.h"

#ifdef _MSC_VER
#define strcasecmp stricmp
#endif

static char **global_argv;
static int global_argc;

static int check_param(const char *param)
{
	int i;
	for(i = global_argc-1;i >= 2;--i)
	{
		if(strcasecmp(global_argv[i], param) == 0)
			return i;
	}

	return 0;
}

int main(int argc, char *argv[])
{
	APEG_STREAM *stream;
	void *mpeg_dat;
	int showed = 0, skipped = 0;
	int ret, depth;
	float w = 0.0f, h = 0.0f;
	int sw, sh;
	int render_spin = FALSE;
	GLuint tex_id;

	if(allegro_init() != 0)
	{
		fprintf(stderr, "Error initializing Allegro: %s\n", allegro_error);
		return -1;
	}

	install_allegro_gl();

	if(argc < 2)
	{
		allegro_message("Usage: %s <filename> [options]\n"
		                "  <filename> can be either an MPEG/Ogg file or an embedded datafile object.\n"
		                "  [-memplay] load the mpeg into memory before playing.\n"
						"  [-speed x] playback at x times speed.\n"
		                "  [-downsample x] downsample audio to the power of x (0,1,2;default: 0).\n"
		                "  [-downchannel] force mono audio output.\n"
		                "  [-largeaudiobuffer] use a large audio buffer (for CPU intense streams).\n"
		                "  [-nosound] disable audio playback.\n"
		                "  [-quality x] use quality level x (0,1,2;default: 2).\n"
		                "  [-framedrop] enable frame-dropping.\n"
		                "  [-fs] fullscreen playback, suitable for watching.\n"
		                "\n"
		                "The default playback mode is more of an OpenGL showoff than a mode suitable\n"
		                "for watching. Using fullscreen mode with -fs is more suitable for watching.\n",
		                argv[0]);
		return 1;
	}

	global_argv = argv;
	global_argc = argc;

	install_timer();
	install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL);
	set_volume_per_voice(0);

	// Set gfx depth
	depth = desktop_color_depth();
	if(!depth)
		depth = 32;

	// Force 32-bit RGBA decoding
	apeg_set_display_depth(32);

	// Set video decode quality
	if((ret = check_param("-quality")) > 0)
	{
		++ret;
		if(argc > ret)
		{
			int q = atoi(argv[ret]);
			apeg_set_quality(q);
		}
	}

	// Set downsampling rate
	if((ret = check_param("-downsample")) > 0)
	{
		++ret;
		if(argc > ret)
		{
			int d = atoi(argv[ret]);
			apeg_downsample_audio(d);
		}
	}

	// Set downchanneling
	if(check_param("-downchannel"))
		apeg_downchannel_audio(TRUE);

	// Enable smooth playback
	if(check_param("-framedrop"))
		apeg_enable_framedrop(TRUE);

	if(check_param("-largeaudiobuffer"))
		apeg_set_audio_bufsize(apeg_set_audio_bufsize(-1)*3);

	// Disable audio stream?
	if(check_param("-nosound"))
		apeg_ignore_audio(TRUE);

	// Optionally load the stream into memory, then open it
	mpeg_dat = NULL;
	if(check_param("-memplay"))
	{
		int size = file_size(argv[1]);

		if(size > 0)
		{
			mpeg_dat = malloc(size);
			if(mpeg_dat)
			{
				PACKFILE *pf = pack_fopen(argv[1], F_READ);
				if(pf)
				{
					size = pack_fread(mpeg_dat, size, pf);
					pack_fclose(pf);
					if(size < 0)
					{
						free(mpeg_dat);
						mpeg_dat = NULL;
					}
				}
				else
				{
					free(mpeg_dat);
					mpeg_dat = NULL;
				}
			}
		}

		if(!mpeg_dat)
		{
			set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
			allegro_message("Couldn't load %s!\n", argv[1]);
			return APEG_ERROR;
		}

		stream = apeg_open_memory_stream(mpeg_dat, size);
	}
	else
		stream = apeg_open_stream(argv[1]);

	if(!stream)
	{
		allegro_message("APEG Error: %s\n", apeg_error);
		return APEG_ERROR;
	}

	if(!(stream->flags & APEG_HAS_VIDEO))
	{
		allegro_message("No video in '%s'\n", argv[1]);
		apeg_close_stream(stream);
		return APEG_ERROR;
	}

	apeg_get_video_size(stream, &sw, &sh);

	if((stream->flags & APEG_MPG_VIDEO))
		fprintf(stderr, "Video: MPEG-1\n"
		                " %dx%d (%.2f) => %dx%d\n %.02ffps\n %dkbps\n",
		        stream->w, stream->h, stream->aspect_ratio, sw, sh,
		        stream->frame_rate, stream->bit_rate/1000);
	else if((stream->flags & APEG_THEORA_VIDEO))
		fprintf(stderr, "Video: Theora\n"
		                " %dx%d (%.2f) => %dx%d\n %.02ffps\n",
		        stream->w, stream->h, stream->aspect_ratio, sw, sh,
		        stream->frame_rate);

	if((stream->flags & APEG_MPG_AUDIO))
		fprintf(stderr, "Audio: MPEG layer-%d\n"
		                " %dhz\n %d channels\n %dkbps\n",
		        stream->audio.layer, stream->audio.freq,
		        stream->audio.channels, stream->audio.kbps);
	else if((stream->flags & APEG_VORBIS_AUDIO))
		fprintf(stderr, "Audio: Vorbis\n"
				        " %dhz\n %d channels\n",
		        stream->audio.freq, stream->audio.channels);


	allegro_gl_set(AGL_RENDERMETHOD, 1);
	allegro_gl_set(AGL_SUGGEST, AGL_RENDERMETHOD);

	// Check if the user wants fullscreen playback
	if(check_param("-fs"))
	{
		int w, h, r;

		// Try to duplicate the current resolution
		if(get_desktop_resolution(&w, &h) != 0)
		{
			w = 640;
			h = 480;
		}

		if((r = get_refresh_rate()))
			request_refresh_rate(r);

		// Set fullscreen mode now
		set_color_depth(depth);
		if(set_gfx_mode(GFX_OPENGL_FULLSCREEN, w, h, 0, 0) != 0)
		{
			allegro_message("Couldn't set mode %ix%ix%i:\n%s\n", w, h, depth,
			                allegro_error);
			apeg_close_stream(stream);
			return -1;
		}
		render_spin = FALSE;
	}

	if(!gfx_driver)
	{
		set_color_depth(depth);
		if(set_gfx_mode(GFX_OPENGL_WINDOWED, 640, 480, 0, 0) != 0)
		{
			allegro_message("Couldn't create 640x480x%i window:\n%s\n",
			                depth, allegro_error);
			apeg_close_stream(stream);
			return -1;
		}
		render_spin = TRUE;
	}


	install_keyboard();

	// Reset APEG's color tables since we just entered a valid video mode
	apeg_reset_colors(stream);

	// Set the timer speed
	if((ret = check_param("-speed")) > 0)
	{
		++ret;
		if(ret < argc)
		{
			float m = atof(argv[ret]);
			if(m > 0.0f)
				apeg_set_stream_rate(stream, m);
		}
	}

	// Try to set a reasonable background mode
	if(set_display_switch_mode(SWITCH_BACKGROUND) == -1)
		set_display_switch_mode(SWITCH_BACKAMNESIA);

	// Generate an OpenGL texture, along with scale and offset info
	if(1)
	{
		char *data;
		int x, y;

		// Set a good matrix
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		if(!render_spin)
		{
			float aspect;
			if(stream->aspect_ratio > 0.0)
				aspect = stream->aspect_ratio - (float)SCREEN_W/(float)SCREEN_H;
			else
				aspect = (float)stream->w/(float)stream->h -
				         (float)SCREEN_W/(float)SCREEN_H;

			if(aspect > 0.0)
				glOrtho(-1.0, 1.0, 1.0+aspect, -1.0-aspect, -1, 1);
			else if(aspect < 0.0)
				glOrtho(-1.0-aspect, 1.0+aspect, 1.0, -1.0, -1, 1);
			else
				glOrtho(-1.0, 1.0, 1.0, -1.0, -1, 1);
		}
		else
			gluPerspective(60.0f, 1.0, 2.0, 4000.0);

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

		glCullFace(GL_FRONT);
		glEnable(GL_CULL_FACE);

		// Make a power-of-2 texture size (OpenGL might not like it
		// otherwise)
		x = stream->bitmap->line[1] - stream->bitmap->line[0];
		x |= x>>1; x |= x>>2;
		x |= x>>4; x |= x>>8;
		x |= x>>16; x++;

		y = stream->h - 1;
		y |= y>>1; y |= y>>2;
		y |= y>>4; y |= y>>8;
		y |= y>>16; y++;

		// Allocate dummy texture data (so we can create the texture)
		data = malloc(x*y*4);
		if(!data)
			return 1;

		// Create the texture w/ linear filtering (don't use mipmapping!)
		glEnable(GL_TEXTURE_2D);

		glGenTextures(1, &tex_id);
		glBindTexture(GL_TEXTURE_2D, tex_id);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

		// Tell OpenGL to store it as RGBA 8.8.8.8, since it would save on
		// any real-time conversion the driver would have to do
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA8, x, y, 0, GL_RGBA,
		             GL_UNSIGNED_BYTE, data);

		free(data);

		// Get width and height offset points for the video image
		w = stream->w / (float)x;
		h = stream->h / (float)y;
	}

	// Rest a second.. sometimes a disk access will occur from all the loading.
	rest(1000);

	// Reset the video timer (it has been running since we opened the mpeg).
	stream->timer = -1;
	while(stream->timer == -1)
		;

	// Get the next video frame
	while((ret = apeg_advance_stream(stream, FALSE)) == APEG_OK)
	{
		// Stop if a key is pressed
		if(keypressed())
			break;

		// If the bitmap was updated, update the screen, else mark it as
		// skipped, or just rest a tad
		if(stream->frame_updated > 0)
		{
			// Update texture image data
			glTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, stream->bitmap->w,
			                stream->bitmap->h, GL_RGBA, GL_UNSIGNED_BYTE,
			                stream->bitmap->line[0]);

			++showed;
		}
		else
		{
			// Missed frame
			if(stream->frame_updated == 0)
				++skipped;

			// No change yet
			if(stream->frame_updated < 0 && !stream->audio.flushed)
				rest(1);

			continue;
		}

		glClear(GL_COLOR_BUFFER_BIT);
		glLoadIdentity();
		if(render_spin)
		{
			// Draw some colored rotating screens
			glTranslatef(0, 0, -stream->w/2);
			glRotatef(stream->frame*3.14159f/(stream->frame_rate/4.0f), 0, 1, 0);

			glBegin(GL_QUADS);
				glColor3f(1, 1, 1);
				glTexCoord2f(w, 0);
				glVertex3f(stream->w/2, stream->h/2, -stream->w/2);
				glTexCoord2f(w, h);
				glVertex3f(stream->w/2, -stream->h/2, -stream->w/2);
				glTexCoord2f(0, h);
				glVertex3f(-stream->w/2, -stream->h/2, -stream->w/2);
				glTexCoord2f(0, 0);
				glVertex3f(-stream->w/2, stream->h/2, -stream->w/2);

				glColor3f(1, 0, 0);
				glTexCoord2f(w, 0);
				glVertex3f(stream->w/2, stream->h/2, stream->w/2);
				glColor3f(0, 0, 1);
				glTexCoord2f(w, h);
				glVertex3f(stream->w/2, -stream->h/2, stream->w/2);
				glColor3f(1, 1, 1);
				glTexCoord2f(0, h);
				glVertex3f(stream->w/2, -stream->h/2, -stream->w/2);
				glTexCoord2f(0, 0);
				glVertex3f(stream->w/2, stream->h/2, -stream->w/2);

				glTexCoord2f(w, 0);
				glVertex3f(-stream->w/2, stream->h/2, -stream->w/2);
				glTexCoord2f(w, h);
				glVertex3f(-stream->w/2, -stream->h/2, -stream->w/2);
				glColor3f(0, 0, 1);
				glTexCoord2f(0, h);
				glVertex3f(-stream->w/2, -stream->h/2, stream->w/2);
				glColor3f(0, 1, 0);
				glTexCoord2f(0, 0);
				glVertex3f(-stream->w/2, stream->h/2, stream->w/2);

				glTexCoord2f(w, 0);
				glVertex3f(-stream->w/2, stream->h/2, stream->w/2);
				glColor3f(0, 0, 1);
				glTexCoord2f(w, h);
				glVertex3f(-stream->w/2, -stream->h/2, stream->w/2);
				glTexCoord2f(0, h);
				glVertex3f(stream->w/2, -stream->h/2, stream->w/2);
				glColor3f(1, 0, 0);
				glTexCoord2f(0, 0);
				glVertex3f(stream->w/2, stream->h/2, stream->w/2);
			glEnd();
		}
		else
		{
			// Draw a fullscreen quad
//			glTranslatef(((float)SCREEN_W-(SCREEN_W*w_mult))/2.0f, ((float)SCREEN_H-(SCREEN_H*h_mult))/2.0f, 0);
			glColor3f(1, 1, 1);
			glBegin(GL_QUADS);
				glTexCoord2f(w, 0);
				glVertex2f(1.0, -1.0);

				glTexCoord2f(w, h);
				glVertex2f(1.0, 1.0);

				glTexCoord2f(0, h);
				glVertex2f(-1.0, 1.0);

				glTexCoord2f(0, 0);
				glVertex2f(-1.0, -1.0);
			glEnd();
		}

		allegro_gl_flip();
	}
	apeg_set_stream_rate(stream, 0.0);

	// APEG_EOF == no error
	if(ret == APEG_EOF)
		ret = APEG_OK;

	// If error, print a message, else print some stats.
	set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
	if(ret == APEG_ERROR)
		allegro_message("APEG Error: %s\n", apeg_error);
	else if((stream->flags & APEG_HAS_VIDEO) && showed+skipped > 0)
		allegro_message("---------------------------\n"
		                "Playback Statistics\n"
		                "Frames Shown:  %i\n"
		                "Frames Skipped: %i\n"
		                "Average Speed: %.2ffps (%.1f%%)\n\n",
		                showed, skipped,
		                showed * stream->frame_rate / (float)(showed+skipped),
		                showed * 100.0f / (float)(showed+skipped));

	// Close the mpeg
	apeg_close_stream(stream);
	free(mpeg_dat);

	return ret;
}
END_OF_MAIN()

#else
#error - Not building the AllegroGL example
#endif
