\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : A simulated lift
\ CATEGORY    : Examples
\ AUTHOR      : Coos Haak 
\ LAST CHANGE : July 13, 1994, Coos Haak 
\ ----------------------------------------------------------------------



        ?DEF -lift [IF] -lift [THEN]


        MARKER -lift


DOC

See Vijgeblad 3 (1982) page 38-43.

Structure of a lift instance:
| x-pos | y-pos | asked y | waited | passenger |

Create a lift with 'n LIFT: NAME'. n is the horizontal position.

NAME displays the lift and leaves a flag that is true when it is
stopped at the correct position and the waiting period is over.
This is an example of cooperative multitasking controlled by the
user, not the CPU, so executing NAME at regular intervals is
required to keep things running.

You can give the command 'n TO NAME' to send the lift to another
floor.

LEVEL NAME gives the current level, accurate if NAME gives true.

ENDDOC


statoff

privates

\ : tone          ( x1 x2 -- )                  \ Use this if the sound
\        2drop                                  \ irritates you
\        ;  private

20 constant waiting                     private \ steps to wait

0 value pointer                         private \ points to structure

doer: doindex       ( addr -- x )               \ Index into a lift structure
        @ pointer []cell @
        ;  private

: toindex
        inline# @ pointer []cell !
        ;  private

: +toindex
        inline# @ pointer []cell +!
        ;  private

methods doindex

: to                ( x "name" -- )             \ Set new value
        postpone toindex l,
        ;

: +to               ( x "name" -- )             \ Change value
        postpone +toindex l,
        ;

end-methods

: index             ( x "name" -- )             \ Index into a lift structure
        create  , private                       \ Offset
        doindex
        ;  private

0 index .x                                      \ horizontal position
1 index .y                                      \ vertical position
2 index .ask                                    \ asked position
3 index .wait                                   \ waited on new level
4 index .pas                                    \ passenger flag

: top           ( -- )                          \ top line of screen
        .x 0 at-xy
        ;  private

: at            ( y -- )                        \ y is from bottom up
        l/scr 1- swap - .x swap at-xy
        ;  private

: display               ( addr -- )             \ display lift
        to pointer
        .y 3 + at ." /-------\"
        .pas
        if      .y 2 + at ." |  \O/  |"
                .y 1+  at ." |  _X_  |"
        else    .y 2 + at ." |       |"
                .y 1+  at ." |       |"
        then
        .y     at ." \-------/"
        ;  private

: hide                  ( addr -- )             \ undisplay lift
        to pointer
        .y 3 + at ."          "
        .y 2 + at ."          "
        .y 1+  at ."          "
        .y     at ."          "
        ;  private

: sound         ( -- )
        1 .y 200 * 500 + tone
        ;  private

doer: dolift        ( a-addr -- flag )          \ display lift and get status
        to pointer
        .ask .y <                               \ go lower ?
        if      pointer hide
                -1 +to .y
                waiting to .wait                \ we have to wait
                sound
                top ."   DOWN  "
                .y 4 + at ."    | |   "
        else    .ask .y >                       \ go higher ?
                if      pointer hide
                        1 +to .y
                        waiting to .wait        \ we have to wait
                        top ."    UP   "
                        sound
                else    top ."   FREE  "
                then
        then
        pointer display                         \ current position
        .wait 0=                                \ no more waiting and
        .y .ask =                               \ reached position ?
        and
        if      2 choose to .pas true
        else    -1 +to .wait false
        then
        ;  private

: tolift                ( y -- )                \ set position
        inline# to pointer to .ask
        ;  private

: dolevel
        inline# to pointer .y
        ;  private

: lift:         ( x -- )                        \ a lift instance
        create  , 0 , 0 , waiting , 0 ,         \ build it
        dolift                                  \ use it --> flag
        ;

methods dolift

: to            ( position "name" -- )          \ set position
        postpone tolift l,
        ;

: level         ( "name" -- x )                 \ get level
        postpone dolevel l,
        ;

end-methods

prefix level

deprive

 0 lift: lift1
10 lift: lift2
20 lift: lift3
30 lift: lift4
40 lift: lift5
50 lift: lift6
60 lift: lift7
70 lift: lift8

: randomfloor
        5 choose 0=
        if      0
        else    l/scr 4 / choose 4 *
        then
        ;

: run
        page hide-cursor
        l/scr 5 - to lift1 l/scr 5 - to lift2 l/scr 5 - to lift3
        l/scr 5 - to lift4 l/scr 5 - to lift5 l/scr 5 - to lift6
        l/scr 5 - to lift7 l/scr 5 - to lift8
        begin   lift1 lift2 lift3 lift4 lift5 lift6 lift7 lift8
                and and and and and and and
        until
        begin   lift1
                if      randomfloor to lift1
                then
                lift2
                if      randomfloor to lift2
                then
                lift3
                if      randomfloor to lift3
                then
                lift4
                if      randomfloor to lift4
                then
                lift5
                if      randomfloor to lift5
                then
                lift6
                if      randomfloor to lift6
                then
                lift7
                if      randomfloor to lift7
                then
                lift8
                if      randomfloor to lift8
                then
                100 ms
                stop?
        until
\        0 to lift1 0 to lift2 0 to lift3 0 to lift4
\        0 to lift5 0 to lift6 0 to lift7 0 to lift8
\        begin   lift1 lift2 lift3 lift4 lift5 lift6 lift7 lift8
\                and and and and and and and
\        until
        show-cursor 0 l/scr 2 - at-xy
        ;

                            \ (* End of Source *) /
