\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : Meta compiler for CHForth version 1.1.0 
\ FILENAME    : ERRORS.FRT 
\ DESCRIPTION : Exception handling and forgetting
\ AUTHOR      : Coos Haak, Utrecht 
\ CREATED     : May 11, 1994
\ ----------------------------------------------------------------------



INTERNAL:

TH> T.FGTMES
: F:MESS
        BODY> @ TO MESS-LINK
        ;  COMPILE-ONLY

EXTRA:

\G Parse ccc delimited by a " (double-quote) and compile the string
\G in the dictionary. The string is displayed when n is passed to
\G .MESS or THROW .
: MESS"         ( n "ccc<quote>" -- )                   \ EXTRA "mess-quote"
        S" " HEAD, REVEAL HIDDEN
        ['] F:MESS LAST @ HEAD>FORGET H!
        HERE MESS-LINK , TO MESS-LINK , ",
        ;

\ Standard ANS messages

#-3 MESS" stack overflow"
#-4 MESS" stack underflow"
#-5 MESS" returnstack overflow"
#-6 MESS" returnstack underflow"
#-8 MESS" dictionary overflow"
#-9 MESS" invalid memory address"
#-10 MESS" division by zero"
#-11 MESS" result out of range"
#-13 MESS" undefined word"
#-14 MESS" interpreting a compile-only word"
#-15 MESS" invalid FORGET"
#-16 MESS" attempt to use zero-length string as a name"
#-22 MESS" control structure mismatch"
#-25 MESS" returnstack imbalance"
#-28 MESS" user interrupt"
#-29 MESS" compiler nesting"
#-32 MESS" invalid name argument"
#-33 MESS" block read exception"
#-34 MESS" block write exception"
#-35 MESS" invalid block number"
#-36 MESS" invalid file position"
#-37 MESS" file I/O exception"
#-38 MESS" non-existent file"
#-49 MESS" search-order overflow"
#-50 MESS" search-order underflow"
\ #-52 MESS" control-flow stack overflow"
#-57 MESS" exception in sending or receiving a character"
#-58 MESS" missing terminating [ELSE] or [THEN]"

\ Messages of this Forth system

#-513 MESS" is not unique"
#-514 MESS" execution halted"
#-515 MESS" wrong use of DPSWAP"
#-516 MESS" no defining word"
#-517 MESS" not defining methods"
#-518 MESS" is undefined, compiling forward reference"
#-519 MESS" listsegment full"
#-520 MESS" headersegment full"
#-521 MESS" program contains errors"
#-522 MESS" local stack overflow"
#-523 MESS" local stack underflow"
#-524 MESS" illegal opcode for this processor"
#-525 MESS" unresolved forward definition"
#-526 MESS" no special routine for this character"
#-527 MESS" is not portable "
#-528 MESS" or part of it is not yet implemented"
#-529 MESS" is in a non-portable number format "
#-530 MESS" already defining methods"
#-531 MESS" character can not be converted"
#-532 MESS" missing terminating ENDDOC"
#-533 MESS" missing terminating *)"

\ DOS messages

FE01 MESS" function number invalid"
FE02 MESS" file not found"
FE03 MESS" path not found"
FE04 MESS" too many open files (no handles available)"
FE05 MESS" access denied"
FE06 MESS" invalid handle"
FE07 MESS" memory control blocks destroyed"
FE08 MESS" insufficient memory"
FE09 MESS" memory block address invalid"
FE0A MESS" environment invalid"
FE0B MESS" format invalid"
FE0C MESS" access code invalid"
FE0D MESS" data invalid"
FE0F MESS" invalid drive"
FE10 MESS" attempted to remove current directory"
FE11 MESS" not same device"
FE12 MESS" no more files"
FE13 MESS" disk write-protected"
FE14 MESS" unknown unit"
FE15 MESS" drive not ready"
FE16 MESS" unknown command"
FE17 MESS" data error (CRC)"
FE18 MESS" bad request structure length"
FE19 MESS" seek error"
FE1A MESS" unknown media type (non-DOS disk)"
FE1B MESS" sector not found"
FE1C MESS" printer out of paper"
FE1D MESS" write fault"
FE1E MESS" read fault"
FE1F MESS" general failure"
FE20 MESS" sharing violation"
FE21 MESS" locking violation"
FE22 MESS" disk change invalid"
FE23 MESS" FCB unavailable"
FE24 MESS" sharing buffer overflow"
FE3D MESS" print queue full"
FE3E MESS" queue not full"
FE3F MESS" not enough space to print file"
FE50 MESS" file exists"
FE52 MESS" cannot make directory"
FE53 MESS" fail on INT 24h"
FE54 MESS" too many redirections"
FE55 MESS" duplicate redirections"
FE56 MESS" invalid password"
FE57 MESS" invalid parameter"

ORPHAN (RESET)                                  \ reset stacks and the like
                MOV     AX, CS
                MOV     DS, AX                  \ reset DS to CS
                MOV     T' SRCSEG AX            \ and source seg. (see strings)
                CLI
                MOV     SS, stkseg              \ set stack segment
                MOV     SP, T' SP0              \ data stack
                MOV     BP, T' RP0              \ returnstack
                STI
                MOV     AX, T' LSP0             \ locals stack
                MOV     lsp AX
                XOR     BX, BX                  \ top of stack is zero (opt)
                MOV     catcher BX              \ top catch frame      (opt)
                MOV     ES, lstseg              \ listsegment are colon defs.
                NEXT
END-CODE

ORPHAN DOSBK                    \ interrupt $23 is revectored to this
                PUSH    AX
                MOV     AH, # 0         \ read key, ^C
                INT     16              \ and throw it away
                POP     AX
                CLC                     \ signal DOS to ignore
                RET     FAR
END-CODE

ORPHAN BIOSBK                   \ interrupt $1B is revectored to this
                PUSH    AX
                PUSH    DS
                PUSH    CS
                POP     DS
                MOV     AX, # 02EB      \ jump over NEXT code
                CALL    brkit
                POP     DS
                POP     AX
                IRET
LABEL brkit
                MOV     !_doc AX
                MOV     !_doe AX
                MOV     !_rep AX
                MOV     !_agn AX
                MOV     !_loo AX
                RET
LABEL __brk
                MOV     AX, # AD26      \ reset jump to do NEXT
                CALL    brkit
                GOTO    _abrt
                $EVEN
END-CODE

HERE _abrt !-T
:ORPHAN ABNORM
        (LIT) T[ T' COLD ,-L T] @ (LIT) DFTCOLD =
        IF      #-28 SHOW-ERROR             \ No message in TURNKEY
        THEN
        TRUE TO RESTART?                    \ Signal a Ctrl-Break
        COLD
        ;

HERE _rpov !-T
:ORPHAN ROVERFL
        #-5 THROW ;

HERE _rpun !-T
:ORPHAN RUNDERFL
        #-6 THROW ;

HERE _spov !-T
:ORPHAN OVERFL
        #-3 THROW ;

HERE _spun !-T
:ORPHAN UNDERFL
        #-4 THROW ;

HERE _lsov !-T
:ORPHAN LOVFL
        #-522 THROW ;

HERE _lsun !-T
:ORPHAN LUNFL
        #-523 THROW ;

ORPHAN DIVERR                   \ interrupt 0 is revectored to this
                STI
                GOTO    _div0
                $EVEN
END-CODE

HERE _div0 !-T
:ORPHAN _DIVERR
        #-10 THROW ;

ORPHAN OPCERR                   \ interrupt 6 is revectored to this
                STI                     \ only possible on 286 and higher
                GOTO    _opco
                $EVEN
END-CODE

HERE _opco !-T
:ORPHAN _OPCERR
        #-524 THROW ;

HERE _prter !-T
:ORPHAN _PRTER
        #-57 THROW ;

:ORPHAN UNRES
        #-525 THROW ;

INTERNAL:

\ Print the errorline and type markers under the word. Any control
\ characters in the source line (notably tabs) are printed as
\ spaces to keep the line and the markers aligned.
: MARK-WORD             ( -- )                          \ EXTRA
        ERR# -1 = ERR# -2 = OR
        IF      EXIT                                    \ ABORT and ABORT"
        THEN
        PARSED-WORD SWAP SOURCE DROP -                  \ length word, offset
        BLK @                                           \ loading block ?
        IF      #64 /MOD #64 *                          \ fit into line
                CR SOURCE DROP + #64 TYPE               \ type source line
        ELSE    CR SOURCE 0                             \ type source line
                ?DO     COUNT BL MAX EMIT               \ tabs as space
                LOOP                                    \ for alignment
                DROP
        THEN                                            \ aligned ^^^
        CR C/L UMIN SPACES                              \ offset spaces
        0                                               \ length word
        ?DO     '^' EMIT                                \ markers
        LOOP
        ;

\ Find the message string that is assigned to exception number n and
\ return a true flag. Else return a false flag.
: FINDMESSAGE           ( n -- false )          \ EXTRA
        TO ERR#
        MESS-LINK                               \ walk through the messages
        BEGIN   DUP CELL+ @ ERR# =              \ match error number
                IF      2 CELLS + TO ERR$
                        TRUE EXIT               \ found
                THEN
                @ ?DUP 0=
        UNTIL
        FALSE                                   \ not found
        ;

EXTRA:

\G Display the message that is assigned to exception number n as
\G with MESS" . If the message is not found, display the exception
\G number and the name of the word where the exception occured. If n
\G is -1 or -2 nothing is displayed. Store the number in ERR# .
: .MESS                 ( n -- )                        \ EXTRA
        TO ERR#                                         \ Keep exception number
        VIDEO                                           \ output only to screen
        ERR# -1 =
        IF      EXIT                                    \ ABORT
        THEN
        ERR# -2 =
        IF      CR ERR$ COUNT TYPE EXIT                 \ ABORT"
        THEN
        CR ERR# FINDMESSAGE                             \ find message
        IF      PARSED-WORD TYPE SPACE                  \ known error
                ERR$ COUNT TYPE
        ELSE    PUSH BASE DECIMAL                       \ decimal now
                ." Exception " ERR# .                   \ type exception number
                POP BASE                                \ restore
                ." in " PARSED-WORD TYPE                \ unknown error
        THEN
        ;

\G If the last exception occurred during loading of a file, display
\G the name of the file and the line number where the exception
\G occurred.
: .WHERE                ( -- )                          \ EXTRA
        SOURCE-ID 0>                                    \ only while loading
        IF      'NAME ERRNAME ! #LINES @ ERRLINE !      \ save info
                CR ." Exception in "
                ERRNAME @ COUNT TYPE                    \ type filename
                PUSH BASE DECIMAL                       \ decimal now
                ." , line " ERRLINE @ .                 \ type linenumber
                POP BASE                                \ restore
        THEN
        ;

\G Show the type of the last exception number stored in ERR# by
\G .MESS . Display nothing if ERR# equals -1 or -2.
: ERROR-TYPE            ( -- )                          \ EXTRA
        ERR# -1 = ERR# -2 = OR
        IF      EXIT                                    \ ABORT and ABORT"
        THEN
        PUSH BASE DECIMAL
        CR ERR# DUP FF00 AND CASE
            FF00 OF ." ANSI"            ENDOF
            FE00 OF ." MS-DOS"  FF AND  ENDOF
            FD00 OF ." CHForth"         ENDOF
            ." General"
        ENDCASE
        ."  exception " .
        POP BASE
        ;

\G Display the exception message and information where the exception
\G with number n occurred and the type of the exception and display
\G the source line with the exception word marked out.
: SHOW-ERROR            ( n -- )                        \ EXTRA
        .MESS                                           \ show message
        .WHERE                                          \ Where did it occur
        ERROR-TYPE                                      \ show type of error
        MARK-WORD                                       \ show marked line
        LOG-ERROR                                       \ may be logged
        ;

\G Abort with exception message: not implemented, used in some
\G definitions.
: NOT-IMPLEMENTED       ( -- )          \ EXTRA
        #-528 THROW                             \ I am reluctant to define this
        ;

\G Check the current depth of the stack with the one stored by !CSP
\G Exception -29 will occur when they do not match.
: ?CSP          ( -- )                  \ EXTRA "question-c-s-p"
        DEPTH (LIT) _csp @ <> #-29 ?ERROR       \ stack is not same level
        ;

\G Save the current depth of the stack for checking with ?CSP .
: !CSP          ( -- )                  \ EXTRA "store-c-s-p"
        DEPTH (LIT) _csp !                      \ save level of stack
        ;

\G Check x1 and x2. Exception -22 occurs when they are not equal.
: ?PAIRS        ( x1 x2 -- )            \ EXTRA "question-pairs"
        <> #-22 ?ERROR                                  \ error when different
        ;

FORTH:

\G Push an exception frame on the exception stack and then execute
\G the execution token xt (as with EXECUTE ) in such a way that
\G control can be transferred to a point just after CATCH if THROW
\G is executed during the execution of xt.
\G
\G If the execution of xt completes normally (i.e. the exception
\G frame pushed by this CATCH is not popped by an execution of THROW
\G ) pop the execution frame and return zero on top of the data
\G stack, above whatever stack items would have been returned by xt
\G EXECUTE . Otherwise, the remainder of the execution semantics are
\G given by THROW .
: CATCH         ( i*x xt -- j*x 0 | i*x n )     \ FORTH
        SP@ >R LSP@ >R (LIT) catcher @ >R       \ save stack pointers
        RP@ (LIT) catcher !                     \ remember level of returnstack
        EXECUTE                                 \ execute word
        R> (LIT) catcher ! R>DROP R>DROP FALSE  \ throw catch frame away
        ;  ANS

\G If any bits of n are non-zero, pop the topmost exception frame
\G from the exception stack, along with everything on the return
\G stack above that frame. Then restore the input source
\G specification in use before the corresponding CATCH and adjust
\G the depths of all three stacks so that they are the same as the
\G depth saved in the exception frame (i is the same number as i in
\G the input arguments to the corresponding CATCH ), put n on top of
\G the data stack, and transfer control to a point just after the
\G CATCH that pushed that exception frame.
: THROW         ( k*x n -- k*x | i*x n )        \ FORTH
        ?DUP                                    \ throw zero and continue
        IF      (LIT) catcher @ RP!             \ otherwise restore stacks
                R> (LIT) catcher !              \ and go to after CATCH
                R> LSP! R> SWAP >R SP! DROP R>
        THEN
        ;  ANS

EXTRA:

\G If x is not zero, exception n occurs. Else drop both numbers
\G from the stack and continue.
: ?ERROR        ( x n -- )                      \ EXTRA "question-error"
        SWAP                                    \ get flag on top
        IF      THROW                           \ when true, do THROW
        ELSE    DROP                            \ else drop error number x
        THEN
        ;

INTERNAL:

CREATE NIL$ 0 ,

FORTH:

\G ( i*x -- ) ( R: j*x -- )
\G Perform the function of -1 THROW . When no other exception frame
\G is present other than the one pushed by QUIT , empty the stacks
\G and perform QUIT . When no file is currently open, display no
\G message. Otherwise, contrary to the Standard, display some
\G information about the file and the line where ABORT was called.
\G Store a zero-length string in ERR$ .
: ABORT                                 \ FORTH
        NIL$ TO ERR$                            \ No messages
        -1 THROW                                \ See ANS Forth manual
        ;  ANS

INTERNAL:

\ Only used at the start of QUIT to reset everthing
: START-FORTH       ( -- )
        CLOSE-ALL-FILES TERMINAL
        HERE LIMIT U>                                   \ are pointers within
        LHERE LLIMIT U> OR                              \ own areas?
        HHERE HLIMIT U> OR              
        IF      DPSWAP #-515 .MESS                      \ issue message, now OK
        THEN
        FROM STATE [
        IF      LAST @ DUP HDP ! HEAD> DP !
        THEN
        ;

EXTRA:

\ Invented by Willem Ouwerkerk, find the higest (last defined) definition
\ in any word list.
\ Now a very neat, historically responsible FORGET is possible

\G Return the dictionary entry address of the newest definition with
\G dictionary entry address dea and the word list identification wid
\G in which it is compiled. Used in FORGET .
: HIGHEST       ( -- wid dea )          \ EXTRA
        0 0 2>R                                 \ initial values
        VOC-LINK                                \ scan all vocabularies
        BEGIN   DUP VOC@ R@ U>                  \ is top word the newest
                IF      R>DROP R>DROP           \ drop previous pair
                        DUP DUP VOC@ 2>R        \ keep wid and dea as tags
                THEN
                @ ?DUP 0=                       \ not at end of VOC-LINK
        UNTIL
        2R>                                     \ get the two pointers
        ;

:ORPHAN (DOFORGET)      ( -- )                  \ the main forget routine

        ['] ?CRASH >BODY @ _DP @ U>             \ sometimes you say EMPTY
        IF      ['] THROW ['] ?CRASH >BODY !    \ and then ..
        THEN

        'ACCEPT @ _DP @ U>                      \ anyway, ACCEPT is expected
        IF      ['] MINIACCEPT 'ACCEPT !        \ to work !
        THEN

        BEGIN   HIGHEST DUP _HDP @ U< INVERT    \ a word is to be forgotten
        WHILE   2>R R@ HEAD>FORGET H@           \ does it have a forget routine
                IF      R@ HEAD> >BODY          \ >BODY on stack
                        R@ HEAD>FORGET H@       \ and execute routine
                        EXECUTE
                THEN
                R> H@ R> VOC!                   \ unlink word
        REPEAT
        2DROP                                   \ drop wid and dea

        (LIT) ^VECT DUP                         \ scan all VECTORs
        BEGIN   @ DUP _DP @ U<                  \ unlink words above HERE
        UNTIL
        OVER !                                  \ ready

        BEGIN   @ ?DUP                          \ some may have routines
        WHILE                                   \ in their bodies that
                BEGIN   DUP CELL- @             \ must be forgotten
                        _DP @ U< INVERT
                WHILE   DUP CELL- DUP @ >BODY @ L@ SWAP !
                REPEAT
        REPEAT

        BL 0                                    \ reset []KEY vectors
        DO      I []KEY @ _DP @ U< INVERT       \ for control keys
                IF      I []KEY OFF
                THEN
        LOOP

        0 100                                   \ reset []KEY vectors
        DO      I []KEY @ _DP @ U< INVERT       \ for extended keys
                IF      I []KEY OFF
                THEN
                100
        +LOOP

        _DP @ DP ! _HDP @ HDP !                 \ ready, set pointers
        ;

\G Forget the definition with execution token xt.
: (FORGET)      ( xt -- )               \ EXTRA "paren-forget"
        DUP FENCE @ U< #-15 ?ERROR              \ don't forget below fence
        DUP >HEAD DUP 0= #-15 ?ERROR            \ this one has no header
        DUP (LIT) ENDHEADS @ U< #-15 ?ERROR     \ another fence
        _HDP ! _DP ! DOFORGET                   \ ok, set pointers and do it
        ;

\G Perform the function of FORGET on all definitions that were
\G compiled after the last execution of EMPTY , EXTEND or SAVE .
: EMPTY         ( -- )                  \ EXTRA
        FENCE @ _DP !                           \ set pointers to default
        (LIT) ENDHEADS @ _HDP !                 \ values
        DOFORGET
        HERE BYTES !                            \ reset some variables
        LHERE LBYTES !
        HHERE HBYTES !
        ;

\G Mark all definition so that they can not be forgotten.
: EXTEND        ( -- )                  \ EXTRA
        HERE FENCE ! HHERE (LIT) ENDHEADS !     \ extend the system
        EMPTY
        ;

FORTH:

\G Skip leading space delimiters. Parse name delimited by a space.
\G Find name in the compilation word list, then delete name from the
\G dictionary along with all words added to the dictionary after
\G name. Exception -13 occurs if name cannot be found. Exception -15
\G occurs if FORGET removes a word required for correct execution.
\G
\G Note: this word is obsolescent and is included as a concession to
\G existing implementations.
\G
\G Note: In CHForth words can be protected against FORGET with
\G EXTEND and SAVE .
: FORGET        ( "name" -- )                   \ FORTH
        ' (FORGET)                              \ find word and forget it
        ;  ANS

                            \ (* End of Source *) /
