\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : Generic output to a file 
\ CATEGORY    : Utilities 
\ AUTHOR      : Coos Haak 
\ LAST CHANGE : June 11, 1994, Coos Haak, FAPPEND 
\ LAST CHANGE : April 25, 1994, Coos Haak 
\ ----------------------------------------------------------------------



        MARKER -outfile


DOC

Create an output file with
        S" filename.ext" FOPEN
Send data to it with
        'C' FEMIT               a character
        S" string" FTYPE        a string
        FCR                     a new line
Close it with
        FCLOSE

FOPEN discards an old file with the same name.
FAPPEND uses an old file and append the data at the end of it.

The stack mechanism allows one to have more files open at one time, but
opening a new file causes the output to go to that file.
When you close the file, the output will go again to the previous file.

Remember to use FCLOSE for each file before you leave Forth. It can be made
automatically, but programming it is not worth the trouble.

ENDDOC

privates

#1024 constant /filebuffer              private \ size of file buffer

#4 constant /filestack                  private \ maximum count of files
#3 cells constant /stackitem            private \ cells on stack
                                                \ segment,pointer,handle

create filestack                        private
        here /filestack /stackitem * dup allot erase

/filestack value fsp                    private \ file stack pointer

: []file        ( -- addr )                     \ get current item address
        fsp /stackitem * filestack +
        ;  private

: outfile       ( -- addr )                     \ get address of handle
        []file 2 cells +
        ;  private

: bufferpointer ( -- addr )                     \ get address of pointer
        []file cell+
        ;  private

: buffersegment ( -- addr )                     \ get address of segment
        []file
        ;  private

: flushbuffer           ( -- )                  \ send buffer
        buffersegment @ 0                       \ x-addr
        bufferpointer @                         \ len
        outfile @                               \ fid
        writex-file throw                       \ write from extended address
        bufferpointer off                       \ reset pointer
        ;  private

\G Write a character to a file opened by FOPEN / FAPPEND .
: FEMIT                 ( c -- )                \ OUTFILE
        bufferpointer @ /filebuffer =           \ buffer full?
        if      flushbuffer                     \ write it to file
        then
        buffersegment @ bufferpointer @ c!x     \ put it in the buffer
        bufferpointer incr                      \ increment pointer
        ;

#-622 mess" output file stack overflow"
#-623 mess" output file stack underflow"

\G Create a file and append text to it with FTYPE FCR and FEMIT .
\G Uses the file stack created with FOPEN / FAPPEND and FCLOSE .
: FOPEN                 ( c-addr u -- )         \ OUTFILE
        fsp 0= #-622 ?error                     \ stack full message
        -1 +to fsp                              \ push stack
        /filebuffer alloc throw buffersegment ! \ get a segment
        bufferpointer off                       \ new buffer
        w/o create-file throw                   \ create the file
        outfile !                               \ save handle
        ;

\G Open an existing file and append text to it with FTYPE FCR and
\G FEMIT . This file is on a stack, manipulated by FOPEN /
\G FAPPEND and FCLOSE .
: FAPPEND               ( c-addr u -- )         \ OUTFILE
        fsp 0= #-622 ?error                     \ stack full message
        2dup file-status nip                    \ existing file
        if      fopen exit                      \ no, create it
        then
        -1 +to fsp                              \ push stack
        /filebuffer alloc throw buffersegment ! \ get a segment
        bufferpointer off                       \ new buffer
        r/w open-file throw                     \ open the file
        dup outfile !                           \ save handle
        file-size throw                         \ get the size
        outfile @ reposition-file throw         \ append at the end
        ;

\G Close a file and return to the last one, if any, on the FOPEN
\G or FAPPEND and FCLOSE stack.
: FCLOSE                ( -- )                  \ OUTFILE
        fsp /filestack = #-623 ?error           \ stack empty message
        flushbuffer                             \ write remaining contents
        outfile @ close-file throw              \ close the file
        buffersegment @ dealloc throw           \ release buffer
        1 +to fsp                               \ pop stack
        ;

\G Write a string to a file opened by FOPEN / FAPPEND .
: FTYPE                 ( c-addr u -- )         \ OUTFILE
        bounds                                  \ setup for loop
        ?do     i c@ femit                      \ write each character
        loop
        ;

\G Write CR to a file opened by FOPEN / FAPPEND .
: FCR                   ( -- )                  \ OUTFILE
        ^M femit ^J femit                       \ DOS wants it this way
        ;

\G Write a number of chars to a file opened by FOPEN or FAPPEND .
: FCHARS                ( char u -- )           \ OUTFILE
        0 max 0                                 \ only positive count
        ?do     dup femit
        loop
        drop                                    \ drop character
        ;

deprive
                        \ (* END OF INFORMATION *) /
