\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's
\ DESCRIPTION : An extended ACCEPT
\ CATEGORY    : Standard Programs
\ AUTHOR      : Willem Ouwerkerk, adapted to CHForth by Coos Haak
\ LAST CHANGE : October 28, 1994, Willem Ouwerkerk
\ ----------------------------------------------------------------------

        MARKER -accept

DOC  History mechanism

1.00 Small ACCEPT with one line HISTORY mechanism
1.10 Added multi line circular HISTORY mechanism to it.
1.20 Also added extended cursor control
1.21 Some improvements and cleaning up code ...
1.22 Cursor word left and right added and del. line left of cursor,
     tabulate left & right and erase history buffer.
1.30 Adapted to CHForth by C.H & W.O
1.40 Added INJECT rearranged some words

This history mechanism will not waste any more memory, then needed for
a reasonable amount of space, for remembering typed words for a short
while. Thus making working with tForth a lot smoother.
This is the way it is done:

        ---------------------------------------------------------
Start 00|CNT1|text1|0|CNT2|text2|0|CNT3|text3|0|CNT4|TEXT4|0|0|0|00 End
        ---------------------------------------------------------
                                   ^                         ^
                                READ.ADR                 WRITE.ADR

When buffer is full, or the space at the end is to small, next is
controlled by the routine SWAP.HIST.BUF :

        ---------------------------------------------------------
Start 00|CNT5|longtext5|0|00000000|CNT3|text3|0|CNT4|TEXT4|0|000|00 End
        ---------------------------------------------------------
                          ^        ^
                      WRITE.ADR  READ.ADR

As you see, the damaged text will be erased by the FILL.GAP routine.

ENDDOC

EDITOR ALSO  DEFINITIONS

#200 CONSTANT HIST-LEN                  \ Size of keyboard history buffer.

0 C,
HERE HIST-LEN ALLOT
0 C,
CONSTANT HIST-BUF                       \ Address of keyboard history buffer.

0 VALUE WRITE-ADR
0 VALUE READ-ADR
0 VALUE HIST?
0 VALUE INS
0 VALUE BUFFER-ADR
0 VALUE CNT
0 VALUE CUR
0 VALUE LIM
0 VALUE END?

FORTH DEFINITIONS

0 VALUE ESC?                            \ *WO* 27 Oct
0 VALUE INJECT?

EDITOR DEFINITIONS

: CUR-LEFT      ( -- )      ?AT  SWAP 1- SWAP  AT-XY ;
: CUR-RIGHT     ( -- )      ?AT  SWAP 1+ SWAP  AT-XY ;
: (CUR-LEFT)    ( n -- )    0 ?DO CUR-LEFT LOOP ;
: (CUR-RIGHT)   ( n -- )    0 ?DO CUR-RIGHT LOOP ;
: <BS           ( -- )      ^H EMIT  SPACE  ^H EMIT ;
: BELL          ( -- )      #5 #1760 TONE ;
: CUR-POS       ( -- cadr ) BUFFER-ADR CUR + ;
: LAST-POS      ( -- ladr ) BUFFER-ADR CNT + ;
: DIFF          ( -- n )    CNT CUR - ;
: INLINE?       ( -- flag ) CUR 0> ;
: BLANK?        ( -- flag ) CUR-POS C@ BL = ;
: INCR-CNT      ( n -- )    DUP +TO CNT +TO CUR ;
: TYPE-LINE     ( -- )      CUR-POS DIFF TYPE ;
: RESTORE-CUR   ( -- )      DIFF CHAR+ (CUR-LEFT) ;
: TO-EOL        ( -- )      TYPE-LINE CNT TO CUR ;
: STORE-CHAR    ( ch -- 1 ) CUR-POS C! 1 ;
: ROOM?         ( -- n )    LIM CNT < 0= ;              \ *WO* 27 Oct
: (BS)          ( -- ch )   -1 INCR-CNT <BS ;
: DO-BS'S       ( n -- )    0 ?DO (BS) LOOP ;
: TO-HOME       ( -- )      CUR (CUR-LEFT) CLEAR CUR ;
: ERASE-TO-EOL  ( -- )      DIFF TO-EOL DO-BS'S ;
: DELETE-LINE   ( -- )      TO-HOME ERASE-TO-EOL ;

: FLIP-INSERT   ( -- )
    INS INVERT DUP
    IF      BLOCK-CURSOR
    ELSE    LINE-CURSOR
    THEN
    TO INS TYPE-LINE SPACE RESTORE-CUR
    ;

: WIPE-HISTORY  ( -- )
    HIST-BUF CHAR- HIST-LEN CHAR+ CHAR+ ERASE
    CLEAR READ-ADR  CLEAR WRITE-ADR
    ;

: READ-HIST     ( -- adr )
    HIST-BUF READ-ADR  +
    ;

: WRITE-HIST    ( -- adr )
    HIST-BUF WRITE-ADR +
    ;

: WRAP-HIST-BUF ( -- )                      \ From end to begin buffer !!
    WRITE-ADR CNT + 2 CHARS + HIST-LEN < INVERT
    IF  CLEAR WRITE-ADR
    THEN
    ;

: ADJUST-READ   ( -- )                          \ If count is destroyed !!
    BEGIN   READ-HIST C@  DUP 0=  SWAP BL
            < INVERT OR                         \ From 1 to 31 is valid ??
            READ-ADR HIST-LEN < AND             \ And not end of buffer ?
    WHILE   1 +TO READ-ADR                      \ Next char. !!
    REPEAT
    READ-ADR HIST-LEN < INVERT
    IF      CLEAR READ-ADR
    THEN
    ;

: FILL-GAP      ( -- )                      \ Fill undefined gap with zero's
    WRITE-HIST
    BEGIN   DUP C@
    WHILE   DUP C0! CHAR+
    REPEAT
    DROP
    1 +TO WRITE-ADR ADJUST-READ
    ;

: SCAN-HIST     ( 1|-1 -- )                 \ Scan forw/backw. to a zero
    BEGIN   DUP +TO READ-ADR READ-HIST C@ 0=
    UNTIL
    ABS +TO READ-ADR
    ;

: NEXT-HIST     ( -- )                      \ Next line in HISTORY buffer
    1 SCAN-HIST  -1 +TO READ-ADR
    BEGIN   1 +TO READ-ADR  READ-HIST C@
    UNTIL
    READ-ADR HIST-LEN < INVERT
    IF      CLEAR READ-ADR
    THEN
    ;

: PREV-HIST     ( -- )                  \ Previous line in History buf.
    0
    BEGIN   READ-ADR 0<
            IF      HIST-LEN TO READ-ADR    \ To end of buffer
            THEN
            CHAR+ DUP HIST-LEN =
            -1 +TO READ-ADR
            READ-HIST C@ OR
    UNTIL
    DROP  -1 SCAN-HIST
    ;

: !HISTORY      ( -- )
    CNT DUP>R WRITE-HIST C!                 \ Store count,
    BUFFER-ADR WRITE-HIST CHAR+ R@ CMOVE    \ and text from TIB
    R> CHAR+ +TO WRITE-ADR                  \ then correct write address.
    ;

: PUT-HIST      ( -- )                      \ Put text in HISTORY buffer, but
    CNT 2 >  HIST? 0= AND                   \ only when longer then 2 chars or
    IF                                      \ not unchanged from HISTORY buf.
        WRAP-HIST-BUF                       \ Buffer full ? to start again,
        !HISTORY                            \ put text from TIB in it.
        FILL-GAP                            \ Erase destroyed text, if any
    THEN
    ;

: INJECT        ( $adr -- )                 \ Into TIB, set ?HIST
    DELETE-LINE
    BUFFER-ADR LIM ERASE
    TRUE TO HIST?
    COUNT >R  BUFFER-ADR                    \ *WO* 27 Oct
    R> LIM MIN  >R
    R@ CMOVE  R> TO CNT  CLEAR CUR
    CLEAR INJECT?
    ;

: COPY-HIST     ( -- )                      \ Next line to TIB
    READ-HIST INJECT
    ;

ALSO FORTH DEFINITIONS

: .HISTORY      ( -- )                      \ Print lines in HISTORY buffer
    HIST-BUF ."  contains:" CR
    BEGIN   BEGIN   DUP C@ 0=               \ Scan thru zero's
            WHILE   CHAR+
            REPEAT
            DUP HIST-BUF HIST-LEN + U<      \ Not at end of buffer
    WHILE   CR DUP READ-HIST =              \ Adr equal to read address ?
            IF      ." >> "                 \ Yes, point to line
            ELSE    3 SPACES
            THEN
            COUNT 2DUP TYPE + CHAR+         \ Type line from buffer
    REPEAT
    DROP CR CR ." That's all" CR
    ;

PREVIOUS DEFINITIONS

: INS-CHAR      ( ch1 -- ch2 )
    ROOM?
    IF      CUR-POS DUP CHAR+ DIFF CMOVE>   \ Make room in buffer
            DUP STORE-CHAR INCR-CNT         \ Store character
            SPACE TYPE-LINE RESTORE-CUR     \ Change line on screen
    ELSE    DROP ^G                         \ Buffer full !
    THEN
    ;

: OVERWRITE-CHAR    ( ch1 -- ch2 )
    DIFF                                    \ Within line
    IF      DUP STORE-CHAR +TO CUR          \ Yes, overwrite character
    ELSE    ROOM?                           \ No, still space left
            IF      DUP STORE-CHAR INCR-CNT \ Store character
            ELSE    DROP ^G
            THEN
    THEN
    ;

: CHARACTER     ( ch -- )                   \ Ins. or overwrite new char
    INS
    IF      INS-CHAR
    ELSE    OVERWRITE-CHAR
    THEN
    DUP BL >                                \ Line is only changed with
    IF      CLEAR HIST?                     \ a normal character
    THEN
    DUP 7 =                                 \ A sound needed ?
    IF      DROP BELL EXIT                  \ A quiet short beep
    THEN
    EMIT
    ;

: CURSOR-LEFT   ( -- )
    INLINE?                                 \ Not in left corner ?
    IF      -1 +TO CUR CUR-LEFT             \ Cursor left
    THEN
    ;

: CURSOR-RIGHT  ( -- )
    DIFF                                    \ Within line
    IF      1 +TO CUR CUR-RIGHT EXIT        \ Yes, move cursor
    THEN
    ROOM?
    IF      BL STORE-CHAR INCR-CNT SPACE    \ Add space to end of line
    THEN
    ;

: DELETE        ( -- )
    DIFF
    IF      CUR-POS CHAR+ CUR-POS DIFF CMOVE    \ Delete char in buffer
            -1 +TO CNT  TYPE-LINE SPACE         \ Change screen
            RESTORE-CUR EXIT
    THEN
    BELL
    ;

: BACKSPACE     ( -- )
    INLINE?                                 \ Still on text ?
    IF      CUR-POS DUP CHAR- DIFF CHAR+    \ Delete char. in buffer
            CMOVE (BS) TYPE-LINE SPACE      \ Change screen
            RESTORE-CUR EXIT
    THEN
    BELL
    ;

: WORD-LEFT     ( -- )                      \ Cursor one word left
    INLINE?
    IF      CURSOR-LEFT
    THEN
    BEGIN   BLANK? INLINE? AND
    WHILE   CURSOR-LEFT
    REPEAT
    BEGIN   BLANK? INVERT INLINE? AND
    WHILE   CURSOR-LEFT
    REPEAT
    INLINE?
    IF      CURSOR-RIGHT
    THEN
    ;

: WORD-RIGHT    ( -- )                      \ Cursor one word right
    BEGIN   BLANK? INVERT DIFF 0> AND
    WHILE   CURSOR-RIGHT
    REPEAT
    BEGIN   BLANK? DIFF 0> AND
    WHILE   CURSOR-RIGHT
    REPEAT
    ;

: +TABULATE     ( -- )                      \ To next tab position
    4 CUR 4 MOD - 0
    DO      CURSOR-RIGHT
    LOOP
    ;

: -TABULATE     ( -- )                      \ To previous tab position
    4 CUR 4 MOD - 0
    DO      CURSOR-LEFT
    LOOP
    ;

: ERASE-TO-SOL  ( -- )                      \ Delete line left of cursor
    INLINE?
    IF      CUR-POS BUFFER-ADR DIFF >R DELETE-LINE
            R@ CMOVE R> TO CNT TO-EOL TO-HOME
    THEN
    ;

: RETURN        ( -- cnt )                  \ *WO* 27 Oct
    TYPE-LINE PUT-HIST  SPACE  CNT
    DUP 2 > IF  NEXT-HIST  THEN
    TRUE TO END?
    ;

: ESCAPE        ( -- cnt0 )                 \ *WO* 27 oct
    DELETE-LINE  0
    TRUE TO END?  TRUE TO ESC?
    ;

: RESTORE-LINE  ( -- )                      \ *WO* 28 Oct
    BUFFER-ADR  CNT LIM MIN
    TUCK TYPE  TO CUR  TO-HOME
    ;

\ Move pointer first then copy line, this way looks natural.

: HIST-UP       ( -- )
    PREV-HIST COPY-HIST RESTORE-LINE
    ;

: HIST-DOWN     ( -- )
    NEXT-HIST COPY-HIST RESTORE-LINE
    ;

: HISTORY       ( --  )
    PAGE CR ." History" .HISTORY
    ;

: HELP          ( -- )
    PAGE CR   ." Extra functions of the input line are:"
    CR CR     ."     Home         = Cursor to start of line"
    CR        ."     End          = Cursor to end of line"
    CR        ."     ^Home        = Delete line left of cursor"
    CR        ."     ^End         = Erase text right of cursor"
    CR        ."     Cursor up    = Previous line from history buffer"
    CR        ."     Cursor down  = Next line from history buffer"
    CR        ."     ^Cursor left = Cursor one word left"
    CR        ."     ^Cursor right= Cursor one word right"
    CR        ."     Tab          = Cursor to next TAB position"
    CR        ."     Shift_tab    = Cursor to previous TAB position"
    CR        ."     F1           = Type this help info"
    CR        ."     ALT-H        = Show contents of history buffer"
    CR        ."     Esc          = Escape from input line"
    CR        ."     ^D           = Delete whole line"
    CR        ."     ^E           = Erase whole HISTORY buffer"
    CR        ."     ^R           = Restore text from input buffer"
    CR CR     ." And cursor control, insert mode, delete & backspace !!"
    CR CR  ( 0 )
    ;

\ Coos, om NACCEPT met een externe string op te laten komen moet je
\ het volgende doen:
\
\ CREATE $BUF $20 ALLOT   S" Coos Haak " $BUF PLACE
\
\ TRUE TO INJECT?  $BUF TIB #TIB ACCEPT
\ 
\ Het gevolg is dat de string uit $BUF getypt wordt, met de cursor
\ aan het begin van deze tekst.
\
: NACCEPT   ( c-addr1 c-addr2 u | c-addr u1 -- u2 ) \ *WO* 28 Oct
    TO LIM  TO BUFFER-ADR  
    CLEAR END?  CLEAR ESC?                          \ Added 27 Oct
    0 BUFFER-ADR !                                  \ Added 28 Oct
    CLEAR CNT  CLEAR CUR  INJECT?                   \ Added 27 Oct
    IF      INJECT  RESTORE-LINE
    THEN  
    BEGIN   EKEY EKEY>CHAR
            IF      DUP BL $100 WITHIN
                    IF      CHARACTER
                    ELSE    []KEY @ ?DUP
                            IF      EXECUTE
                            THEN
                            EKEY? IF EKEY DROP THEN \ Added 28 Oct
                    THEN
            ELSE    []KEY @ ?DUP
                    IF      EXECUTE
                    THEN
                    EKEY? IF EKEY DROP THEN         \ Added 28 Oct
            THEN
    END? UNTIL                                      \ Changed 27 Oct
    ;

: _HELP         HELP  RESTORE-LINE ;
: _HISTORY      HISTORY  RESTORE-LINE ;
: _PAGE         PAGE  RESTORE-LINE ;

\ Install all special key strokes

' BACKSPACE     $0008 []KEY !
' DELETE        $5300 []KEY !
' CURSOR-LEFT   $4B00 []KEY !
' CURSOR-RIGHT  $4D00 []KEY !
' WORD-LEFT     $7300 []KEY !
' WORD-RIGHT    $7400 []KEY !
' +TABULATE     $0009 []KEY !
' -TABULATE     $0F00 []KEY !
' TO-EOL        $4F00 []KEY !
' TO-HOME       $4700 []KEY !
' ESCAPE        $0003 []KEY !
' DELETE-LINE   $0004 []KEY !
' WIPE-HISTORY  $0005 []KEY !
' RETURN        $000D []KEY !
' RESTORE-LINE  $0012 []KEY !
' ERASE-TO-SOL  $7700 []KEY !
' ERASE-TO-EOL  $7500 []KEY !
' _HELP         $3B00 []KEY !
' _HISTORY      $2300 []KEY !
' FLIP-INSERT   $5200 []KEY !
' HIST-UP       $4800 []KEY !
' HIST-DOWN     $5000 []KEY !
' ESCAPE        $001B []KEY !
' _PAGE         $2E00 []KEY !

WIPE-HISTORY

' NACCEPT  'ACCEPT !

PREVIOUS FORTH DEFINITIONS

DOC  Test ACCEPT with different buffer sizes

CREATE BUF $40 ALLOT

: A BUF $08 ACCEPT . ;
: B BUF $10 ACCEPT . ;
: C BUF $20 ACCEPT . ;
: D BUF $30 ACCEPT . ;

ENDDOC

                            \ (* End of Source *) /

