\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : Intel 8080 assembler, disassembler and emulator 
\ CATEGORY    : Simulations 
\ AUTHOR      : Coos Haak 
\ LAST CHANGE : May 16, 1994, Coos Haak 
\ ----------------------------------------------------------------------

        NEEDS -assembler

        NEEDS -paragraphs

        NEEDS -stack

        MARKER -8080lib

warning off

hex

1000 segment trgseg

: ts:
        trgseg @ swap ;

: c@-t
        ts: c@x ;

: c!-t
        ts: c!x ;

: @-t
        dup c@-t swap 1+ c@-t join ;

: !-t
        >r split r@ 1+ c!-t r> c!-t ;

: dump-t
        bounds 0 local counter
        ?do     cr i h. space i 10 bounds
                do      i c@-t b. counter 1+ dup to counter 3 and 0=
                        if      space
                        then
                loop
                trgseg @ i 10 stypex stop? ?leave
        10 +loop
        ;

0 value adres

vector .label    ' h. is .label
vector label?    ' 0= is label?

: b>s
        dup $80 and
        if      $FF00 or
        then ;

vocabulary 8080dis

8080dis definitions

0 value opcode

: nextb
        adres c@-t 1 +to adres ;

: nextw
        adres @-t 2 +to adres ;

: longlabel
        adres @-t label?
        if      adres @-t .label
        else    adres @-t h.
        then
        2 +to adres ;

: ..r
        case
                0 of    ." B"   endof
                1 of    ." C"   endof
                2 of    ." D"   endof
                3 of    ." E"   endof
                4 of    ." H"   endof
                5 of    ." L"   endof
                6 of    ." M"   endof
                7 of    ." A"   endof
        endcase ;

: ..r0
        opcode 7 and ..r ;

: ..r1
        opcode 3 rshift 7 and ..r ;

: ..x
        opcode 3 rshift 6 and
        case
        0 of    'B' emit        endof
        2 of    'D' emit        endof
        4 of    'H' emit        endof
        6 of    ." SP"  endof
        endcase ;

: ..p
        opcode 3 rshift 6 and
        case
        0 of    'B' emit        endof
        2 of    'D' emit        endof
        4 of    'H' emit        endof
        6 of    ." PSW" endof
        endcase ;

: self.l
        create
        does>   body> >head (.head) tuck type 8 swap - spaces ;

: .self
        create
        does>   body> >head .head ;

: comma
        ',' emit ;

\ 00

.self nop       self.l lxi      self.l stax     self.l inx
self.l inr      self.l dcr      self.l mvi      .self rcl
self.l dad      self.l ldax     self.l dcx      .self rcr
.self ???

: .lxi
        lxi ..x comma longlabel ;

: .stax
        stax ..x ;

: .inx
        inx ..x ;

: .inr
        inr ..r1 ;

: .dcr
        dcr ..r1 ;

: .mvi
        mvi ..r1 comma nextb b. ;

: .dad
        dad ..x ;

: .ldax
        ldax ..x ;

: .dcx
        dcx ..x ;

\ 10

.self ral       .self rar

self.l shld     .self daa       self.l lhld     .self cma

: .shld
        shld longlabel ;

: .lhld
        lhld longlabel ;

\ 30

self.l sta      .self stc       self.l lda      .self cmc

: .sta
        sta longlabel ;

: .lda
        lda longlabel ;

self.l mov      .self hlt

: .mov
        mov ..r1 comma ..r0 ;

\ 40

self.l add      self.l adc      self.l sub      self.l sbb
self.l ana      self.l xra      self.l ora      self.l cmp

: .add
        add ..r0 ;

: .adc
        adc ..r0 ;

: .sub
        sub ..r0 ;

: .sbb
        sbb ..r0 ;

: .ana
        ana ..r0 ;

: .xra
        xra ..r0 ;

: .ora
        ora ..r0 ;

: .cmp
        cmp ..r0 ;

\ C0

.self rnz       self.l pop      self.l jnz      self.l jmp
self.l cnz      self.l push     self.l adi      self.l rst
self.l rz       .self ret       self.l jz       self.l cz
self.l call     self.l aci

: .pop
        pop ..p ;

: .jnz
        jnz longlabel ;

: .jmp
        jmp longlabel ;

: .cnz
        cnz longlabel ;

: .push
        push ..p ;

: .adi
        adi nextb b. ;

: .rst
        rst opcode 3 rshift 7 and '0' or emit ;

: .jz
        jz longlabel ;

: .cz
        cz longlabel ;

: .call
        call longlabel ;

: .aci
        aci nextb b. ;

\ D0

self.l rnc      self.l jnc      self.l out      self.l cnc
self.l sui      self.l rc       self.l jc       self.l in
self.l cc       self.l sbi

: .jnc
        jnc longlabel ;

: .out
        out nextb b. ;

: .cnc
        cnc longlabel ;

: .sui
        sui nextb b. ;

: .jc
        jc longlabel ;

: .in
        in nextb b. ;

: .cc
        cc longlabel ;

: .sbi
        sbi nextb b. ;

\ E0

self.l rpo      self.l jpo      .self xthl      self.l cpo
self.l ani      self.l rpe      .self pchl      self.l jpe
.self xchg      self.l cpe      self.l xri

: .jpo
        jpo longlabel ;

: .cpo
        cpo longlabel ;

: .ani
        ani nextb b. ;

: .jpe
        jpe longlabel ;

: .cpe
        cpe longlabel ;

: .xri
        xri nextb b. ;

\ F0

self.l rp       self.l jp       .self di        self.l cp
self.l ori      self.l rm       .self sphl      self.l jm
.self ei        self.l cm       self.l cpi

: .jp
        jp longlabel ;

: .cp
        cp longlabel ;

: .ori
        ori nextb b. ;

: .jm
        jm longlabel ;

: .cm
        cm longlabel ;

: .cpi
        cpi nextb b. ;

create opcode-table
( 00 )  ' nop ,         ' .lxi ,        ' .stax ,       ' .inx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' rcl ,
        ' ??? ,         ' .dad ,        ' .ldax ,       ' .dcx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' rcr ,
( 10 )  ' ??? ,         ' .lxi ,        ' .stax ,       ' .inx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' ral ,
        ' ??? ,         ' .dad ,        ' .ldax ,       ' .dcx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' rar ,
( 20 )  ' ??? ,         ' .lxi ,        ' .shld ,       ' .inx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' daa ,
        ' ??? ,         ' .dad ,        ' .lhld ,       ' .dcx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' cma ,
( 30 )  ' ??? ,         ' .lxi ,        ' .sta ,        ' .inx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' stc ,
        ' ??? ,         ' .dad ,        ' .lda ,        ' .dcx ,
        ' .inr ,        ' .dcr ,        ' .mvi ,        ' cmc ,
( 40 )  ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
( 50 )  ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
( 60 )  ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
( 70 )  ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' hlt ,         ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
        ' .mov ,        ' .mov ,        ' .mov ,        ' .mov ,
( 80 )  ' .add ,        ' .add ,        ' .add ,        ' .add ,
        ' .add ,        ' .add ,        ' .add ,        ' .add ,
        ' .adc ,        ' .adc ,        ' .adc ,        ' .adc ,
        ' .adc ,        ' .adc ,        ' .adc ,        ' .adc ,
( 90 )  ' .sub ,        ' .sub ,        ' .sub ,        ' .sub ,
        ' .sub ,        ' .sub ,        ' .sub ,        ' .sub ,
        ' .sbb ,        ' .sbb ,        ' .sbb ,        ' .sbb ,
        ' .sbb ,        ' .sbb ,        ' .sbb ,        ' .sbb ,
( A0 )  ' .ana ,        ' .ana ,        ' .ana ,        ' .ana ,
        ' .ana ,        ' .ana ,        ' .ana ,        ' .ana ,
        ' .xra ,        ' .xra ,        ' .xra ,        ' .xra ,
        ' .xra ,        ' .xra ,        ' .xra ,        ' .xra ,
( B0 )  ' .ora ,        ' .ora ,        ' .ora ,        ' .ora ,
        ' .ora ,        ' .ora ,        ' .ora ,        ' .ora ,
        ' .cmp ,        ' .cmp ,        ' .cmp ,        ' .cmp ,
        ' .cmp ,        ' .cmp ,        ' .cmp ,        ' .cmp ,
( C0 )  ' rnz ,         ' .pop ,        ' .jnz ,        ' .jmp ,
        ' .cnz ,        ' .push ,       ' .adi ,        ' .rst ,
        ' rz ,          ' ret ,         ' .jz ,         ' ??? ,
        ' .cz ,         ' .call ,       ' .aci ,        ' .rst ,
( D0 )  ' rnc ,         ' .pop ,        ' .jnc ,        ' .out ,
        ' .cnc ,        ' .push ,       ' .sui ,        ' .rst ,
        ' rc ,          ' ??? ,         ' .jc ,         ' .in ,
        ' .cc ,         ' ??? ,         ' .sbi ,        ' .rst ,
( E0 )  ' rpo ,         ' .pop ,        ' .jpo ,        ' xthl ,
        ' .cpo ,        ' .push ,       ' .ani ,        ' .rst ,
        ' rpe ,         ' pchl ,        ' .jpe ,        ' xchg ,
        ' .cpe ,        ' ??? ,         ' .xri ,        ' .rst ,
( F0 )  ' rp ,          ' .pop ,        ' .jp ,         ' di ,
        ' .cp ,         ' .push ,       ' .ori ,        ' .rst ,
        ' rm ,          ' sphl ,        ' .jm ,         ' ei ,
        ' .cm ,         ' ??? ,         ' .cpi ,        ' .rst ,

: .all-opcodes
        nextb dup to opcode opcode-table []cell @ execute ;

forth definitions

: .inst
        [ 8080dis ]
        cr adres h. adres label?
        if      adres .label
        then
        #16 htab adres >r .all-opcodes
        #40 htab r> adres over - 2dup bounds
        do      i c@-t b.
        loop
        space trgseg @ -rot #64 htab stypex space
        eol ;   forth

: disasm
        to adres
        begin   .inst key ^[ =
        until ;

vocabulary 8080sim

8080sim definitions

: byte
        create  random c,
        does>   c@ ;

methods byte

: to
        postpone literal postpone c!
        ;

: +to
        postpone literal postpone c+!
        ;

: clear
        postpone literal postpone c0!
        ;

end-methods

create flags    0 c,

0 value intflag

: carry@
        1 flags c@ and ;

: carry!
        0<> 1 and flags c@ %11111110 and or flags c! ;

code (flags)
                add     bl, # 0
                pushf
                pop     bx
                and     bx, # %11111110
                next    end-code

: flags!
        (flags) flags c@ 1 and or flags c! ;

: zero@
        %01000000 flags c@ and ;

: half@
        %00010000 flags c@ and ;

: sign@
        %10000000 flags c@ and ;

: parity@
        %00000100 flags c@ and ;

0 value pc
0 value sp
byte opcode
byte _b
byte _c
byte _d
byte _e
byte _h
byte _l
byte _a

: _bc
        _c _b join ;

: _bc!
        split to _b to _c ;

: _de
        _e _d join ;

: _de!
        split to _d to _e ;

: _hl
        _l _h join ;

: _hl!
        split to _h to _l ;

: @inlw
        pc @-t 2 +to pc ;

: @inlb
        pc c@-t 1 +to pc ;

: get-arg
        7 and
        case
                0 of    _b      endof
                1 of    _c      endof
                2 of    _d      endof
                3 of    _e      endof
                4 of    _h      endof
                5 of    _l      endof
                6 of    _hl c@-t        endof
                7 of    _a      endof
        endcase ;

: ???
        click ;

: nop ;

: lxi
        @inlw swap 3 and
        case
                0 of    _bc!    endof
                1 of    _de!    endof
                2 of    _hl!    endof
                3 of    to sp   endof
        endcase ;

: stax
        _a swap 1 and
        case
                0 of    _bc     endof
                1 of    _de     endof
        endcase
        c!-t ;

: inx
        3 and
        case
                0 of    _bc 1+ _bc!     endof
                1 of    _de 1+ _de!     endof
                2 of    _hl 1+ _hl!     endof
                3 of    1 +to sp endof
        endcase ;

: inr
        7 and
        case
                0 of    _b 1+ dup to _b endof
                1 of    _c 1+ dup to _c endof
                2 of    _d 1+ dup to _d endof
                3 of    _e 1+ dup to _e endof
                4 of    _h 1+ dup to _h endof
                5 of    _l 1+ dup to _l endof
                6 of    _hl c@-t 1+ dup _hl c!-t        endof
                7 of    _a 1+ dup to _a endof
        endcase
        split carry! flags! ;

: dcr
        7 and
        case
                0 of    _b 1- dup to _b endof
                1 of    _c 1- dup to _c endof
                2 of    _d 1- dup to _d endof
                3 of    _e 1- dup to _e endof
                4 of    _h 1- dup to _h endof
                5 of    _l 1- dup to _l endof
                6 of    _hl c@-t 1- dup _hl c!-t        endof
                7 of    _a 1- dup to _a endof
        endcase
        split carry! flags! ;

: mvi
        @inlb swap 7 and
        case
                0 of    to _b   endof
                1 of    to _c   endof
                2 of    to _d   endof
                3 of    to _e   endof
                4 of    to _h   endof
                5 of    to _l   endof
                6 of    _hl c!-t        endof
                7 of    to _a   endof
        endcase ;

: rcl
        carry@ _a 2* or split carry! dup to _a flags! ;

: dad
        3 and
        case
                0 of    _bc     endof
                1 of    _de     endof
                2 of    _hl     endof
                3 of    sp      endof
        endcase
        _hl + _hl! ;

: ldax
        1 and
        case
                0 of    _bc     endof
                1 of    _de     endof
        endcase
        c@-t to _a ;

: dcx
        3 and
        case
                0 of    _bc 1- _bc!     endof
                1 of    _de 1- _de!     endof
                2 of    _hl 1- _hl!     endof
                3 of    -1 +to sp endof
        endcase ;

: rcr
        _a carry@ join 7 lshift split dup to _a flags! 0<> carry! ;

: 00s
        $F and
        case
                0 of    nop     endof
                1 of    0 lxi   endof
                2 of    0 stax  endof
                3 of    0 inx   endof
                4 of    0 inr   endof
                5 of    0 dcr   endof
                6 of    0 mvi   endof
                7 of    rcl     endof
                8 of    ???     endof
                9 of    0 dad   endof
                $A of   0 ldax  endof
                $B of   0 dcx   endof
                $C of   1 inr   endof
                $D of   1 dcr   endof
                $E of   1 mvi   endof
                $F of   rcr     endof
        endcase ;

: ral
        _a dup join 7 rshift split carry! dup to _a flags! ;

: rar
        _a dup join 7 lshift split dup to _a flags! 0<> carry! ;

: 10s
        $F and
        case
                0 of    ???     endof
                1 of    1 lxi   endof
                2 of    1 stax  endof
                3 of    1 inx   endof
                4 of    2 inr   endof
                5 of    2 dcr   endof
                6 of    2 mvi   endof
                7 of    ral     endof
                8 of    ???     endof
                9 of    1 dad   endof
                $A of   1 ldax  endof
                $B of   1 dcx   endof
                $C of   3 inr   endof
                $D of   3 dcr   endof
                $E of   3 mvi   endof
                $F of   rar     endof
        endcase ;

: shld
        _hl @inlw !-t ;

: daa
        click ;

: lhld
        @inlw @-t _hl! ;

: cma
        _a invert dup to _a flags! 0 carry! ;

: 20s
        $F and
        case
                0 of    ???     endof
                1 of    2 lxi   endof
                2 of    shld    endof
                3 of    2 inx   endof
                4 of    4 inr   endof
                5 of    4 dcr   endof
                6 of    4 mvi   endof
                7 of    daa     endof
                8 of    ???     endof
                9 of    2 dad   endof
                $A of   lhld    endof
                $B of   2 dcx   endof
                $C of   5 inr   endof
                $D of   5 dcr   endof
                $E of   5 mvi   endof
                $F of   cma     endof
        endcase ;

: sta
        _a @inlw c!-t ;

: stc
        1 carry! ;

: lda
        @inlw c@-t to _a ;

: cmc
        carry@ invert carry! ;

: 30s
        $F and
        case
                0 of    ???     endof
                1 of    3 lxi   endof
                2 of    sta     endof
                3 of    3 inx   endof
                4 of    6 inr   endof
                5 of    6 dcr   endof
                6 of    6 mvi   endof
                7 of    stc     endof
                8 of    ???     endof
                9 of    3 dad   endof
                $A of   lda     endof
                $B of   3 dcx   endof
                $C of   7 inr   endof
                $D of   7 dcr   endof
                $E of   7 mvi   endof
                $F of   cmc     endof
        endcase ;

: 00s-30s
        dup 4 rshift 3 &exec: 00s 10s 20s 30s ;

: mov
        dup 76 =
        if      drop click exit
        then
        8 /mod 7 and swap get-arg swap
        case
                0 of    to _b   endof
                1 of    to _c   endof
                2 of    to _d   endof
                3 of    to _e   endof
                4 of    to _h   endof
                5 of    to _l   endof
                6 of    _hl c!-t        endof
                7 of    to _a   endof
        endcase ;

: add
        get-arg _a + split carry! dup to _a flags! ;

: adc
        get-arg _a carry@ + + split carry! dup to _a flags! ;

: sub
        get-arg _a swap - split carry! dup to _a flags! ;

: sbb
        get-arg _a swap - carry@ - split carry! dup to _a flags! ;

: ana
        get-arg _a and dup to _a flags! ;

: xra
        get-arg _a xor dup to _a flags! ;

: ora
        get-arg _a or dup to _a flags! ;

: cmp
        get-arg _a swap - split carry! flags! ;

: 80s
        dup 3 rshift 7 &exec: add adc sub sbb ana xra ora cmp ;

: ret
        sp @-t to pc 2 +to sp ;

: rnz
        zero@ 0=
        if      ret
        then ;

: pop
        sp @-t 2 +to sp swap 3 and
        case
                0 of    _bc!    endof
                1 of    _de!    endof
                2 of    _hl!    endof
                3 of    split to _a flags c!    endof
        endcase ;

: jmp
        @inlw to pc ;

: jnz
        zero@ 0=
        if      jmp exit
        then
        2 +to pc ;

: call
        @inlw -2 +to sp pc sp !-t to pc ;

: cnz
        zero@ 0=
        if      call exit
        then
        2 +to pc ;

: push
        3 and
        case
                0 of    _bc     endof
                1 of    _de     endof
                2 of    _hl     endof
                3 of    flags c@ _a join        endof
        endcase
        -2 +to sp sp !-t ;

: adi
        _a @inlb + split carry! dup to _a flags! ;

: rst
        -2 +to sp pc sp !-t %00111000 and to pc ;

: rz
        zero@
        if      ret
        then ;

: jz
        zero@
        if      jmp exit
        then
        2 +to pc ;

: cz
        zero@
        if      call exit
        then
        2 +to pc ;

: aci
        _a @inlb + carry@ + split carry! dup to _a flags! ;

: c0s
        $F and
        case
                0 of    rnz     endof
                1 of    0 pop   endof
                2 of    jnz     endof
                3 of    jmp     endof
                4 of    cnz     endof
                5 of    0 push  endof
                6 of    adi     endof
                7 of    opcode rst      endof
                8 of    rz      endof
                9 of    ret     endof
                $A of   jz      endof
                $B of   ???     endof
                $C of   cz      endof
                $D of   call    endof
                $E of   aci     endof
                $F of   opcode rst      endof
        endcase ;

: rnc
        carry@ 0=
        if      ret
        then ;

: jnc
        carry@ 0=
        if      jmp exit
        then
        2 +to pc ;

: out
        1 +to pc _a b. ;

: cnc
        carry@ 0=
        if      call exit
        then
        2 +to pc ;

: sui
        _a @inlb - split carry! dup to _a flags! ;

: rc
        carry@
        if      ret
        then ;

: jc
        carry@
        if      jmp exit
        then
        2 +to pc ;

: in
        1 +to pc key to _a ;

: cc
        carry@
        if      call exit
        then
        2 +to pc ;

: sbi
        _a @inlb - carry@ - split carry! dup to _a flags! ;

: d0s
        $F and
        case
                0 of    rnc     endof
                1 of    1 pop   endof
                2 of    jnc     endof
                3 of    out     endof
                4 of    cnc     endof
                5 of    1 push  endof
                6 of    sui     endof
                7 of    opcode rst      endof
                8 of    rc      endof
                9 of    ???     endof
                $A of   jc      endof
                $B of   in      endof
                $C of   cc      endof
                $D of   ???     endof
                $E of   sbi     endof
                $F of   opcode rst      endof
        endcase ;

: rpo
        parity@ 0=
        if      ret
        then ;

: jpo
        parity@ 0=
        if      jmp exit
        then
        2 +to pc ;

: xthl
        sp @-t _hl sp !-t _hl! ;

: cpo
        parity@ 0=
        if      call exit
        then
        2 +to pc ;

: ani
        _a @inlb and dup to _a flags! 0 carry! ;

: rpe
        parity@
        if      ret
        then ;

: pchl
        _hl to pc ;

: jpe
        parity@
        if      jmp exit
        then
        2 +to pc ;

: xchg
        _de _hl _de! _hl! ;

: cpe
        parity@
        if      call exit
        then
        2 +to pc ;

: xri
        _a @inlb xor dup to _a flags! 0 carry! ;

: e0s
        $F and
        case
                0 of    rpo     endof
                1 of    2 pop   endof
                2 of    jpo     endof
                3 of    xthl    endof
                4 of    cpo     endof
                5 of    2 push  endof
                6 of    ani     endof
                7 of    opcode rst      endof
                8 of    rpe     endof
                9 of    pchl    endof
                $A of   jpe     endof
                $B of   xchg    endof
                $C of   cpe     endof
                $D of   ???     endof
                $E of   xri     endof
                $F of   opcode rst      endof
        endcase ;

: rp
        sign@ 0=
        if      ret
        then ;

: jp
        sign@ 0=
        if      jmp exit
        then
        2 +to pc ;

: di
        clear intflag ;

: cp
        sign@ 0=
        if      call exit
        then
        2 +to pc ;

: ori
        _a @inlb or dup to _a flags! 0 carry! ;

: rm
        sign@
        if      ret
        then ;

: sphl
        _hl to sp ;

: jm
        sign@
        if      jmp exit
        then
        2 +to pc ;

: ei
        1 to intflag ;

: cm
        sign@
        if      call exit
        then
        2 +to pc ;

: cpi
        _a @inlb - split carry! flags! ;

: f0s
        $F and
        case
                0 of    rp      endof
                1 of    3 pop   endof
                2 of    jp      endof
                3 of    di      endof
                4 of    cp      endof
                5 of    3 push  endof
                6 of    ori     endof
                7 of    opcode rst      endof
                8 of    rm      endof
                9 of    sphl    endof
                $A of   jm      endof
                $B of   ei      endof
                $C of   cm      endof
                $D of   ???     endof
                $E of   cpi     endof
                $F of   opcode rst      endof
        endcase ;

: fetch
        pc c@-t to opcode 1 +to pc ;

: c0s-f0s
        dup 4 rshift 3 &exec: c0s d0s e0s f0s ;

forth definitions

: reset
        [ 8080sim ] clear pc clear intflag ;

: interrupt
        [ 8080sim ] intflag
        if      clear intflag rst
        else    drop
        then ;

: =row
        dup h. space
        dup 10 bounds
        do      i c@-t b.
        loop
        space 10 bounds
        do      i c@-t semit
        loop ;

: .regs
        [ 8080sim ]
        cr ." BC " _bc =row
        cr ." DE " _de =row
        cr ." HL " _hl =row
        cr ." A  " _a b.
        cr ." SP " sp =row
        cr ." PC " pc =row
\       SZ-A-P-C
        cr ." Flags: " flags c@ $80 and if ." MI " else ." PL "
        then
        flags c@ $40 and if ." ZR " else ." NZ "
        then
        flags c@ 4 and if ." PE " else ." PO "
        then
        flags c@ 1 and if ." CY " else ." NC "
        then
        intflag if ." IE " else ." ID "
        then ;
        
: exec
        [ 8080sim ]
        opcode dup $C0 and
        case
                0 of    00s-30s endof
                $40 of  mov     endof
                $80 of  80s     endof
                $C0 of  c0s-f0s endof
        endcase
        .regs ;

: step
        [ 8080sim ] hide-cursor home fetch exec
        pc to adres #10 0
        do      .inst
        loop
        show-cursor ;

: steps
        0
        ?do     step
        loop ;

: trace
        begin   step key >upc dup 'I' =
                if      opcode $FB <> intflag and
                        if      $38 rst
                        then
                then
                ^[ =
        until ;

vocabulary 8080asm

8080asm definitions

create aprior   4 cells allot
' drop aprior !
' drop aprior cell+ cell+ !

: a;!
        aprior cell+ cell+ 2! ;

: clr-a;
        0 ['] drop a;! ;

: a;
        aprior 2@ execute aprior cell+ cell+ 2@ aprior 2! clr-a; ;

0 value here-t

: c,-t
        here-t c!-t 1 +to here-t ;

: ,-t
        here-t !-t 2 +to here-t ;

0 constant b    1 constant c    2 constant d    3 constant e
4 constant h    5 constant l    6 constant m    7 constant a
6 constant psw  6 constant sp

: (m1)
        c@ c,-t ;

: m1
        create  c,
        does>   ['] (m1) a;! a; ;

00 m1 nop       07 m1 rlc       0F m1 rrc       17 m1 ral       1F m1 rar
27 m1 daa       2F m1 cma       37 m1 stc       3F m1 cmc
76 m1 hlt
E3 m1 xthl      E9 m1 pchl      EB m1 xchg
F3 m1 di        F9 m1 sphl      FB m1 ei

: (m2)
        c@ + c,-t ;

: m2
        create  c,
        does>   ['] (m2) a;! a; ;

80 m2 add       88 m2 adc       90 m2 sub       98 m2 sbb
A0 m2 ana       A8 m2 xra       B0 m2 ora       B8 m2 cmp

: (m3)
        c@ swap 3 lshift + c,-t ;

: m3
        create  c,
        does>   ['] (m3) a;! a; ;

02 m3 stax      03 m3 inx       04 m3 inr       05 m3 dcr
09 m3 dad       0A m3 ldax      0B m3 dcx
C1 m3 pop       C5 m3 push      C7 m3 rst

: (m4)
        c@ c,-t c,-t ;

: m4
        create  c,
        does>   ['] (m4) a;! a; ;

C6 m4 adi       CE m4 aci       D3 m4 out       D6 m4 sui       0DB m4 in
DE m4 sbi       E6 m4 ani       EE m4 xri       F6 m4 ori       FE m4 cpi

: (m5)
        c@ c,-t ,-t ;

: m5
        create  c,
        does>   ['] (m5) a;! a; ;

22 m5 shld      2A m5 lhld      32 m5 sta       3A m5 lda
C3 m5 jmp       C9 m1 ret       0CD m5 call

: (db)
        drop c,-t ;

: db
        0 ['] (db) a;! a; ;

: (dw)
        drop ,-t ;

: dw
        0 ['] (dw) a;! a; ;

: (ds)
        drop 0
        ?do     count c,-t
        loop    drop ;

: ds
        0 ['] (ds) a;! a; ;

C2 constant 0=  D2 constant CS  E2 constant PE  F2 constant 0<
C3 constant never

: not
        8 + ;

: (mov)
        drop swap 3 lshift 40 + + c,-t ;

: mov
        0 ['] (mov) a;! a; ;

: (mvi)
        drop swap 3 lshift 6 + c,-t c,-t ;

: mvi
        0 ['] (mvi) a;! a; ;

: (lxi)
        drop swap 3 lshift 1+ c,-t ,-t ;

: lxi
        0 ['] (lxi) a;! a; ;

2 cells constant /label
#100 constant #labels

create labels   #labels /label * allot
0 value lastlabel

: []label
        labels swap /label * + ;

:noname
        local x false lastlabel 0
        ?do     i []label @ x =
                if      invert leave
                then
        loop
        ;  is label?

:noname
        lastlabel 0
        ?do     dup i []label @ =
                if      i []label cell+ @ .head leave
                then
        loop
        drop
        ;  is .label

: >mark
        here-t >s 0 ,-t ;

: >resolve
        here-t s> !-t ;

: <mark
        here-t >s ;

: <resolve
        s> ,-t ;

: cs-swap
        s> s> swap >s >s ;

: if
        >r a; r> c,-t >mark ;

: then
        a; >resolve ;

: ahead
        never if ;

: else
        ahead cs-swap
        then ;

: begin
        a; <mark ;

: until
        >r a; r> c,-t <resolve ;

: again
        never until ;

: while
        if cs-swap ;

: repeat
        again
        then ;

forth definitions

: clearlabels
        [ 8080asm ] clr-a; labels #labels /label * erase clear lastlabel ;

: asm:
        [ 8080asm ] 8080asm !csp clr-a; ;

: asm;
        [ 8080asm ] a; forth ?csp ;

: label
        create  [ 8080asm ] a; here-t lastlabel []label ! here body> >head
                lastlabel []label cell+ ! 1 +to lastlabel here-t , asm:
        does>   @ ;

: org
        [ 8080asm ] >r a; r> to here-t asm: ;

: ==
        [ 8080asm ] org label ;

trgseg 2@ swap 0 fillp

forth

warning on
                            \ (* End of Source *) /
