\name{NHANES}
\alias{NHANES}
\docType{data}
\title{NHANES Data : National Health and Nutrition Examination Survey}
\usage{data(NHANES)}
\description{
  This is a somewhat large interesting dataset, a data frame of 15
  variables (columns) on 9575 persons (rows).
}
\format{
  This data frame contains the following columns:
  \describe{
    \item{Cancer.Incidence}{binary factor with levels \code{No} and \code{Yes}.}
    \item{Cancer.Death}{binary factor with levels \code{No} and \code{Yes}.}
    \item{Age}{numeric vector giving age of the person in years.}
    \item{Smoke}{a factor with levels \code{Current}, \code{Past},
      \code{Nonsmoker}, and \code{Unknown}.}
    \item{Ed}{numeric vector of \eqn{\{0,1\}} codes giving the education level.}
    \item{Race}{numeric vector of \eqn{\{0,1\}} codes giving the
      person's race.%% FIXME : 0 = ?  1 = ?
    }
    \item{Weight}{numeric vector giving the weight in kilograms}
    \item{BMI}{numeric vector giving Body Mass Index, i.e.,
      \code{Weight/Height^2} where Height is in meters, and missings
      (61\% !) are coded as \code{0} originally.}%% rather FIXME?
    \item{Diet.Iron}{numeric giving Dietary iron.}
    \item{Albumin}{numeric giving albumin level in g/l.}
    \item{Serum.Iron}{numeric giving Serum iron in \eqn{\mu}{u}g/l.}
    \item{TIBC}{numeric giving Total Iron Binding Capacity in \eqn{\mu}{u}g/l.}
    \item{Transferin}{numeric giving Transferin Saturation which is just
      \code{100*serum.iron/TIBC}.}
    \item{Hemoglobin}{numeric giving Hemoglobin level.}
    \item{Sex}{a factor with levels \code{F} (female) and \code{M} (male).}
  }
}
% \details{
% }
\source{
 unknown%____ FIXME ____
}
\references{
% ~~ possibly secondary sources and usages ~~
}
\examples{
data(NHANES)
summary(NHANES)
## Missing Data overview :
nNA <- sapply(NHANES, function(x)sum(is.na(x)))
cbind(nNA[nNA > 0])
# Which are just these 6 :
\dontrun{
Diet.Iron         141
Albumin           252
Serum.Iron       1008
TIBC              853
Transferin       1019
Hemoglobin        759
}%dont
}
\keyword{datasets}

\eof
\name{ColorRamps}
\alias{ColorRamps}
\alias{LinGray}
\alias{BTC}
\alias{BTY}
\alias{LinOCS}
\alias{heat.ob}
\alias{magent}
\alias{plinrain}

\title{Color ramps on perceptually linear scales}
\description{Functions for returning colors on perceptually linear
  scales, where steps correspond to 'just detectable differences')
}
\usage{
LinGray (n, beg=1, end=92)
BTC     (n, beg=1, end=256)
LinOCS  (n, beg=1, end=256)
heat.ob (n, beg=1, end=256)
magent  (n, beg=1, end=256)
plinrain(n, beg=1, end=256)
}
\arguments{
  \item{n}{number of colors to return from the ramp}
  \item{beg}{begining of ramp, integer from 1-255}
  \item{end}{end of ramp, integer from 1-255}
}
\value{
  returns an array of colors
}
\details{
  Several precalulated color ramps, that are on a perceptually linear
  color scale.  A perceptually linear color scale is a scale where each
  jump corresponds to a ``just detectable difference'' in color and the
  scale is percieved as linear by the human eye (emprically determined).

  When using the ramps, if \code{beg} is less than \code{end} the ramp
  will be reversed.
}
\references{
  Haim Levkowitz (1997)
  \emph{Color Theory and Modeling for Computer Graphics,
    Visualization, and Multimedia Applications}.
  Kluwer Academic Publishers, Boston/London/Dordrecht.
  \url{http://www.cs.uml.edu/~haim/ColorCenter/}
}
\seealso{
  \code{\link{rainbow}}, \code{\link{terrain.colors}}, \code{\link{rgb}},
  \code{\link{hsv}}
}
\examples{
h <- hexbin(rnorm(10000),rnorm(10000))
plot.hexbin(h, colramp=BTY)
#looks better if you shave the tails
plot.hexbin(h,colramp=function(n){LinOCS(n,beg=15,end=225)})
}
\author{Nicholas Lewin-Koh}
\keyword{color}

\eof
\name{erode.hexbin}
\alias{erode.hexbin}
\title{Erosion of a Hexagon Count Image}

\description{
  This erosion algorithm removes counts from hexagon cells at a rate
  proportional to the cells' exposed surface area.  When a cell becomes
  empty,  algorithm removes the emptied cell and notes the removal
  order.  Cell removal increases the exposure of any neighboring cells.
  The last cell removed is a type of bivariate median.
}

\usage{
erode.hexbin(bin, cdfcut = 0.5)
}

\arguments{
  \item{bin}{an object of type hexbin.}
  \item{cdfcut}{number in (0,1) indicating the confidence level for the
    limits.}
}

\value{
  A bin object with high count cells and a component called erode that
  gives the erosion order.
}

\details{
  The algorithm extracts  high count cells with containing a given
  fraction (cdfcut) of the total counts.  The algorithm extracts all
  cells if cdfcut=0.  The algorithm performs gray-level erosion on the
  extracted cells.  Each erosion cycle removes counts from cells.  The
  counts removed for each cell are a multiple of the cell's exposed-face
  count.  The algorithm choses the multiple so at least one cell will be
  empty or have a count deficit on each erosion cycle.  The erode vector
  contain an erosion number for each cell.   The value of  erode is

  6*erosion\_cycle\_ at\_ cell\_ removal - cell\_deficit\_at\_removal

  Cells with low values are eroded first.   The cell with the highest
  erosion number is a candidate bivariate median.  A few ties in erode
  are common.
}

\seealso{
  \code{\link{hexbin}}, \code{\link{smooth.hexbin}},
  \code{\link{hcell2xy}}, \code{\link{hcell}}, \code{\link{plot.hexbin}},
  \code{\link{hboxplot}}, \code{\link{hdiffplot}},
  \code{\link{hmatplot}}, \code{\link{hexagons}}, \code{\link{hex.legend}}
}

\examples{

set.seed(153)
%% I add the following in to make it to work
x <- rnorm(10000)
y <- rnorm(10000)
bin <- hexbin(x,y)
%% End. JZ

## bivariate boxplot
smbin  <- smooth.hexbin(bin)
erodebin <- erode.hexbin(smbin,cdfcut=.5)
hboxplot(erodebin)

%% MM: This looks wrong -- both the graphic and the logic in "par" here :
# show erosion order
oldpar <- par(no.readonly=TRUE)
screenpar <- plot.hexbin(bin,style= "lat", minarea=1, maxarea=1,
                         legend=FALSE, border=gray(.7))
par(screenpar)      # reset to the plot on the screen

hexagons(erodebin,style= "lat", minarea=1, maxarea=1,pen="green")
xy <- hcell2xy(erodebin)
text(xy, lab = as.character(erodebin$erode),col="white",cex=.65)
par(oldpar)
}
\keyword{hplot}

\eof
\name{hboxplot}
\alias{hboxplot}
\title{2-D generalization of Boxplot}

\description{
  Let bin1 be an hexbin object with an erode component.  This routine
  plots the high counts cells selected by the erode function.  By
  default the high counts cells contain 50 percent of the counts so
  analagous to the interquartile range.  The function distinguishes the
  last cells eroded using color.  These cells correspond to one
  definition of the bivariate median.
}

\usage{
hboxplot(bin1, xbnds = bin1$xbnds, ybnds = bin1$ybnds,
         density = c(-1, -1), border = c(FALSE, FALSE),
         pen = c(2, 3), unzoom = 1.04, reshape = FALSE,
         xlab = "", ylab = "", \dots)
}

\arguments{
  \item{bin1}{an object of type hexbin}
  \item{xbnds}{global x-axis plotting limits for multiple plots.}
  \item{ybnds}{global y-axis plotting limits for multiple plots. }
  \item{density}{fill arguments to polygon()}
  \item{border}{border arguments to polygon()}
  \item{pen}{pen numbers for polygon()}
  \item{unzoom}{plot limit expansion factor when xbnds is missing}
  \item{reshape}{logical value to reshape the plot although xbnds and
    ybnds are present.}
  \item{xlab, ylab}{x- and y- axis labels passed to \code{\link{plot}()}.}
  \item{\dots}{additional arguments passed to \code{\link{plot}()}.}
}

\value{
 If (missing(xbnds) $|$ reshape)screen.par else 'done'
}

\references{ see in \code{\link{hexagons}}.}

\details{
  The \code{density}, \code{border}, and \code{pen} arguments correspond
  to the \code{\link{polygon}} function calls for plotting two types of
  cells.
  The cell types, pen numbers and suggested colors are\cr
  \tabular{lll}{
    TYPE				   \tab PEN \tab COLOR \cr
    cells of bin1                           \tab 2  \tab light gray \cr
    last eroded cells of bin1 (median cells)\tab 1  \tab black \cr
  }

  The erode components of the hexbin objects must be present for the
  medians cells to plot.

  When \code{xbnds} is missing or \code{reshape} is true, the plot
  changes graphics parameters and resets them.   When \code{xbnds} is
  missing the function also zooms in based on the available data to
  provide increased resolution.

  The zoom used the hexagon cell centers.  The unzoom argument backs off
  a bit so the whole hexagon will fit in the plot.

  \code{Hboxplot()} is used as a stand alone function, for producing separate
  legends for \code{\link{hmatplot}()} and for panels in
  \code{\link{hmatplot}()}.
}

\seealso{
  \code{\link{hexbin}}, \code{\link{smooth.hexbin}}, \code{\link{erode.hexbin}},
  \code{\link{hcell2xy}},  \code{\link{hcell}}
  \code{\link{plot.hexbin}}, \code{\link{hmatplot}},
  \code{\link{hexagons}}, \code{\link{hex.legend}}
}

\examples{
#  boxplot of smoothed counts
x <- rnorm(10000)
y <- rnorm(10000)

bin <- hexbin(x,y)
erodebin <- erode.hexbin(smooth.hexbin(bin))

hboxplot(erodebin)
}
\keyword{hplot}











\eof
\name{hcell}
\alias{hcell}
\title{compute hexagon cell id's from x and y map projection coordinates}

\description{
  Creates an hexbin list containing hexagon cell id's corresponding to
  (x,y) pairs.  Used for partitioning data into hexagon regions and
  computation of statistics for each region.  Provides symbol congestion
  control in maps.
}

\usage{
hcell(x, y, xbins, shape, xbnds, ybnds, verbose = getOption("verbose"))
}

\arguments{
  \item{x}{a vector of numbers.  The first coordinate of bivariate
    data for cell id computation.}
  \item{y}{a vector of numbers.  The second coordinate of bivariate
    data for cell id computation. }
  \item{xbins}{number of  hexagon cells partitioning the range of xbnds.}
  \item{xbnds}{ horizontal limits of the region in x units.}
  \item{ybnds}{ vertical limits of the region in y units.}
  \item{shape}{number giving shape=yheight/xwidth of the plotting region.}
  \item{verbose}{logical indicating if some diagnostic output should happen.}
}

\value{
  an object of class hexbin.
  The list structure typically contains the following components:
  \item{cell}{A vector of cell ids to supply to the table function}
  \item{xbins}{number of hexagons across the x axis.  hexagon inner
  diameter =diff(xbnds) / xbins in x units }
  \item{xbnds}{x coordinate bounds for binning and plotting}
  \item{ybnds}{y coordinate bounds for binning and plotting}
  \item{shape}{plot shape= yheight(inches) / xwidth(inches)}
}

\references{
 Carr, D. B., A. R. Olsen, and D. White  (1992)
"Hexagon Mosaic Maps for Display of Univariate and Bivariate
 Geographical Data."  \emph{Cartography and Geographic Information
 Systems}, Vol. 19 No. 4, pp. 228-236, 271
}

\details{
The plot shape must be maintained for hexagons to appear with
equal sides.  Calculations are in single precision.
}

\seealso{
hcell2xy, hgrid, hray, hexbin
}

\examples{
x <- rnorm(1000)
y <- rnorm(1000)
rangex <- range(x)
rangey <- range(y)
diffx <- diff(rangex)
diffy <- diff(rangey)
bin <- hcell(x, y, xbnds = rangex, ybnds = rangey, shape = diffy/diffx)
}
\keyword{manip}

\eof
\name{hcell2xy}
\alias{hcell2xy}
\title{Compute X and Y Coordinates for Hexagon Cells}

\description{
  Computes x and y coordinates from hexagon cell id's.
}
\usage{
hcell2xy(bin)
}

\arguments{
  \item{bin}{hexbin object}
}
\value{
  A list with two components
  \item{x}{}
  \item{y}{}
}

\references{ see in \code{\link{hcell}}.}

\details{
The hexbin object contains all the needed information.  The purpose of
this function is to reduce storage.  The cost is  additional
calculation.
}

\seealso{\code{\link{hcell}}, \code{\link{hray}},
  \code{\link{hexbin}}}
}

\examples{
x <- rnorm(10000)
y <- rnorm(10000)
bin <- hexbin(x,y)
xys <- hcell2xy(bin)
# temporal trends with confidence bounds plotted on a map
# Illustration only pending access to user functions.
# mtapply()  #like tapply but for matrices
# sens.season.slope()  #computes sen's seasonal slope

## This part does not work and commented out
#bin  <- hcell(dat$x,dat$y)   #  x and y are in map projection units
#newdat < dat[,c('month','year','value')]    # extract columns
#stats <- mtapply(newdat,bin$cell,sens.season.slope,season=12)
#plot(mymap,type='l')       # map boundaries in map projection units
#xy <- hcell2xy(bin)  # x and y coordinates for hexagon cell centers
#hray(xy$x, xy$y,val=stat[,1],lower= stat[,2],upper=stat[,3])
}

\keyword{manip}









\eof
\name{hdiffplot}
\alias{hdiffplot}
\title{Plot of Domain and Median Differences of Two `hbin' Objects}
\description{
  Let hbin1 and hbin2 represent two hexbin objects with scaling, plot
  shapes, and bin sizes. This plot distinguishes cells unique to hbin1,
  cells in common, and cells unique to hbin2 using color.  When the
  erode components are present, color also distinguishes the two erosion
  medians.  An arrow shows the vector from the median of hbin1 to the
  median of hbin2.
}

\usage{
hdiffplot(bin1, bin2, xbnds, ybnds, density,
          border = rep(FALSE, 6), pen = 2:7, size = 0.1, lwd = 2,
          eps = 1e-6, unzoom = 1.04,
          xlab = "", ylab = "", main = deparse(mycall), ...)
}

\arguments{
  \item{bin1}{an object of type hexbin.}
  \item{bin2}{an object of type hexbin.}
  \item{xbnds}{global x-axis plotting limits.  Used primarily for
    multiple comparison plots.}
  \item{ybnds}{global y-axis plotting limits.  Used primarily for
    multiple comparison plots. }
  \item{density}{fill arguments to polygon}
  \item{border}{border arguments to polygon}
  \item{pen}{pen numbers for polgyon}
  \item{size}{arrow type size in inches.}
  \item{eps}{distance criteria for distinct medians}
  \item{unzoom}{plot limit expansion factor when xbnds is missing}
%%%------- FIXME --------
  \item{lwd}{To be added}
  \item{xlab}{To be added}
  \item{ylab}{To be added}
  \item{main}{main title for the plot; automatically constructed by default.}
  \item{...}{To be added}
}

\value{
invisible("Done")
}

\details{
  The hexbin objects for comparison, \code{hbin1} and \code{hbin2}, must
  have the same plotting limits and cell size.  The plot produces a
  comparison overlay of the cells in the two objects.  If external
  global scaling is not supplied, the algorithm determines plotting
  limits to increase resolution.  For example, the objects may be the
  result of the \code{\link{erode.hexbin}()} and include only high count cells
  containing 50 of the counts.  The density, border, and pen arguments
  correspond to the polygon function calls for plotting six types of
  cells.  The cell types are respectively:
  \tabular{l}{
    unique cells of hbin1,\cr
    joint cells,\cr
    unique cells of hbin2,\cr
    median cell of hbin1,\cr
    median cell of hbin2,\cr
    median cell if identical.\cr
  }

  The \code{erode} components of the hexbin objects must be present for the
  medians to plot.  The algorithm select a single cell for the median if
  there are algorithmic ties.

  The \code{pen} numbers for types of cells start at Pen 2.  Pen 1 is
  presumed black.  The suggested six additional colors are light blue,
  light gray, light red, blue, red, and black.  The reference below
  %% Carr (1991)
  shows an example for black and white printing.  That plot changes the
  six colors to light gray, dark gray, white, black, black, and black.
  It changes the 4th, 5th, and 6th argument of border to TRUE.  It also
  changes 4th, 5th and 6th argument of density to 0.  In other words
  cells in common do not show and medians cells appear as outlines.

  When \code{xbnds} is missing, the plot changes graphics parameters and resets
  them.   The function also zooms in based on the available data to
  provide increased resolution.
}

\references{ see in \code{\link{hexagons}}.}%>> ./hexagons.Rd

\seealso{
  \code{\link{hexbin}}, \code{\link{smooth.hexbin}}, \code{\link{erode.hexbin}}, % MISSING: hthin,
  \code{\link{hcell2xy}}, \code{\link{hcell}}, \code{\link{plot.hexbin}},
  \code{\link{hboxplot}}, \code{\link{hmatplot}},
  \code{\link{hexagons}}, \code{\link{hex.legend}}.
}

\examples{
#  Comparison of two bivariate boxplots
x1 <- rnorm(10000)
y1 <- rnorm(10000)
x2 <- rnorm(10000,mean=.3)
y2 <- rnorm(10000,mean=.3)
xbnds <- range(x1,x2)
ybnds <- range(y2,y2)

bin1 <- hexbin(x1,y1,xbnds=xbnds,ybnds=ybnds)
bin2 <- hexbin(x2,y2,xbnds=xbnds,ybnds=ybnds)
erodebin1 <- erode.hexbin(smooth.hexbin(bin1))
erodebin2 <- erode.hexbin(smooth.hexbin(bin2))

hdiffplot(erodebin1,erodebin2)
}

\keyword{hplot}

\eof
\name{hex.legend}
\alias{hex.legend}
\title{plot a legend for plot.hexbin}
\description{
  Plots the legend for the plot.hexbin function. Provides a legend indicating
  the count representations.
}
\usage{
hex.legend(legend, ysize, lcex, inner, style = "colorscale",
        minarea = 0.05, maxarea = 0.8, mincnt = 1, maxcnt,
        trans = NULL, inv = NULL, colorcut,
        density = NULL, border = FALSE, pen = NULL,
        colramp = function(n){ LinGray(n,beg = 90,end = 15) })
}
\arguments{
  \item{legend}{positive number giving width of the legend in inches.}
  \item{ysize}{height of legend in inches}
  \item{lcex}{the characters expansion size for the text in the legend,
    see \code{\link{par}(cex=)}.}
  \item{inner}{the inner diameter of a hexagon in inches.}
  \item{style}{the hexagon style, one of
    ("c","lat","cm","nest","nestcm"), see \code{\link{hexagons}}.}
  \item{minarea, maxarea}{fraction of the cell area for the lowest and largest
    count, respectively.}
  \item{mincnt, maxcnt}{minimum and maximum count accepted in \code{plot}.}
  \item{trans}{a transformation for the counts such as \code{sqrt()}.}
  \item{inv}{the inverse transformation.}
  \item{colorcut}{numeric vector of values covering [0, 1] the determine
    hexagon color classes boundaries and hexagon legend size boundaries.}
  \item{border}{argument for \code{\link{polygon}()}.  Draw the border
    for each hexagon.}
  \item{density}{argument for \code{\link{polygon}()} filling.
    A \code{0} causes the polygon not to be filled.}
  \item{pen}{Polygon() col argument. Determines the color with which the
    polygon will be filled}
  \item{colramp}{function accepting an integer \code{n} as an argument and
    returning n colors.}
}
\details{
  \code{\link{plot.hexbin}} calls this \code{hex.legend} to produce a legend by
  setting the graphics parameters, so \code{hex.legend} itself is not a
  standalone function.

  The legend function is \bold{preliminary}.  Later version will include
  refinements and handle extreme cases (small and large) for cell size
  and counts.

  See the \bold{Details} section of \code{\link{hexagons}}'s help page.
}
\value{
  This function does not return any value.
}
\references{ see in \code{\link{hexagons}}.}
\author{
  Dan Carr <dcarr@voxel.galaxy.gmu.edu>

  ported by Nicholas Lewin-Koh <kohnicho@comp.nus.edu.sg>
}

\seealso{\code{\link{hexbin}}, \code{\link{smooth.hexbin}}, \code{\link{erode.hexbin}},
  \code{\link{hcell}}, \code{\link{hcell2xy}},
  \code{\link{plot.hexbin}}, \code{\link{hboxplot}}, \code{\link{hdiffplot}},
  \code{\link{hmatplot}}, \code{\link{hexagons}}}

\examples{
  # Not a stand alone function
  # Call from plot.hexbin
  hex.legend(legend = 2, ysize = 1,lcex=8,inner=0.2,
             maxcnt = 100, colorcut = c(0.5,0.5))
}
\keyword{aplot}

\eof
\name{hexagons}
\alias{hexagons}
\title{Add Hexagon Cells to Plot}
\description{
  Plots cells in an hexbin object.  The function distinquishes among
  counts using 5 different styles.
  This function is the hexagon plotting engine called from
  \code{\link{plot.hexbin}()} and \code{\link{hex.legend}()}.
}
\usage{
hexagons(dat, style = "colorscale", minarea = 0.05, maxarea = 0.8, mincnt = 1,
         maxcnt = max(dat$cnt), trans = NULL, colorcut, density = -1,
         border = FALSE, pen=NULL,
         colramp = function(n){LinGray(n,beg=90,end=15)},
         verbose = getOption("verbose"))
}
\arguments{
  \item{dat}{an object of class \code{hexbin}, see \code{\link{hexbin}}.}
  \item{style}{character specifying the type of plotting; must be one of
    ("grayscale","lattice","centroids","nested.lattice","nested.centroids").}
  \item{minarea}{The fraction of cell area for the lowest count.}
  \item{maxarea}{The fraction of the cell area for the largest count}
  \item{mincnt}{Cells with smaller counts are not shown}
  \item{maxcnt}{Cells with large counts are not shown}
  \item{trans}{A transformation for the counts such as sqrt()}
  \item{colorcut}{A vector of values covering [0, 1] which determine
    hexagon color classes boundaries or hexagon size boundaries.}
  \item{density}{Polygon() fill argument.  0 causes the polygon not to
    be filled.}
  \item{border}{Polygon() border argument.  Draw the border for each hexagon}
  \item{pen}{Polygon() col argument. Determines the color with which the
    polygon will be filled}
  \item{colramp}{A function accepting n as its argument and returning n
    colors. n is determined }
  \item{verbose}{logical indicating if some diagnostic output should happen.}
}
\section{SIDE EFFECTS}{Adds hexagons to the plot.}

\details{
  The five plotting styles have the following effect:
  \describe{
    \item{\code{style="lattice"} or \code{"centroids"}}{

      Plots the hexagons in different sizes based on counts.  The
      \code{"lattice"} version centers the hexagons at the cell centers
      whereas \code{"centroids"} moves centers the hexagons close to the
      center of mass for the cells.  In all cases the hexagons will not
      plot outside the cell unless \code{maxarea > 1}.  Counts are rescaled
      into the interval [0,1] and colorcuts determine the class
      boundaries for sizes and counts. The pen argument for this style
      should be a single color or a vector of colors of
      \code{length(bin$cnt)}.}

    \item{\code{style="colorscale"}}{
      Counts are rescaled into the interval [0,1] and colorcuts determines
      the class boundaries for the color classes. For this style the
      function specified in coloramp is used to define the n colors for
      the n+1 color cuts. In for this style the pen argument is ignored.
      %% S-plus: In motif color options try polygon:  black 16 white
    }

    \item{\code{style="nested.lattice"} and \code{"nested.centroids"}}{
      Counts are partioned into classes by power of 10.  The encoding nests
      hexagon size within powers of 10 color contours.

      If the pen argument is used it should be a matrix of colors with 2
      columns and either \code{ceiling(log10(max(bin$cnt)))} or
      \code{length(bin$cnt)} rows. The default uses the R color palatte
      so that pens numbers 2-11 determine colors for completely filled
      cell Pen 2 is the color for 1's, Pen 3 is the color for 10's, etc.
      Pens numbers 12-21 determine the color of the foreground hexagons. The
      hexagon size shows the relative count for the power of 10. Different
      color schemes give different effects including 3-D illusions
      %% S-plus :
      %%   One motif color option for the first 4 powers is black \#BBB \#36F
      %%   \#0E3 \#F206 \#FFF4 \#FFF
      %%
      %%   A second option is for the first 5 power is black \#FFF \#08F \#192
      %%   \#F11 \#FF04 \#000 \#999 \#5CF \#AFA \#FAAF \#000
    }
  }

  \emph{Hexagon size encoding \code{minarea} and \code{maxarea}}
  determine the area of the smallest and largest hexagons
  plotted.  Both are expressed fractions of the bin cell size.  Typical
  values might be .04 and 1.  When both values are 1, all plotted
  hexagons are bin cell size, if \code{maxarea} is greater than 1 than
  hexagons will overlap. This is sometimes interesting with the lattice
  and centroid styles.

  \emph{Count scaling}

  \code{relcnt <- (trans(cnt)-trans(mincnt)) / (trans(maxcnt)-trans(mincnt))}
  \cr
  \code{area <- minarea + relcnt*maxarea}

  By default the transformation \code{trans()} is the identity
  function.  The legend routine requires the transformation inverse
  for some options.

  \emph{Count windowing \code{mincnt} and \code{maxcnt}}
  Only routine only plots cells with cnts in [mincnts,   maxcnts]
}
\references{
  Carr, D. B. (1991)
  Looking at Large Data Sets Using Binned Data Plots,
  pp. 7--39 in \emph{Computing and Graphics in Statistics};
  Eds. A. Buja and P. Tukey, Springer-Verlag, New York.
}
\author{
  Dan Carr <dcarr@voxel.galaxy.gmu.edu>
  ported and extended by Nicholas Lewin-Koh <nikko@hailmail.net>}
\note{}

\seealso{\code{\link{hexbin}}, \code{\link{smooth.hexbin}},
  \code{\link{erode.hexbin}}, \code{\link{hcell}}, \code{\link{hcell2xy}},
  \code{\link{plot.hexbin}}, \code{\link{hboxplot}}, \code{\link{hdiffplot}},
  \code{\link{hmatplot}}, \code{\link{hex.legend}}}

\examples{
#  A quick example

set.seed(506)
x <- rnorm(10000)
y <- rnorm(10000)

# bin the points
bin <- hexbin(x,y)

## setup coordinate system:
plot(bin$xbnd, bin$ybnd, type="n", main = "Bivariate rnorm(10000)", asp=1)
hexagons(bin)

#  A better approach uses plot.hexbin
#   and controls the plot shape
plot(bin)

# A mixture distribution
x <- c(rnorm(5000),rnorm(5000,4,1.5))
y <- c(rnorm(5000),rnorm(5000,2,3))
bin <- hexbin(x,y)

# Show color control and overplotting of hexagons
plot(bin$xbnd, bin$ybnd, type="n", main = "Bivariate mixture (10000)", asp=1)
hexagons(bin, style= "lattice", border = gray(.1), pen = gray(.6),
         minarea = .1, maxarea = 1.5)

# And if we had all the information...
if(require(gpclib)){
  h1<-chull(x[1:5000],y[1:5000])
  h2<-chull(x[5001:10000],y[5001:10000])
  h2<-h2+5000
  h1<-as(cbind(x[1:5000],y[1:5000])[h1, ], "gpc.poly")
  h2<-as(cbind(x,y)[h2, ], "gpc.poly")
  plot(bin$xbnd, bin$ybnd, type="n", main = "Bivariate mixture (10000)", asp=1)
  plot(h1,poly.args = list(col ="#CCEBC5"),add = TRUE)
  plot(h2,poly.args = list(col ="#FBB4AE"),add = TRUE)
  plot(intersect(h1, h2), poly.args = list(col = 2), add = TRUE)  
  hexagons(bin, style= "lattice", border = gray(.1), pen = gray(.6),
           minarea = .1, maxarea = 1.5)
}

}
\keyword{aplot}

\eof
\name{hexbin}
\alias{hexbin}
\title{bivariate binning into hexagon cells}
\description{
  Creates an list object. Basic components are a cell id and a count
  of points falling in each occupied cell.
}
\usage{
hexbin(x, y, xbins=30, shape=1, xbnds=range(x), ybnds=range(y))
}
\arguments{
  \item{x}{a vector of numbers.  The first coordinate of bivariate
      data to be binned.}
  \item{y}{a vector of numbers.  The second coordinate of bivariate
      data to be binned.}
  \item{xbins}{The number of bins partitioning the range of xbnds.}
  \item{shape}{The shape=yheight/xwidth of the plotting regions.}
  \item{xbnds}{The horizontal limits of the binning region in x units.}
  \item{ybnds}{The  vertical limits of the binning region in y units.}
}
\value{
  an object of type hexbin
  The list structure typically contains the following components:
  \item{cell}{A vector of cell ids that can be mapped into the (x,y)
      bin centers in data units.}
  \item{cnt}{A  vector of counts in the cells.}
  \item{xcm}{The x center of mass (average of x values) for the cell.}
  \item{ycm}{The y center of mass (average of y values) for the cell.}
  \item{xbins}{ number of hexagons across the x axis. hexagon inner
      diameter =diff(xbnds)/xbins in x units}
  \item{xbnds}{ x coordinate bounds for binning and plotting}
  \item{ybnds}{ y coordinate bounds for binning and plotting}
  \item{shape}{ plot shape= yheight(inches)/xwidth(inches)}
  \item{dim}{The i and j limits of cnt treated as a matrix cnt[i,j]}
}

\seealso{
  smooth.hexbin, erode.hexbin, hthin, hcell2xy, hcell,
  plot.hexbin, hboxplot, hdiffplot, hmatplot, hexagons, hex.legend
}

\references{
  Carr, D. B. et al. (1987)
  "Scatterplot Matrix Techniques for Large N",
  \emph{JASA} \bold{83}, 398, 424--436.
}

\details{
Returns counts for non-empty cells.  The plot shape must be maintained for
 hexagons to appear with equal sides.  Calculations are in single precision.
}

\examples{
x <- rnorm(10000)
y <- rnorm(10000)
bin <- hexbin(x,y)

## Using plot method for hexbin objects:
plot(bin, style = "nested.lattice")
# lower resolution binning and overplotting with counts
bin <- hexbin(x,y,xbins=25)
screenpar <- plot.hexbin(bin, style="lattice", minarea=1, maxarea=1,
                   density=0, border=TRUE)
oldpar <- par(screenpar)     # reset graphics to the plot on the screen
xy <- hcell2xy(bin)
text(xy$x,xy$y,as.character(bin$cnt),adj=.5,cex=.3)
points(x,y)       # to show points rather than counts
par(oldpar)       #reset graphics
}

\keyword{dplot}


\eof
\name{hmatplot}
\alias{hmatplot}
\title{Draw Matrix of 2-D Boxplots For Hexbin Objects}
\description{
  Produces a matrix of 2-D boxplots for hexbin objects classified
  by one or two factors.  The function also interposes difference plots
  between adjacent boxplots.

  This is a two-way (or one-way)  multiple panel plot.  The panels with
  odd row and column numbers show 2-D boxplots produce by
  hboxplot(). Panels between boxplots show the difference of adjacent
  2-D boxplots using see hdiffplot().
}
\usage{
hmatplot(namemat, rlabels, clabels,
         brd = c(0.5, 0.7, 0.7, 0.5),
         mai = rep(0.4, 4), unzoom = 1.04, cex = 1.5,
         border = list(hbox=c(FALSE,FALSE),hdiff=rep("white",6)),
         pen = list( hbox = c(2, 3),hdiff = 2:7 ),
         density = c(-1, -1, -1, 0, 0, 0), size = 0.06, lwd = 2)
}
\arguments{
  \item{namemat}{a matrix of  hexbin object names}
  \item{rlabels}{row labels for the matrix}
  \item{clabels}{column labels for the matrix}
  \item{brd}{border space for the plot.}
  \item{mai}{margin for individual panels, (must be smaller that brd)}
  \item{unzoom}{plot limit expansion factor when xbnds is missing}
  \item{cex}{character expansion size for labels}
  \item{border}{a list with elements "hbox" which are the arguments to
hboxplot() and "hdiff" which are arguments to hdiffplot()}
  \item{pen}{a list with elements "hbox" which are the arguments to
hboxplot() and "hdiff" which are arguments to hdiffplot()}
  \item{density}{argument to hdiffplot()}
  \item{size}{arrow tip size in hdiffplot()}
  \item{lwd}{arrow thinkness in hdiffplot()}
}
\details{
  Namemat is a matrix of hexbin object names corresponding to
  classification by two factors.  Typically the hexbin objects are the
  result of identically scaled binning, smoothing and erosion.  However,
  data variations lead to different contours in the boxplots.  This
  function determines global zoom-in scaling that covers all of
  hboxplot() results.

  Be careful that rlabels and clabels correspond to the bin objects.

  hboxplot() uses pen 2 median and pen 3 as the default.
  hdiffplot() used pens 4 to 9  for the respective cells as the default.

  Suggested polygon colors \#000 \#000 \#AAA \#58F \#FFF \#F11 \#FFF
  \#FFF.
  Suggested background: white
  Suggested lines : black

  The default turns off the polygon fill for the medians in hdiffplot.
}
\value{
  \item{list(xbnds,ybnds,xsize,ysize,plot.par)}{xbnds and ybnds given the ranges
    for plotting a single plot in data units. xsize and ysize give the
    size of a single plot in inches. plot.par is a list of lists, where
    each list is the plot parameters of the hboxplots, and the
    hdiffplots for rows and columns respectively}
}

\references{ see in \code{\link{hexagons}}.}

\author{
  Dan Carr <dcarr@voxel.galaxy.gmu.edu>
  ported by Nicholas Lewin-Koh <kohnicho@comp.nus.edu.sg>
}
\seealso{\code{\link{hexbin}}, \code{\link{smooth.hexbin}}, \code{\link{erode.hexbin}},
  \code{\link{hcell2xy}},  \code{\link{hcell}},
  \code{\link{plot.hexbin}}, \code{\link{hboxplot}}, \code{\link{hmatplot}},
  \code{\link{hexagons}}, \code{\link{hex.legend}}}

\examples{
# nhanes data set
# remove missing data, and extract variable of interest
# make this so you can do data(nh)

data(NHANES)# pretty large data set!
good <- !(is.na(NHANES$Albumin) | is.na(NHANES$Transferin))
NH.vars <- NHANES[good, c("Age","Sex","Albumin","Transferin")]

# extract dependent variables and find  ranges for global binning
x <- NH.vars[,"Albumin"]
rx <- range(x)
y <- NH.vars[,"Transferin"]
ry <- range(y)

# define index for the twoway layout
age <- cut(NH.vars$Age,c(1,45,65,200))
sex <- NH.vars$Sex
subs <- tapply(age,list(age,sex))

#  bivariate bins for each factor combination
for (i in 1:length(unique(subs))) {
  good <- subs==i
  assign(paste("nam",i,sep=""),
         erode.hexbin(hexbin(x[good],y[good],xbins=23,xbnds=rx,ybnds=ry)))
}

# define arguments to hmatplot
nam <- matrix(paste("nam",1:6,sep=""),ncol=3,byrow=TRUE)
rlabels <-c("Females","Males")
clabels  <- c("Age <= 45","45 < Age <= 65","Age > 65")
zoom <- hmatplot(nam,rlabels,clabels,border=list(hbox=c("black","white"),hdiff=rep("white",6)))

# enlarged legend
hboxplot(nam1, zoom$xbnds,zoom$ybnds, reshape=TRUE, xlab= "Albumin (mg/dl)",
         ylab="Transferrin Sat. (\)", cex=1.2)

str(lapply(NHANES, function(x){if(
is.numeric(x)){cat(names(x),": NA ", sum(is.na(x))," Obs: ",sum(!is.na(x)),"\n")}}))
}
\keyword{hplot}

\eof
\name{hray}
\alias{hray}
\title{represent values and confidence bounds on a map using rays}
\description{
  Adds  rays and optional confidence arcs (sectors)  to an existing plot.
}
\usage{
hray(x, y, val, lowbnd = NULL, highbnd = NULL,
     minval = if(is.null(lowbnd)) min(val) else min(lowbnd),
     maxval = if(is.null(highbnd)) max(val) else max(highbnd),
     raylength = 0.3, raylwd = 2, raypen = 1, dotlength = 0.06,
     dotsides = 8, dotlwd = 1, dotden = 0, dotborder = TRUE,
     dotpen = 1,
     ticlength = 0.06, ticsides = 8, ticlwd = 1, ticpen = 1,
     arclength = 0.75, arcsides = 18, arclwd = 1, arcden = -1,
     arcborder = FALSE, arcpen = 2,
     minangle =  - pi/2, maxangle = pi/2, clockwise = FALSE)
}
\arguments{
  \item{x}{vector of xcoordinates for plotting}
  \item{y}{vector of ycoordinates for plotting}
  \item{val}{vector of values to translate into ray angles}
  \item{lowbnd}{The lower bound}
  \item{highbnd}{The higher bound}
  \item{minval}{Constant for scaling val.  Minval maps into minangle.}
  \item{maxval}{Constant for scaling val.  Maxval maps into maxangle.}
  \item{raylength}{Constant or vector giving the ray length in inches.
    No rays plotted if 0 or FALSE.}
  \item{raylwd}{Ray line width}
  \item{raypen}{Ray color number}
  \item{dotlength}{Radius of dot in inches, No dots plotted if 0 or FALSE}
  \item{dotsides}{Number of polygon sides for the "dot"}
  \item{dotlwd}{Dot line width}
  \item{dotden}{Dot file option (see polygon)}
  \item{dotborder}{Logical, plot border of dot (see polygon)}
  \item{dotpen}{Dot color number}
  \item{ticlength}{Radius of tic in inches.  No tics plotted if 0 or FALSE}
  \item{ticsides}{Number of tic radii.}
  \item{ticlwd}{Tic line width}
  \item{ticpen}{Tic color number}
  \item{arclength}{Sector radius in inches.  No sector plotted if
    is.null(lowbnd) }
  \item{arcsides}{Number of polygon sides for the maximum sector}
  \item{arclwd}{Sector line width}
  \item{arcden}{Sector fill option (see polygon)}
  \item{arcborder}{Logical, plot border of sector (see polygon)}
  \item{arcpen}{Sector color number}
  \item{minangle}{Constant:  Minimum angle for rays or ray bounds (radians)}
  \item{maxangle}{Constant:  Maximum angle for ryas or ray bounts (radians)}
  \item{clockwise}{Logical: Direction from minangle to maxangle }
}
\details{
  This function has been made very general to support contining
  experiments.  The addition of confidence arcs is a useful.
  However,the big arcs call attention to the least accurate of the
  estimates. Consequently experiments are in progress to use a shorter
  radii for less accurate estimates.

  If raylength (inches) is a constant the function replicates it to be a
  vector.  When the function plots confidence bounds, the arclength is
  treated in one of two ways.  If arclength is a constant (fraction),
  the function recalculates the arclength as a fraction of the
  raylength.  Otherwise the function uses the arclength (inches) as
  is. Both ray and arc length can be data dependent.

  The tic option provides reference angles to help people to read angles
  more accurately.  The argments control standard Splus graphics arguments.

  The dot option puts a dot at the base of the ray.  Typically an open
  dot surrounds the tics.  The arguments control standar Splus graphics
  arguments.

  Outliers are not so easily spotted when encoded as rays.  To highlight
  selected observations, determine global scaling, call hray with each
  data subset, and change graphics arguments such as pen (and associated
  color) , length and lwd arguments.
}
\value{
  This function does not return any value.
}
\section{SIDE EFFECTS:}{
  Adds ray, confidence arcs, angle reference tics and
  based dot to a plot.
}
\references{ see in \code{\link{hcell}}.}

\author{
  Dan Carr <dcarr@voxel.galaxy.gmu.edu>

  ported by Nicholas Lewin-Koh <kohnicho@comp.nus.edu.sg>
}

\seealso{\code{\link{hcell}}, \code{\link{hcell2xy}}}

\examples{
# Random  locations values and bounds
x <- rnorm(100)
y <- rnorm(100)
val <- rnorm(100)
inc <- abs(rnorm(100,sd=.4))
lowbnd <- val-inc
highbnd <- val+inc

#no confidence bounds
plot(x,y,type='n')
hray(x,y,val)

#confidence bounds
plot(x,y,type='n')
hray(x,y,val,lowbnd,highbnd)

# Clockwise orientation
plot(x,y,type='n')
hray(x,y,val,lowbnd,highbnd)

# No tics and small filled dots
plot(x,y,type='n')
hray(x,y,val,lowbnd,highbnd,ticlength=0,dotside=20,dotlength=.025,dotden=-1)

# bivariate rays for smoothed data on a hexagon grid
# Data not provided
#plot(mymap,type='l') ## Do not work. Commented out.
#hray(grid$x,grid$y,pred$so4)
#hray(grid$x,grid$y,pred$no3,clockwise=TRUE,tics=FALSE)
## End of commented out code

# temporal trends with confidence bounds plotted on a map
# Data and user function not provided
#  mtapply()   like tapply but for matrices
#  sens.season.slope()  computes sen's seasonal slope
#bin  <- hcell(x,y)
#  x and y are in map projection units
#newdat < dat[,c('month','year','value')]
# extract columns
#stats <- mtapply(newdat,bin$cell,sens.season.slope,season=12)
#plot(mymap,type='l')
# map boundaries in map projection units
#xy <- hcell2xy(bin)
# x and y coordinates for hexagon cell centers
#hray(xy$x, xy$y,stat[,1],lowbnd= stat[,2],highbnd=stat[,3])
}
\keyword{aplot}

\eof
\name{identify.hexbin}
\alias{identify.hexbin}
\title{Identifying Hexagons on a Hexbin Graphic}
\description{
  Identify reads the position of the graphics pointer (typicaly
  controlled by the \emph{mouse}) when the (first mouse) button is
  pressed.  It then searches the coordinates of the hexagon centers for
  the coordinate closest to the pointer.  If this coordinate is close to
  the pointer, its index will be returned as part of the value of the call.
}
\usage{
\method{identify}{hexbin}(x, labels, offset=0, ...)
}
\arguments{
  \item{x}{a \code{\link{hexbin}} object.}
  \item{labels}{vector giving labels for each of the hexagons.  If
    supplied, it must have the same length as \code{x$n}.  The default
    is to use the cell counts.}
  \item{offset}{the distance (in character widths) which separates the
    label from identified points.}
  \item{\dots}{further arguments to \code{\link{par}()} and
    \code{\link{identify.default}()}.}
}
\value{
  If pos is FALSE, an integer vector containing the indexes of the
  identified points.  If pos is TRUE, a list containing a component ind,
  indicating which points were identified and a component pos,
  indicating where the labels were placed relative to the identified
  points.
}
\details{
  If in addition, plot is TRUE, the point is labelled with the
  corresponding element of text.

  The labels are placed either below, to the left, above or to the right
  of the identified point, depending on where the cursor was.

  The identification process is terminated by pressing any mouse button
  other than the first.

  If the window is resized or hidden and then exposed before the
  identification process has terminated, any labels drawn by identify will
  disappear. These will reappear once the identification process has
  terminated and the window is resized or hidden and exposed again. This
  is because the labels drawn by identify are not recorded in the device's
  display list until the identification process has terminated.
}
\seealso{
   \code{\link{identify}}, \code{\link{plot.hexbin}},
   \code{\link{hexagons}}, \code{\link{hexbin}}.
}
\author{Nicholas Lewin-Koh \email{nikko@hailmail.net}}
\examples{
if(dev.interactive()) {
  x <- rnorm(1000)
  y <- rnorm(1000)
  hb <- hexbin(x,y)
  plot(hb)
  identify(hb)
  ## Now with user supplied labels
  identify(hb,hb$cells)
}
}%ex
\keyword{iplot}

\eof
\name{plot.hexbin}
\alias{plot.hexbin}
\title{plotting hexagon cells and legend}
\description{
  Plots the counts in an hexbin object using 5 different styles. Provides
  a legend indicating the count representations.
}
\usage{
plot.hexbin(x, style = "colorscale", legend = 1, lcex = 1,
      minarea = 0.04, maxarea = 0.8, mincnt = 1, maxcnt = max(bin$cnt),
      trans = NULL, inv = NULL,
      colorcut = seq(0, 1, length = min(17, maxcnt)),
      border = FALSE, density = NULL, pen = NULL,
      colramp = function(n){ LinGray(n,beg = 90,end = 15) },
      xlab = "", ylab = "", verbose = getOption("verbose"), \dots)
}
\arguments{
  \item{x}{an object of type hexbin.}
  \item{style}{string, one of \code{("colorscale", "lattice", "centroids",
      "nested.lattice", "nested.centroids")}, see \code{\link{hexagons}}.}
  \item{legend}{width of the legend in inches.  If False or 0 the
    legend is not produced.}
  \item{lcex}{characters  expansion size for the text in the legend}
  \item{minarea}{fraction of cell area for the lowest count}
  \item{maxarea}{fraction of the cell area for the largest count}
  \item{mincnt}{cells with fewer counts are ignored.}
  \item{maxcnt}{cells with more counts are ignored.}
  \item{trans}{\code{\link{function}} specifying a transformation for
    the counts such as \code{sqrt}.}
  \item{inv}{the inverse transformation of \code{trans}.}
  \item{colorcut}{vector of  values covering [0, 1] that determine
    hexagon color class boundaries and hexagon legend size boundaries. }
  \item{border}{Polygon() border argument.  Draw the border for each hexagon}
  \item{density}{Polygon() fill argument.  0 causes the polygon not to
    be filled.}
  \item{pen}{Polygon() col argument. Determines the color with which the
    polygon will be filled}  
  \item{colramp}{function accepting an integer \code{n} as an argument and
    returning n colors.}
  \item{xlab}{xlabel argument to plot().}
  \item{ylab}{ylabel argument to plot().}
  \item{verbose}{logical indicating if some diagnostic output should happen.}
  \item{\dots}{additional arguments from and passed to  \code{plot} methods.}
}
\details{
  The legend function is preliminary.  Later version will include
  refinements and handle extreme cases (small and large) for cell size
  and counts.

  For \code{style}, \code{minarea} etc, see the \bold{Details} section of
  \code{\link{hexagons}}'s help page.


  \emph{Graphics parameters}

  This function determines the plot shape, so hexagons appear as
  hexagons.  The process alters graphics parameters.  The parameters are
  reset to that before the function call when the function exits.  The
  function returns the parameters for plotting the hexagons
  invisibly.  Users can store the parameters and reinvoke them to add to
  the plot.  See the third example below.
}
\value{
  invisible(par)
}

\references{ see in \code{\link{hexagons}}.}
\author{
  Dan Carr <dcarr@voxel.galaxy.gmu.edu>
  ported by Nicholas Lewin-Koh <kohnicho@comp.nus.edu.sg>
}
\seealso{\code{\link{hexbin}}, \code{\link{smooth.hexbin}}, \code{\link{erode.hexbin}},
  \code{\link{hcell}}, \code{\link{hcell2xy}},
  \code{\link{hboxplot}}, \code{\link{hdiffplot}},
  \code{\link{hmatplot}}.
}
\examples{
# simple binning
x <- rnorm(10000)
y <- rnorm(10000)
bin <- hexbin(x,y)

# basic plot
plot(bin)
# nested lattice
plot(bin, style= "nested.lattice")

# controlling the colorscheme
plot(bin,colramp=BTY,colorcut=c(0,.1,.2,.3,.4,.6,1))

# A mixture distribution
x <- c(rnorm(5000),rnorm(5000,4,1.5))
y <- c(rnorm(5000),rnorm(5000,2,3))
bin <- hexbin(x,y)

pens<-c("#ECE2F0","#A6BDDB","#1C9099","#FFF7BC","#FEC44F","#D95F0E")
pens<-matrix(pens,3,2)
plot(bin, style = "nested.lattice",pen=pens)
# now really crazy
plot(bin, style = "nested.lattice",pen=pens,border=2,density=35)

# lower resolution binning and overplotting with counts
oldpar <- par(no.readonly=TRUE)
bin <- hexbin(x,y,xbins=25)
scrPar <- plot.hexbin(bin, style= "lattice",legend=FALSE,
                      minarea=1,maxarea=1,border="white")
par(scrPar) # reset graphics to the plot on the screen
xy <- hcell2xy(bin)
text(xy$x,xy$y,as.character(bin$cnt),adj=.5,cex=.3,col="red")
points(x,y,pch=18,col="red") # to show points rather than counts
par(oldpar) # reset graphics

# Be creative, have fun!

}
\keyword{hplot}

\eof
\name{rgb2hsv}
\alias{rgb2hsv}
\title{Numeric Conversion of RGB to HSV Color Space}
\description{
  Given a 3-row matrix of RGB (Red, Green, Blue) values in \eqn{[0,1]} return
  a similar matrix of HSV (Hue, Saturation, Value) values.
}
\usage{
rgb2hsv(r, g = NULL, b = NULL, delta = 1)
}
\arguments{
  \item{r}{vector of \dQuote{red} values in \eqn{[0,1]},
    or 3-row rgb matrix.}
  \item{g}{vector of \dQuote{green} values, or \code{\link{NULL}} when
    \code{r} is a matrix.}
  \item{b}{vector of \dQuote{blue} values, or \code{\link{NULL}} when
    \code{r} is a matrix.}
  \item{delta}{a \dQuote{gamma correction}.}
}
\value{
}
%\references{ ~put references to the literature/web site here ~ }
\seealso{\code{\link{hsv}}, \code{\link{rgb}}, \code{\link{col2rgb}}.}
\examples{
256* (rgb3 <- floor(256 * matrix(runif(3*12), 3,12))/256)
(hsv3 <- rgb2hsv(rgb3))
rgbBack <- col2rgb(hsv(h=hsv3[1,], s=hsv3[2,], v=hsv3[3,]))
table(256*rgb3 - rgbBack)# not all 0; I had about half 1's
stopifnot(max(1 - c(256*rgb3) / c(rgbBack)) < 0.01)
}
\keyword{color}


\eof
\name{smooth.hexbin}
\alias{smooth.hexbin}
\title{Hexagon Bin Smoothing}
\description{
  Compute a discrete kernel smoother that covers seven cells, namely a
  center   cell and its six neighbors.  With two iterations effective the kernel
  covers 1+6+12=19 cells.
}

\usage{
smooth.hexbin(bin, wts=c(48,4,1))
}

\arguments{
  \item{bin}{an object of type hexbin}
  \item{wts}{numeric vector of length 3 for relative weights of the
    center, the six neighbor cells, and twelve second neighbors.}
}

\value{
  an object of class hexbin
  The object includes the additional component \code{wt}.
}
\references{ see in \code{\link{hexagons}}.}

\details{
  This discrete kernel smoother uses the center cell, immediate
  neighbors and second neighbors to smooth the counts.  The counts for
  each resulting cell is a linear combination of previous cell counts
  and weights.  The weights are
  \tabular{ll}{
    1 center cell,       \tab weight = wts(1)\cr
    6 immediate neighbors\tab weight = wts(2)\cr
    12 second neighbors  \tab  weight =wts(3)\cr
  }
  If a cell, its immediate and second neighbors all have a value of
  max(cnt), the new maximum count would be max(cnt)sum(wt).  It is
  possible for the counts to overflow.

  The domain for cells with positive counts increases.  The hexbin
  components, xbins, xbnds, ybnds, and dim all reflect this increase.
  Note the typically dim[2]=xbins+1

  The intent was provide a fast, iterated immediate neighbor smoother.
  However, the current hexbin plotting routines only support shifting even
  numbered rows to the right.  Future work can

 (1) can add a shift indicator to hexbin objects that indicates left or
     right shifting.\cr
 (2) generalize plot.hexbin() and hexagons()\cr
 (3) provide an iterated kernel.\cr

 With wt(3)=0, the smoother only uses the immediate neighbors.  With a
 shift indicator the domain could increase by 2 rows (one bottom and on
 top) and 2 columns (one left and one right).  However the current
 implementation increases the domain by 4 rows and 4 columns, thus
 reducing plotting resolution.
}

\seealso{
hexbin,  erode.hexbin,  hthin, erode.hexbin, hcell2xy, hcell, plot.hexbin, hboxplot,
hdiffplot, hmatplot,  hexagons, hex.legend
}
\examples{
x <- rnorm(10000)
y <- rnorm(10000)
bin <- hexbin(x,y)
# show the smooth counts in gray level
smbin  <- smooth.hexbin(bin)
plot.hexbin(smbin)
# Compare the smooth and the origin
smbin1 <- smbin
smbin1$cnts <- ceiling(smbin$cnts/sum(smbin$wts))
plot.hexbin(smbin1)
smbin2 <- smooth.hexbin(bin,wts=c(1,0,0))  # expand the domain for comparability
plot.hexbin(smbin2)
}

\keyword{misc}





\eof
\name{summary.hexbin}
\alias{summary.hexbin}
\title{Summary function for hexbin objects}
\description{
  Returns a summary in the form of a formatted table of a hexbin object
}
\usage{
\method{summary}{hexbin}(object, \dots)
}
\arguments{
  \item{object}{an object of class \code{\link{hexbin}}}
  \item{\dots}{further arguments passed from and to methods.}
}
\details{
  Prints a summary to the terminal of a hexbin object with the
  components listed listed below.
  \item{call}{the call that produced the hexbin object}
  \item{dim}{the dimensions of the hexagon grid}
  \item{stats}{A summary of the grid cells, cell counts, and the cell
    center x and y coordinates }
}
\author{Nicholas Lewin-Koh \email{nikko@hailmail.net}}
\seealso{\code{\link{hexbin}}, \code{\link{summary}}.}
\examples{
  x <- rnorm(1000)
  y <- rnorm(1000)
  hb <- hexbin(x,y)
  summary(hb)
}
\keyword{print}


\eof
