.packageName <- "annotate"
    # Defines the chromLocation class

    # Define the class structure of the chromLocation object
    setClass("chromLocation", representation(organism="character",
                                             dataSource="character",
                                             chromLocs="list",
                                             probesToChrom="environment",
                                             chromInfo="numeric",
                                             geneSymbols="environment"
                                             ))

    # Define the accessors
    if (is.null(getGeneric("organism")))
        setGeneric("organism", function(object)
                   standardGeneric("organism"))

    setMethod("organism", "chromLocation", function(object)
              object@organism)

    if (is.null(getGeneric("dataSource")))
        setGeneric("dataSource", function(object)
                   standardGeneric("dataSource"))

    setMethod("dataSource", "chromLocation", function(object)
              object@dataSource)

    if (is.null(getGeneric("nChrom")))
        setGeneric("nChrom", function(object)
                   standardGeneric("nChrom"))

    setMethod("nChrom", "chromLocation", function(object)
              length(object@chromInfo))

    if (is.null(getGeneric("chromNames")))
        setGeneric("chromNames", function(object)
                   standardGeneric("chromNames"))

    setMethod("chromNames", "chromLocation", function(object)
              names(object@chromInfo))

    if (is.null(getGeneric("chromLocs")))
        setGeneric("chromLocs", function(object)
                   standardGeneric("chromLocs"))

    setMethod("chromLocs", "chromLocation", function(object)
              object@chromLocs)

    if (is.null(getGeneric("chromLengths")))
        setGeneric("chromLengths", function(object)
                   standardGeneric("chromLengths"))

    setMethod("chromLengths", "chromLocation", function(object) {
        z <- as.numeric(object@chromInfo)
        ## Unknown chromosome lengths come out as NA from the
        ## data package, put this as 0 as we want a numeric vector
        z[is.na(z)] <- 0
        z
    })

    if (is.null(getGeneric("probesToChrom")))
        setGeneric("probesToChrom", function(object)
                   standardGeneric("probesToChrom"))

    setMethod("probesToChrom", "chromLocation", function(object)
              object@probesToChrom)

    if (is.null(getGeneric("chromInfo")))
        setGeneric("chromInfo", function(object)
                   standardGeneric("chromInfo"))
    setMethod("chromInfo", "chromLocation", function(object)
              object@chromInfo)

    if (is.null(getGeneric("geneSymbols")))
        setGeneric("geneSymbols", function(object)
                   standardGeneric("geneSymbols"))
    setMethod("geneSymbols", "chromLocation", function(object)
              object@geneSymbols)

    setMethod("show", "chromLocation", function(object) {
        cat("Instance of a chromLocation class with the following fields:\n")
        cat("\tOrganism: ", organism(object), "\n\t")
        cat("Data source: ", dataSource(object), "\n\t")
        cat("Number of chromosomes for this organism: ", nChrom(object), "\n\t")

        ## Build up a matrix of chromosome names & their locations
        cat("Chromosomes of this organism and their lengths in base pairs:")
        cNames <- chromNames(object)
        cLens <- chromLengths(object)
        for (i in 1:nChrom(object)) {
            cat("\n\t\t",cNames[i],":",cLens[i])
        }
        cat("\n")
    })
##takes an environment/hash table with the chrom locations and
##named list, one element for each distinct chromosome name and
##each element of that list is a named vector, the names are the
##probeids and the values are the locations
CHRLOC2chromLoc <- function(chrEnv) {
    chrLocs <- contents(chrEnv)

    ## Need to extract out the ones w/ multiple mappings
    chrLens <- sapply(chrLocs, length)
    multis <- split(chrLens, factor(chrLens))

    ## First handle the single mapped genes
    singleNames <- names(multis$"1")
    singleLocs <- chrLocs[singleNames]
    chromNames <- unlist(sapply(singleLocs, function(y) {
        if (is.na(y))
            y
        else
            names(y)
    }))
    chromNames <- factor(chromNames)
    a <- split(singleLocs, chromNames)
    chrLocList <- lapply(a, function(x) {g <- unlist(lapply(x, function(y)
                                                        {names(y) <- NULL; y})); g})

    ## Now handle the multi mapped genes
    ## !!! FIXME:
    ## !!! This is *very* inefficient.  Make this better
    ## !!!
    if (length(multis) > 1) {
        for (i in 2:length(multis)) {
            curNames <- names(multis[[i]])
            curLocs <- chrLocs[curNames]
            for (j in 1:length(curLocs)) {
                curGene <- curLocs[[j]]
                curGeneChroms <- names(curGene)
                names(curGene) <- rep(curNames[j],length(curGene))
                for (k in 1:length(curGene))
                    chrLocList[[curGeneChroms[k]]] <-
                        c(chrLocList[[curGeneChroms[k]]], curGene[k])
            }
        }
    }


    chrLocList
}

buildChromLocation <- function(dataPkg) {
    if (!require(dataPkg, character.only=TRUE))
        stop(paste("Package:",dataPkg,"is not available"))

    pEnv <- paste("package",dataPkg,sep=":")

    chrLocList <- CHRLOC2chromLoc(get(paste(dataPkg,"CHRLOC",sep=""), pos=pEnv))

    ## !!! Need to get the version info for dataSource
    newCC <- new("chromLocation",
                 organism=get(paste(dataPkg,"ORGANISM",sep=""),pos=pEnv),
                 dataSource=dataPkg,
                 chromLocs=chrLocList,
                 chromInfo=get(paste(dataPkg,"CHRLENGTHS",sep=""),pos=pEnv),
                 probesToChrom=get(paste(dataPkg,"CHR",sep=""),pos=pEnv),
                 geneSymbols=get(paste(dataPkg,"SYMBOL",sep=""),pos=pEnv))

    return(newCC)
}

usedChromGenes <- function(eSet, chrom, specChrom) {
    ## Passed an instance of an exprSet, a chromosome name, and
    ## an instance of a chromLocation object - will return the
    ## set of genes in the eset that exist on the named chromosome,
    ## ordered by location

    ## Extract the gene names of the chromosome of interest
    cLocs <- chromLocs(specChrom)
    genes <- cLocs[[chrom]]

    ## Extract out of the expr set the genes that belong on this chrom
    usedGenes <- genes[names(genes) %in% geneNames(eSet)]

    ## Order the genes by location
    ord <- order(abs(usedGenes))
    usedGenes <- as.list(usedGenes[ord])

    return(usedGenes)
}


.fillGenEnv <- function(chromLocList, chromLocEnv) {
# Given a chromLocs list, will fill the chromLocEnv with the appropriate
# data

    for (i in 1:length(chromLocList)) {
        # Convert the numeric name to the character name
        newName <- labels(chromLocList[i]);

        # Get the gene names and location data
        chromData <- chromLocList[[i]]
        geneNames <- labels(chromData)
        newStrand <- vector(mode="character", length=length(geneNames))
        newPos <- as.numeric(chromData[geneNames])
        newStrand <- ifelse(newPos>0, "+", "-")
        newPos <- abs(newPos)

        newLoc <- list()

        for (j in 1:length(geneNames)) {
            ## Instantiate a new location object for this gene
            newLoc[[j]] <- new("chromLoc",
                               chrom=newName,position=newPos[j],
                               strand=newStrand[j])
        }
        multiassign(geneNames, newLoc, env=chromLocEnv)
    }
}
findNeighbors <- function(chrLoc, llID, chromosome, upBase, downBase,
                          mergeOrNot = TRUE){

    require(chrLoc, character.only = TRUE) ||
                           stop(paste("Chromomosome location chrLoc",
                                      "is not available on the system",
                                      "Either build one or get one from",
                                      "BioConductor"))

    if(checkArgs(llID, chromosome, upBase, downBase) == "swap"){
        temp <- upBase
        upBase <- downBase
        downBase <- temp
    }
    upBase <- as.numeric(ifelse(missing(upBase), 0, upBase))
    downBase <- as.numeric(ifelse(missing(downBase), 0, downBase))
    if(missing(chromosome)){
        chromosome <- findChr4LL(llID, get(paste(chrLoc,
                                                 "LOCUSID2CHR", sep = "")),
                                gsub("CHRLOC", "", chrLoc))
    }
    if(!missing(llID)){
        # Find the location for the target gene
        location <- as.numeric(get(llID, get(paste(chrLoc,
                                      chromosome, "START", sep = ""))))
    }else{
        location <- (downBase - upBase)/2
    }
    upperB <- getBoundary(location, upBase, TRUE)
    downB <- getBoundary(location, downBase, FALSE)
    neighbors <- list()
    # There may be chances that a llID be mapped to genes on different CHR
    for(i in chromosome){
        start <- unlist(multiget(ls(get(paste(chrLoc, chromosome,
                                              "START", sep = ""))),
                                 get(paste(chrLoc, chromosome,
                                           "START", sep = ""))),
                        use.names = TRUE)
        end <- unlist(multiget(ls(get(paste(chrLoc, chromosome,
                                              "END", sep = ""))),
                                 get(paste(chrLoc, chromosome,
                                           "END", sep = ""))),
                        use.names = TRUE)
        if(!missing(llID)){
            # greb the ones in the range
            foundUp <- weightByConfi(start[start > upperB &
                                           start < min(location)])
            foundDown <- weightByConfi(end[end < downB &
                                           end > max(location)])
            if(length(foundUp) != 0 || length(foundDown) != 0){
                if(mergeOrNot){
                    neighbors[[as.character(i)]] <- unique(c(foundUp,
                                                             foundDown))
                }else{
                    neighbors[[as.character(i)]] <-
                        list(upstream = foundUp, downstream = foundDown)
                }
            }
        }else{
            found <- weightByConfi(c(start[start >= upperB &
                                           start <= location],
                                     end[end <= downB & end >= location]))
            if(length(found) != 0){
                 neighbors[[as.character(i)]] <- unique(found)
            }
        }
    }

    if(length(neighbors) == 0){
        warning("No Genes in the defined region satisfy the condition")
    }else{
        return(neighbors)
    }
}

checkArgs <- function(llID, chromosome, upBase, downBase){
    # llID is not required if search for genes within a range
    if(missing(llID)){
        # Both upBase, downBase, and chromosome must be there if
        # llID is missing
        if(any(missing(upBase), missing(downBase), missing(chromosome))){
            stop(paste("Search can not be conducted with llID and",
                       "at least one of upBase, downBase and ",
                       "chromosome missing"))
        }else{
            if(as.numeric(upBase) < as.numeric(downBase)){
                warning(paste("upBase value is smaller then downBase",
                              "value. Values have been swapped"))
                return("swap")
            }
            if(as.numeric(upBase) == as.numeric(downBase)){
                stop("upBase and downBase can not be the same")
            }
        }
    }else{
        if(missing(upBase)){
            warning(paste("upBase is missing. Search will be",
                          "conducted for genes downstream only"))
        }
        if(missing(downBase)){
            warning(paste("downBase is missing. Search will be",
                          "conducted for genes upstream only"))
        }
    }
    return("OK")
}

findChr4LL <- function(llID, chrEnv, organism){
    options(show.error.message = FALSE)
    chr <- try(get(llID, chrEnv))
    options(show.error.message = TRUE)
    if(inherits(chr, "try-error")){
        stop(paste("LocusLink id", llID, "could not be found in any",
                   "of the chromosomes in the data package"))
    }else{
        if(length(chr) == 1){
            if(!is.element(chr, getValidChr(organism))){
                warning(paste("LocusLink id", llID, "is currently",
                           "not known to be associated with any",
                           "chromosome"))
            }
            return(chr)
        }else{
            chr <- chr[is.element(chr, getValidChr(organism))]
            return(unique(chr))
        }
    }
}


getValidChr <- function(organism){
    switch(toupper(organism),
           HUMAN = return(c(1:22, "X", "Y")),
           MOUSE = return(c(1:19, "X", "Y")),
           RAT = return(c(1:20, "X", "Y")),
           stop(paste("Unknow organism", organism)))
}

getBoundary <- function(loc, base, lower = TRUE){
    if(as.numeric(loc[1]) == 0){
        return(base)
    }else{
        if(lower){
            boundary <- as.numeric(loc[1]) - base
            if(boundary < 0){
                return(0)
            }else{
                return(boundary)
            }
        }else{
            return(as.numeric(loc[1]) + base)
        }
    }
}

weightByConfi <- function(foundLLs){
    if(length(foundLLs) != 0){
        temp <- unique(names(foundLLs))
        foundLLs <- gsub("(^.*)\\..*", "\\1", temp)
        names(foundLLs) <- gsub("^.*\\.(.*)", "\\1", temp)
        # Remove LLs named Unconfident if one named Confident exists
        if(any(duplicated(foundLLs))){
            foundLLs <- c(foundLLs[names(foundLLs) == "Confident"],
                          foundLLs[names(foundLLs) != "Confident"])
            foundLLs <- foundLLs[!duplicated(foundLLs)]
        }

        return(foundLLs)
    }else{
        return("")
    }
}
##copyright 2002 R. Gentleman, all rights reserved
##helper functions for dealing with data environments (soon to be hash
##tables)

getGO <- function(x, data) {
     library(data, character.only=TRUE)
     GOenv <- get(paste(data, "GO",sep=""))
     multiget(x, env=GOenv)
 }

 getGOdesc <- function(x, which=c("MF","BP","CC") ) {
     require("GO") || stop("need the GO library")
     d <- match.arg(which)
     de <- switch(d, MF=GOMFID2TERM, BP=GOBPID2TERM, CC=GOCCID2TERM,
                  stop(paste(which, "did not match a GO data type")))
     ans <- multiget(x, env=de)
     ans[is.na(ans)] <- NULL
     ans
 }

  getSYMBOL <- function(x, data) {
     library(data, character.only=TRUE)
     GOenv <- get(paste(data, "SYMBOL",sep=""))
     unlist(multiget(x, env=GOenv))
 }

  getPMID <- function(x, data) {
      library(data, character.only=TRUE)
      PMenv <- get(paste(data, "PMID", sep=""))
      multiget(x, env=PMenv)
  }

  getLL <- function(x, data) {
      library(data, character.only=TRUE)
      LLenv <- get(paste(data, "LOCUSID", sep=""))
      unlist(multiget(x, env=LLenv))
  }

  if( !isGeneric("contents") && !exists("contents", mode="function") )
       setGeneric("contents", function(object)
                  standardGeneric("contents"))

  setMethod("contents", "environment",
     function(object)
         multiget(ls(env=object), env=object))

installDataPackage <- function(pkga, liba=.libPaths()[1]) {
    require("reposTools")||
              stop("installDataPackage requires package reposTools")

    z <- getReposEntry("BIOCData")
    x <- install.packages2(pkga, z, lib=liba)
    if (length(statusList(x)) == 0)
        stop(paste("Data package",pkga,"does not seem to exist",
                   "in the Bioconductor\ndata package repository."))
}
getPMInfo <- function(x) { 
#
# getMLInfo: get medline-related info from a pubmed xml DOM tree
# works with result of Bioconductor annotate::pubmed function
#
# tagVals: utility function for grabbing vector of
# tag values from any DOM tree
#
tagVals <- function(x,tag) { 
 tagNames <- function() {
  store <- character(0)
  add <- function(x) {
   if(inherits(x, "XMLNode") & xmlName(x) == tag) {
     store <<- c(store, xmlValue(x))
    }
   x
   }
  return(list(add=add, tagVals = function() {return(store)}))
  }
 h <- tagNames()
 xmlDOMApply(x, h$add) 
 h$tagVals()
}
#
# here's the main body of getMLInfo.  the function 'arts' creates
# a closure for collecting data on articles in the document returned
# by the pubmed function.  the 'add' element of the closure
# adds information to various local vectors and lists as xmlDOMApply
# walks through the tree.
#
 if (class(x) != "XMLDocument") stop("only applies to XMLDocument")
 arts <- function() {
  pmarts <- list()
  pmart <- list()
  jinfo <- character(0)
  alist <- character(0)
  chemlist <- character(0)
  cura <- character(0)
  cur <- 1
  add <- function(x) {
   if(inherits(x, "XMLNode") & xmlName(x) == "ArticleTitle") {
     pmart[["title"]] <<- xmlValue(x)
    }
   if(inherits(x, "XMLNode") & xmlName(x) == "MedlineTA") {
     pmart[["MedlineTA"]] <<- xmlValue(x)
    }
   if(inherits(x, "XMLNode") & xmlName(x) == "AbstractText") {
     pmart[["abstract"]] <<- xmlValue(x)
    }
   if(inherits(x, "XMLNode") & xmlName(x) == "PubmedArticle") {
     id <- tagVals(x, "PMID")
     pmarts[[id]] <<- pmart
     pmart <<- list()
     cur  <<- cur+1
    }
#
# deal with journal info
# this is an ugly part because tags like Year or Volume can occur in
# different contexts.  Need to know something about the parent.
# but we don't want to assume too much about sequence of nodes
#
   if (inherits(x, "XMLNode") & xmlName(x) == "ISSN") {
     jinfo <<- c(jinfo,ISSN=xmlValue(x))
   }
   if (inherits(x, "XMLNode") & xmlName(x) == "JournalIssue") {
    jikids <- xmlChildren(x)
    for (i in 1:length(jikids))
     {
     if (xmlName(jikids[[i]]) == "Volume")
       jinfo <<- c(jinfo,vol=xmlValue(jikids[[i]]))
     else if (xmlName(jikids[[i]]) == "Issue")
       jinfo <<- c(jinfo,iss=xmlValue(jikids[[i]]))
     else if (xmlName(jikids[[i]]) == "PubDate")
       {
       Year <- tagVals(jikids[[i]],"Year")
       Month <- tagVals(jikids[[i]],"Month")
       Day <- tagVals(jikids[[i]],"Day")
       jinfo <<- c(jinfo,year=Year,month=Month,day=Day)
       }
     }
     pmart[["JrnlInfo"]] <<- jinfo
     jinfo <<- character(0)
   }
#
# deal with author info
#
   if (inherits(x, "XMLNode") & xmlName(x) =="AuthorList") {
     pmart[["authors"]] <<- alist
     alist <<- character(0)
   }
   if (inherits(x, "XMLNode") & xmlName(x) =="Author") {
     alist <<- c(alist,cura)
     cura <<- character(0)
   }
   if (inherits(x, "XMLNode") & xmlName(x) =="LastName") {
     cura <<- paste(cura,last=xmlValue(x),sep="") 
   }
#   if (inherits(x, "XMLNode") & xmlName(x) =="ForeName") {
#     cura <<- paste(cura,fore=xmlValue(x)) 
#   }
   if (inherits(x, "XMLNode") & xmlName(x) =="Initials") {
     cura <<- paste(cura,inits=xmlValue(x)) 
   }
#
# deal with substance info
#
   if (inherits(x, "XMLNode") & xmlName(x) =="ChemicalList") {
     pmart[["chemlist"]] <<- chemlist
     chemlist <<- character(0)
   }
   if (inherits(x, "XMLNode") & xmlName(x) =="NameOfSubstance") {
     chemlist <<- c(chemlist,xmlValue(x))
   }
   x
   }
  return(list(add=add, arts = function() {return(pmarts)}))
  }
 h <- arts()
 xmlDOMApply(xmlRoot(x), h$add) 
 h$arts()
}
makeAnchor <- function(link, title, toMain=FALSE) {
    ## Takes a vector of links and a vector of titles -
    ## returns a vector of anchors.

    ## !! Should allow links to be URL objects as well as strings
    out <- paste("<A HREF=",link,sep="")
    if (toMain)
        out <- paste(out," target=\"main\"", sep="")
    out <- paste(out,">",title,"</A>",sep="")
    out
}


    ## A simple class to represent a basic "HTML Page", currently
    ## being naively stored as a block of text.
    setClass("HTMLPage", representation(fileName="character",
                                        pageText="character",
                                        pageTitle="character"))
    if (is.null(getGeneric("fileName")))
        setGeneric("fileName", function(object, ...)
                   standardGeneric("fileName"))
    setMethod("fileName", "HTMLPage", function(object, ...)
              object@fileName)

    if (is.null(getGeneric("pageText")))
        setGeneric("pageText", function(object, ...)
                   standardGeneric("pageText"))

    setMethod("pageText", "HTMLPage", function(object, ...)
              object@pageText)

    if (is.null(getGeneric("pageTitle")))
        setGeneric("pageTitle", function(object, ...)
                   standardGeneric("pageTitle"))
    setMethod("pageTitle", "HTMLPage", function(object, ...)
              object@pageTitle)

    setMethod("show","HTMLPage", function(object) print(pageText(object)))

    if (is.null(getGeneric("toFile")))
        setGeneric("toFile", function(object, ...)
                   standardGeneric("toFile"))
    setMethod("toFile", "HTMLPage", function(object, ...) {
        cat(pageText(object), file=fileName(object))
    })

    ## Defines a basic framed page.  We're using 3 frames, a top
    ## banner, a side navigation bar and a main page, much like the
    ## bioconductor website.  The object also has it's own HTML page
    ## associated with it via HTMLPage inheritance.
    setClass("FramedHTMLPage", representation(topPage="HTMLPage",
                                              sidePage="HTMLPage",
                                              mainPage="HTMLPage"),
             contains="HTMLPage")

    if (is.null(getGeneric("topPage")))
        setGeneric("topPage", function(object, ...)
                   standardGeneric("topPage"))
    setMethod("topPage", "FramedHTMLPage", function(object, ...)
              object@topPage)

    if (is.null(getGeneric("sidePage")))
        setGeneric("sidePage", function(object, ...)
                   standardGeneric("sidePage"))
    setMethod("sidePage", "FramedHTMLPage", function(object, ...)
              object@sidePage)

    if (is.null(getGeneric("mainPage")))
        setGeneric("mainPage", function(object, ...)
                   standardGeneric("mainPage"))
    setMethod("mainPage", "FramedHTMLPage", function(object, ...)
              object@mainPage)

    setMethod("toFile", "FramedHTMLPage", function(object, ...) {
        toFile(topPage(object))
        toFile(sidePage(object))
        toFile(mainPage(object))

        ## Is there a way to force a call to HTMLPage's 'toFile' here?
        cat(pageText(object), file=fileName(object))
    })

    setMethod("initialize", "FramedHTMLPage",
              function(.Object, topPage=new("HTMLPage"),
                       sidePage=new("HTMLPage"),
                       mainPage=new("HTMLPage"),
                       fileName=new("character"),
                       pageTitle=new("character")) {
                  .Object@pageTitle <- pageTitle
                  .Object@fileName <- fileName
                  .Object@topPage <- topPage
                  .Object@sidePage <- sidePage
                  .Object@mainPage <- mainPage
                  topName <- fileName(topPage(.Object))
                  sideName <- fileName(sidePage(.Object))
                  mainName <- fileName(mainPage(.Object))

                  out <- paste("<HTML>","<HEAD>",sep="\n")
                  t <- paste("<TITLE>",pageTitle(.Object),"</TITLE>")
                  out <- paste(out,t,"</HEAD>",
                               "<frameset rows=\"70,*\" border =\" 0\" frameborder=\" no\" framespacing =\" 0\">",
                               "  <frame name=\"banner\" scrolling=\"no\" noresize target=\"contents\" src=\"",topName,"\" marginwidth=\"0\" marginheight=\"0\">",
                               "  <frameset cols=\"250,*\">",
                               "    <frame name=\"contents\" target=\"main\" src=\"",sideName,"\" marginwidth=\"10\" marginheight=\"10\" scrolling=\"auto\" noresize>",
                               "    <frame name=\"main\" scrolling=\"auto\" noresize src=\"",mainName,"\" marginwidth =\" 10\" marginheight =\" 10\" target=\"_self\">",
                               "  </frameset>","  <noframes>","  <body>","",
                               "  <p>This page uses frames, but your browser doesn't support them.</p>",
                               "", "  </body>","  </noframes>",
                               "</frameset>","</html>",
                               sep="\n")
                  .Object@pageText <- out
                  .Object
              })
    ## Define the class structure of the pubMedAbst object
    setGeneric("pubMedAbst", function(object)
               standardGeneric("pubMedAbst"))

    setClass("pubMedAbst",
             representation(pmid="character", authors="vector", abstText="character",
             articleTitle="character", journal="character",
             pubDate="character", abstUrl="character"))

    setMethod("show", "pubMedAbst", function(object) {
         cat("An object of class pubMedAbs \n")
         slots <- slotNames(object)
         for (what in slots) {
            if (identical(what, ".Data"))
                next
           cat("Slot \"", what, "\":\n", sep = "")
           if( what == "articleTitle" || what == "abstText")
	       cat(paste("\t", substr(slot(object, what), 1, 70),
                   "...\n", sep=""))
	   else
               print(slot(object, what))
            cat("\n")
        }})

    ## Define generics
    if (is.null(getGeneric("authors")))
        setGeneric("authors", function(object)
                   standardGeneric("authors"))

    if (is.null(getGeneric("abstText")))
        setGeneric("abstText", function(object)
                   standardGeneric("abstText"))

    if (is.null(getGeneric("articleTitle")))
        setGeneric("articleTitle", function(object)
                   standardGeneric("articleTitle"))

    if (is.null(getGeneric("journal")))
        setGeneric("journal", function(object)
                   standardGeneric("journal"))

    if (is.null(getGeneric("pubDate")))
        setGeneric("pubDate", function(object)
                   standardGeneric("pubDate"))

    if (is.null(getGeneric("abstUrl")))
        setGeneric("abstUrl",function(object)
                   standardGeneric("abstUrl"))

    if (is.null(getGeneric("pmid")))
        setGeneric("pmid", function(object)
                   standardGeneric("pmid"))

## Methods
    setMethod("authors", "pubMedAbst", function(object)
              object@authors)
    setMethod("abstText", "pubMedAbst", function(object)
              object@abstText)
    setMethod("articleTitle", "pubMedAbst", function(object)
              object@articleTitle)
    setMethod("journal", "pubMedAbst", function(object)
              object@journal)
    setMethod("pubDate", "pubMedAbst", function(object)
              object@pubDate)
    setMethod("abstUrl", "pubMedAbst", function(object)
              object@abstUrl)
    setMethod("pmid", "pubMedAbst", function(object)
              object@pmid)

buildPubMedAbst <- function(xml) {
    ## Passed in a XML tree detailing a single article
    ## will parse the XML and create a new class

    xmlMedline <- xml["MedlineCitation"][[1]]
    xmlArticle <- xmlMedline["Article"]

    ## Disable error messages, and wrap potential error causers
    ## w/ trys
    options(show.error.messages = FALSE)
    on.exit(options(show.error.messages=TRUE))

    ## Get the PMID
    pmid <- xmlMedline["PMID"][[1]]
    pmid <- try(as.character(xmlChildren(pmid)$text)[6])
    if (inherits(pmid,"try-error") == TRUE) {
        pmid <- "No PMID Provided"
    }

    ## Retrieve Article Title
    articleTitle <- xmlArticle[[1]][["Article"]]
    articleTitle <-
    try(as.character(xmlChildren(articleTitle)$text)[6])
    if (inherits(articleTitle,"try-error") == TRUE) {
        articleTitle <- "No Title Provided"
    }

    ## Retrieve the abstract
    abstText <- xmlArticle[[1]]["Abstract"][[1]]["AbstractText"]
    abstText <- try(as.character(xmlChildren(abstText[[1]])$text)[6])
   if (inherits(abstText,"try-error") == TRUE) {
       abstText <- "No Abstract Provided"
   }

    ## Retrieve the date - get the year/month separately and then
    ## join them at the end.  If no month or year provided, subst
    ## "MontH" and "Year" respectively
    pubDateBase <-
        xmlArticle[[1]]["Journal"][[1]]["JournalIssue"][[1]]["PubDate"]
    pubDateMonth <- pubDateBase[[1]]["Month"]
    pubDateMonth <-
        try(as.character(xmlChildren(pubDateMonth[[1]])$text)[6])
    if (inherits(pubDateMonth,"try-error") == TRUE) {
        pubDateMonth <- "Month"
    }
    pubDateYear <- pubDateBase[[1]]["Year"]
    pubDateYear <-
        try(as.character(xmlChildren(pubDateYear[[1]])$text)[6])
    if (inherits(pubDateYear, "try-error") == TRUE) {
        pubDateYear <- "Year"
    }
    ## Join up the date information
    pubDate <- paste(pubDateMonth,pubDateYear)

    ## Get the journal this was published in
    journal <-
        xml["MedlineCitation"][[1]]["MedlineJournalInfo"][[1]]["MedlineTA"]
    journal <- try(as.character(xmlChildren(journal[[1]])$text)[6])
    if (inherits(journal,"try-error") == TRUE) {
        journal <- "No Journal Provided"
    }

    ## Build up a vector of author names, created by assembling the
    ## pieces of each author's name.
    authorList <- xmlArticle[[1]]["AuthorList"]
    authors <- vector()
    numAuthors <- try(length(xmlChildren(authorList[[1]])))
    if (inherits(numAuthors,"try-error") == TRUE) {
        authors[1] <- "No Author Information Provided"
    }
    else {
        for (i in 1:numAuthors) {
            curAuthor <- authorList[[1]][i]
            last <-
                try(as.character(xmlChildren(curAuthor[[1]]["LastName"][[1]])$text)[6])
            if (inherits(last,"try-error") == TRUE) {
                last <- "LastName"
            }

            initial <-
                try(as.character(xmlChildren(curAuthor[[1]]["Initials"][[1]])$text)[6])
            if (inherits(initial,"try-error") == TRUE) {
                initial <- "M"
            }

            authors[i] <- paste(initial,last)
        }
    }

    abstUrl <-
        try(as.character(xmlChildren(xml["PubmedData"][[1]]["URL"][[1]])$text)[6])
    if (inherits(abstUrl,"try-error") == TRUE) {
        abstUrl <- "No URL Provided"
    }

    ## Restore error messages
    options(show.error.messages=TRUE)

    newPMA <- new("pubMedAbst", articleTitle=articleTitle,
                  abstText=abstText, pubDate=pubDate,authors=authors,
                  journal=journal,abstUrl=abstUrl, pmid=pmid)

    return(newPMA)
}

pm.getabst <- function(geneids, basename) {
    pmids <- getPMID(geneids, basename)
    numids <- length(geneids)
    rval <- vector("list", length=numids)
    names(rval) <- geneids
    for(i in 1:numids) {
        pm <- pmids[[i]]
        if( length(pm)==1 && is.na(pm) )
            rval[[i]] <- NA
        else {
            absts <- pubmed(pm)
            a <- xmlRoot(absts)
            numAbst <- length(xmlChildren(a))
            absts <- vector("list", length=numAbst)
            for (j in 1:numAbst)
                absts[[j]] <- buildPubMedAbst(a[[j]])
            rval[[i]] <- absts
        }
    }
    rval
}

pm.abstGrep <- function(pattern, absts, ...)
{
    nabsts <- length(absts)
    rval <- rep(FALSE, nabsts)
    for(i in 1:nabsts) {
        atxt <- abstText(absts[[i]])
        ans <- grep(pattern, atxt, ...)
        if( length(ans) && ans==1 )
            rval[i] <- TRUE
    }
    rval
}

pm.titles <- function (absts) {
     numa <- length(absts)
     rval <- vector("list", length=numa)
     for(j in 1:numa)
         rval[[j]] <- sapply(absts[[j]], function(x) articleTitle(x))
     rval
}

#Copyright 2001 R.Gentleman, all rights reserved
#functions to look up particular genes at different sites

UniGeneQuery <- function(query, UGaddress="UniGene/",
                         type="CID") {
    if (missing(query))
        stop("No query, cannot proceed!")

    ##they are of the form HH.xxxx, where HH specifies the species
    q1 <- strsplit(query, "\\.")
    if( length(q1[[1]]) == 2 ) {
        id <- sapply(q1, function(x) x[2])
        species <- sapply(q1, function(x) x[1])
    }

    ncbiURL <- .getNcbiURL()
    ## Build up the query URL

    query <- paste(ncbiURL, UGaddress,
    "clust.cgi?ORG=",species,"&", type, "=",id, sep="")

    return(query)
}

pmidQuery <- function(query) {
    if (missing(query))
        stop("No query, cannot proceed!")

    query <- paste(query,collapse="%2c")
    ncbiURL <- .getNcbiURL()

    query <- paste(ncbiURL,"/entrez/query.fcgi?cmd=Retrieve&db=PubMed&",
                 "list_uids=",query,"&dopt=Abstract&tool=bioconductor",sep="")

    return(query)
}

locuslinkQuery <- function(query,...,lladdress="LocusLink/", browse=TRUE) {
    params <- list(...)
    params <- unlist(params,use.names=FALSE)

    if (is.na(query))
        stop("No query, cannot proceed!")

    if (length(c(params)) == 0) {
        species = "Hs"
    }
    else {
        species <- paste(params,collapse="&ORG=")
    }


    ncbiURL <- .getNcbiURL()

    ## Build up the query URL

    query <- paste(ncbiURL, lladdress,
    "list.cgi?Q=",query,"&ORG=",species,"&V=0",sep="")

    if (browse)
        browseURL(query)
    else
        return(query)
}

locuslinkByID <- function(..., lladdress="LocusLink/", browse=TRUE) {
    params <- list(...)
    params <- unlist(params)

    if (length(params) == 0)
        stop("No Locuslink ID, cannot proceed")

    ncbiURL <- .getNcbiURL()

    ## Build up the query URL
    if (length(params) == 1) {
        args <- paste(params,collapse="%2c")
        query <- paste(ncbiURL, lladdress, "LocRpt.cgi?l=", args, sep="")
    }
    else {
        args <- paste(params,collapse="&ID=")
        query <- paste(ncbiURL, lladdress, "list.cgi?ID=", args, sep="")
    }

    if (browse)
        browseURL(query)
    else
        return(query)
}

genbank <- function(..., disp=c("data","browser"),
                    type=c("accession", "uid"),
                    pmaddress=.pmfetch("Nucleotide",disp,type)) {
    params <- list(...)
    params <- unlist(params)

    disp <- match.arg(disp)
    type <- match.arg(type)

    if (length(params) == 0) {
        stop("No Gene ID, cannot proceed")
    }

    ncbiURL <- .getNcbiURL()

    ## Build up the query URL
    args <- paste(params,collapse="%2c")
    ## See if we need to transform accession based arguments
    err <- args
    args <- .transformAccession(args, disp, type,db="genbank")

   if (is.null(args)) {
        print(paste("No XML records available for accession number",err))
        return(NULL)
    }

    id <- .getIdTag(disp,type)

    query <- paste(ncbiURL, pmaddress, id, args, sep="")

    ## Determine if we are displaying this data in a browser or
    ## returning an XMLDocument object
    if (disp == "data") {
        return(.handleXML(query))
    }
    else {
        browseURL(query)
    }
}

pubmed  <- function(..., disp=c("data","browser"),
                    type=c("uid","accession"),
                    pmaddress=.pmfetch("PubMed",disp,type)) {
    params <- list(...)
    params <- unlist(params)

    disp <- match.arg(disp)
    type <- match.arg(type)

    if (length(params) == 0) {
        stop("No PMID, cannot proceed")
    }

    ncbiURL <- .getNcbiURL()

    ## Build up the query URL
    args <- paste(params,collapse="%2c")
    ## See if we need to transform accession based arguments
    err <- args
    args <- .transformAccession(args, disp, type,"pubmed")

    if (is.null(args)) {
        print(paste("No XML records available for accession number",err))
        return(NULL)
    }

    id <- .getIdTag(disp,type)

    query <- paste(ncbiURL, pmaddress, id, args, sep="")


    ## Determine if we are displaying this data in a browser or
    ## returning an XMLDocument object
    if (disp == "data") {
        return(.handleXML(query))
    }
    else {
        browseURL(query)
    }
}

accessionToUID <- function(...,db=c("genbank","pubmed")) {
    ## Passed an accession #, returns a pubmed UID

    accNum <- list(...)
    accNum <- unlist(accNum)
    accNum <- paste(accNum,collapse="+OR+")

    db <- match.arg(db)

    ## Certain functions will be passing in a single string of comma
    ## deliminated Accession #s.  Change the commas to "+OR+"
    accNum <- gsub("\\,","+OR+",accNum)

    if (db == "genbank") {
        db <- "nucleotide"
    }
    else {
        db <- "PubMed"
    }

    query <- paste(.getNcbiURL(), "entrez/utils/pmqty.fcgi?db=", db,
                   "&tool=bioconductor&mode=xml&term=",accNum,sep="")

    ## Currently doubling up on code from .handleXML as I can't yet find a
    ## way to retrieve values back through the extra layer of
    ## indirection.

    require(XML) || stop("XML package is unavailable!")
    ## Make sure that XML version is what we require
    ## !!! Need to make this automatic, hardcode version in for now
    xmlVers <- package.description("XML",fields="Version")
    reqXmlVers <- "0.92-2"
    if (compareVersion(xmlVers,reqXmlVers) < 0)
        stop(paste("Installed XML version is ",xmlVers,
                   " while this functionality requires ", reqXmlVers,
                   ":  Please update your XML package.",sep=""))

    options(show.error.messages = FALSE)
    on.exit(options(show.error.messages = TRUE))
    retVal <- NULL
    result <- try(xmlTreeParse(query,isURL=TRUE, handlers=list(Id=function(x,attrs) {retVal <<- xmlValue(x[[1]])})))
    options(show.error.messages = TRUE)

    if (!is.null(retVal)) {
        ## In the event of multiple IDs, it returns as a monolithic
        ## which is space delimited.  Change this to comma deliminated
        retVal <- gsub(" *", "\\,", retVal)
    }

    return(retVal)
}


.handleXML <- function(query,handlers=NULL) {
    ## In the case of an error retrieving proper XML output,
    ## will return NA to the calling function
    require(XML) || stop("Sorry, you need the XML package!")
    ## Make sure that XML version is what we require
    ## !!! Need to make this automatic, hardcode version in for now
    xmlVers <- package.description("XML",fields="Version")
    reqXmlVers <- "0.92-2"
    if (compareVersion(xmlVers,reqXmlVers) < 0)
        stop(paste("Installed XML version is ",xmlVers,
                   " while this functionality requires ", reqXmlVers,
                   ":  Please update your XML package.",sep=""))

    options(show.error.messages = FALSE)
    on.exit(options(show.error.messages = TRUE))
    retVal <- NULL
    xml <- try(xmlTreeParse(query,isURL=TRUE,handlers=NULL,asTree=TRUE))
    options(show.error.messages = TRUE)

    if (inherits(xml,"try-error") == TRUE) {
        return(NA)
    }

    return(xml)
}

.getNcbiURL <- function() {
    ## Returns the URL for NCBI, which should be located in Annotate's
    ## option set
    BioCOpt <- getOption("BioC")

    if (!is.null(BioCOpt)) {
        ncbiURL <- BioCOpt$annotate$urls$ncbi
    }

    if (!exists("ncbiURL")) {
        ncbiURL <- "http://www.ncbi.nih.gov/"
    }

    return(ncbiURL)
}

.getIdTag <- function(disp=c("data","browser"),
                      type=c("uid","accession")) {
    disp <- match.arg(disp)
    type <- match.arg(type)

    if (disp == "data") {
        return("&id=")
    }
    else {
        if (type == "uid") {
            return("&list_uids=")
        }
        else {
            return("&term=")
        }
    }
}

.pmfetch <- function(db="PubMed", disp=c("data","browser"),
                     type=c("uid","accession")) {
    ## Returns the base query string for the pmfetch engine @ pubmed

    disp <- match.arg(disp)
    type <- match.arg(type)

    if (disp == "data") {
        base <-
    "entrez/utils/pmfetch.fcgi?report=xml&mode=text&tool=bioconductor&db="
    }
    else {
        base1 <- "entrez/query.fcgi?tool=bioconductor&cmd="
        if (type == "uid") {
            base2 <- "Retrieve&db="
        }
        else {
            base2 <- "Search&db="
        }
        base <- paste(base1,base2,sep="")
    }
    return(paste(base,db,sep=""))
}

.transformAccession <- function(args, disp, type, db) {
    ## Used to change accession ID arguments to query functions
    ## into UIDs if necessary.  Returns NULL if there aren't any left.
    if ((disp == "data")&&(type=="accession")) {
        args <- accessionToUID(args,db=db)
    }

    return(args)
}

genelocator <- function(x) {
  done<-FALSE
  while(!done) {
    v <- identify(x, n=1)
    if (length(v)==0)
      done <- TRUE
    else
      print(paste("hi I'm number", v))
  }
}

pmAbst2html2 <- function(absts, baseFilename,
                         title,
                         table.center=TRUE) {
    if (!is.list(absts)) {
        if (is(absts,"pubMedAbst"))
            absts <- list(absts)
        else
            stop("'absts' parameter does not seem to be valid.")
    }

    if (missing(baseFilename))
        baseFilename <- ""
    if (missing(title))
        title <- "BioConductor Abstract List"

    topText <- paste("<html>\n<head>\n<title>Bioconductor Abstract List</title>",
                     "\n</head>\n<body bgcolor=#708090>\n",
                     "<H1 ALIGN=CENTER>BioConductor Abstract List</H1>\n",
                     "</body></title>", sep="")
    top <- new("HTMLPage", fileName=paste(baseFilename,"Top.html",sep=""),
               pageText= topText)

    head <- c("Article Title", "Publication Date")
    headOut <- paste("<TH>", head, "</TH>", collapse="\n")
    tableHeader <- paste("<TR>",headOut,"</TR>", sep="\n")
    sideText <- paste("<TABLE BORDER=1>", tableHeader, sep="\n")

    nrows = length(absts)
    pmids <- unlist(lapply(absts,pmid))
    dates <- unlist(lapply(absts,pubDate))
    queries <- unlist(lapply(absts,
                             function(x){pm <- pmid(x);out<-pmidQuery(pm);out}))
    titles <- unlist(lapply(absts, articleTitle))
    anchors <- makeAnchor(queries, titles, toMain=TRUE)
    tds <- paste("<TD>",anchors,"</TD><TD>",dates,"</TD>",sep="",
                 collapse="\n</TR>\n<TR>\n")
    tds <- paste("<TR>",tds,"</TR>")
    sideText <- paste(sideText, tds)
    if (table.center)
        sideText <- paste("<CENTER>",sideText,"</CENTER>", sep="\n")
    sideText <- paste("<html>", "<head>",
                      "<title>BioConductor Abstract List</title>",
                      "</head>","<body bgcolor=#708090>",
                      sideText, "</body>", "</html>", sep="\n")
    side <- new("HTMLPage",
                fileName=paste(baseFilename,"Side.html",sep=""),
                pageText=sideText)

    metaText <- paste("<meta HTTP-EQUIV=\"REFRESH\" CONTENT=\"1;",
                      queries[1],"\">",sep="")
    mainText <- paste("<html>", "<head>",
                      "<title>BioConductor Abstract List</title>",
                      "</head>","<body bgcolor=#708090>",
                      metaText,
                      "</body>","</html>", sep="\n")

    main <- new("HTMLPage",
                fileName=paste(baseFilename,"Main.html",sep=""),
                pageText=mainText)

    page <- new("FramedHTMLPage", topPage=top, sidePage=side, mainPage=main,
                fileName=paste(baseFilename,"index.html",sep=""),
                pageTitle=title)
    toFile(page)
}

pmAbst2HTML <- function(absts, filename, title, simple=TRUE,
                      table.center=TRUE) {
    ## Currently just a very naive implementation of a pmid2html type
    ## of thing.  Intended to be temporary just while I'm testing some
    ## of this stuff.

    if (!is.list(absts)) {
        if (is(absts,"pubMedAbst"))
            absts <- list(absts)
        else
            stop("'absts' parameter does not seem to be valid.")
    }

    if (missing(filename))
        filename <- "absts.html"

    outfile <- file(filename,"w")
    cat("<html>", "<head>", "<TITLE>BioConductor Abstract List</TITLE>",
        "</head>", "<body bgcolor=#708090 >",
        "<H1 ALIGN=CENTER > BioConductor Abstract List </H1>",
        file = outfile, sep = "\n")
    if ( !missing(title) )
        cat("<CENTER><H1 ALIGN=\"CENTER\">", title, " </H1></CENTER>\n",
            file=outfile, sep = "\n")
    if( table.center )
        cat("<CENTER> \n", file=outfile)

    cat("<TABLE BORDER=1>", file = outfile, sep = "\n")
    head <- c("Article Title", "Publication Date")
    headOut <- paste("<TH>", head, "</TH>")
    cat("<TR>",headOut,"</TR>", file=outfile, sep="\n")

    nrows = length(absts)
    pmids <- unlist(lapply(absts,pmid))
    dates <- unlist(lapply(absts,pubDate))
    queries <- unlist(lapply(absts,
                             function(x){pm <- pmid(x);out<-pmidQuery(pm);out}))
    titles <- unlist(lapply(absts, articleTitle))
    anchors <- makeAnchor(queries, titles)
    tds <- paste("<TD>",anchors,"</TD><TD>",dates,"</TD>",sep="")
    for (td in tds)
        cat("<TR>", td, "</TR>", file=outfile,sep="\n")

    cat("</TABLE>",file=outfile)
    if( table.center )
        cat("</CENTER> \n", file=outfile)
    cat("</body>", "</html>", sep = "\n", file = outfile)
    close(outfile)
}

ll.htmlpage <- function (genelist, filename, title, othernames,
                         table.head, table.center=TRUE,
                         repository = "ll")
{
    outfile <- file(filename, "w")
    cat("<html>", "<head>", "<TITLE>BioConductor Linkage List</TITLE>",
        "</head>", "<body bgcolor=#708090 >",
        "<H1 ALIGN=CENTER > BioConductor Linkage List </H1>",
        file = outfile, sep = "\n")
    if( !missing(title) )
        cat("<CENTER><H1 ALIGN=\"CENTER\">", title, " </H1></CENTER>\n",
            file=outfile, sep = "\n")

    if( table.center )
        cat("<CENTER> \n", file=outfile)

    cat("<TABLE BORDER=4>", file = outfile, sep = "\n")
    if( !missing(table.head) ) {
        headout <- paste("<TH>", table.head, "</TH>")
        cat("<TR>", headout, "</TR>", file=outfile, sep="\n")
    }
#    rh <- "<TD> <A HREF=\"http://www.ncbi.nlm.nih.gov/LocusLink/LocRpt.cgi?l="
    nrows <- length(genelist)
    rows <- getTDRows(genelist, repository)
#    rows <- paste(rh, genelist, "\">", genelist, "</A> </TD>",
#        sep = "")
    if( !missing(othernames) ) {
        if( is.list(othernames) ) {
            others <- ""
            for(nm in othernames)
                others <- paste(others,"<TD>", nm, "</TD>", sep="")
        }
        else
            others <- paste("<TD>", othernames, "</TD>", sep="")
        rows <- paste(rows, others)
    }
    for (i in 1:nrows)
        cat("<TR>", rows[i], "</TR>", file = outfile, sep = "\n")
    cat("</TABLE>",file=outfile)
    if( table.center )
        cat("</CENTER> \n", file=outfile)
    cat("</body>", "</html>", sep = "\n", file = outfile)

    close(outfile)
}

getTDRows <- function(ids, repository = "ug"){
    paste("<TD> <A HREF=\"", getQueryLink(ids, repository),
          "\">", ids, "</A> </TD>", sep = "")
}

getQueryLink <- function(ids, repository = "ug"){
    switch(tolower(repository),
           "ug" = return(getQuery4UG(ids)),
           "ll" = return(getQuery4LL(ids)),
           stop("Unknown repository name"))
}

getQuery4UG <- function(ids){
    # UG ids = XX.yyyy. Split by "."
    ugs <- strsplit(ids, "\\.")
    badUG <- function(x) if(length(x) != 2 || nchar(x[1]) != 2)
        return(TRUE) else return(FALSE)
    bIDs <- sapply(ugs, badUG)
    if( any(bIDs) )
        stop(paste("id(s):", paste(ids[bIDs], collapse=", "),
                   "are not correct"))
    temp <- matrix(unlist(ugs, use.names = FALSE),
                   ncol = 2, byrow = TRUE)
    paste("http://www.ncbi.nlm.nih.gov/UniGene/clust.cgi?ORG=",
              temp[,1], "&CID=", temp[,2], sep = "")
}

getQuery4LL <- function(ids){
    paste("http://www.ncbi.nlm.nih.gov/LocusLink/LocRpt.cgi?l=",
          ids, sep = "")
}

.buildAnnotateOpts <- function() {
    if (is.null(getOption("BioC"))) {
        BioC <- list()
        class(BioC) <- "BioCOptions"
        options("BioC"=BioC)
    }

    Annotate <- list()
    class(Annotate) <- "BioCPkg"
    Annotate$urls <- list( ncbi = "http://www.ncbi.nih.gov/",
          data="http://www.bioconductor.org/datafiles/annotate/")

    BioC <- getOption("BioC")
    BioC$annotate <- Annotate
    options("BioC"=BioC)
}

.First.lib <- function(libname, pkgname) {
    require(Biobase) || stop("cannot load annotate without Biobase")
    .buildAnnotateOpts()
}
