/****************************************************************************
**
** Copyright (C) 2012 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the test suite of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** GNU Lesser General Public License Usage
** This file may be used under the terms of the GNU Lesser General Public
** License version 2.1 as published by the Free Software Foundation and
** appearing in the file LICENSE.LGPL included in the packaging of this
** file. Please review the following information to ensure the GNU Lesser
** General Public License version 2.1 requirements will be met:
** http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU General
** Public License version 3.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of this
** file. Please review the following information to ensure the GNU General
** Public License version 3.0 requirements will be met:
** http://www.gnu.org/copyleft/gpl.html.
**
** Other Usage
** Alternatively, this file may be used in accordance with the terms and
** conditions contained in a signed written agreement between you and Nokia.
**
**
**
**
**
** $QT_END_LICENSE$
**
****************************************************************************/


#include <QtTest/QtTest>

#ifdef QTEST_XMLPATTERNS

#include <QtCore/QDir>
#include <QtXmlPatterns/QXmlQuery>
#include <QtXmlPatterns/QXmlSerializer>
#include <QtXmlPatterns/QXmlResultItems>
#include <QtXmlPatterns/QXmlFormatter>

#include "../qxmlquery/MessageSilencer.h"
#include "../qsimplexmlnodemodel/TestSimpleNodeModel.h"

/*!
 \class tst_PatternistExamples
 \internal
 \since 4.4
 \brief Verifies examples for Patternist.
 */
class tst_PatternistExamples : public QObject
{
    Q_OBJECT

private Q_SLOTS:
    void initTestCase();
    void checkQueries() const;
    void checkQueries_data() const;
    void checkXMLFiles() const;
    void checkXMLFiles_data() const;
    void buildSnippets() const;

private:
    QVector<QDir> m_dirs;
    QStringList listFiles(const QStringList &patterns) const;
    enum Constants
    {
        XMLFileCount = 12,
        XQueryFileCount = 52
    };
};

void tst_PatternistExamples::initTestCase()
{
#ifndef Q_OS_WINCE
    m_dirs.append(QDir(QLatin1String("../../../doc/src/snippets/patternist/")));
    m_dirs.append(QDir(QLatin1String("../../../examples/xmlpatterns/xquery/globalVariables/")));
    m_dirs.append(QDir(QLatin1String("../../../examples/xmlpatterns/filetree/")));
    m_dirs.append(QDir(QLatin1String("../../../examples/xmlpatterns/recipes/")));
    m_dirs.append(QDir(QLatin1String("../../../examples/xmlpatterns/recipes/files/")));
#else
    m_dirs.append(QDir(QLatin1String("patternist/")));
    m_dirs.append(QDir(QLatin1String("globalVariables/")));
    m_dirs.append(QDir(QLatin1String("filetree/")));
    m_dirs.append(QDir(QLatin1String("recipes/")));
    m_dirs.append(QDir(QLatin1String("recipes/files/")));
#endif
    for(int i = 0; i < m_dirs.size(); ++i)
        QVERIFY(m_dirs.at(i).exists());
}

/*!
  Returns a QStringList containing absolute filenames that were found in the predefined locations, when
  filtered through \a pattterns.
 */
QStringList tst_PatternistExamples::listFiles(const QStringList &patterns) const
{
    QStringList result;

    for(int i = 0; i < m_dirs.size(); ++i)
    {
        const QDir &dir = m_dirs.at(i);

        const QStringList files(dir.entryList(patterns));
        for(int s = 0; s < files.count(); ++s)
            result += dir.absoluteFilePath(files.at(s));
    }

    return result;
}

/*!
  Check that the queries contains no static errors such as
  syntax errors.
 */
void tst_PatternistExamples::checkQueries() const
{
    QFETCH(QString, queryFile);

    QFile file(queryFile);
    QVERIFY(file.open(QIODevice::ReadOnly));

    QXmlQuery query;

    /* Two queries relies on this binding, so provide it such that we don't get a compile error. */
    query.bindVariable(QLatin1String("fileToOpen"), QVariant(QString::fromLatin1("dummyString")));

    /* This is needed for the recipes example. */
    query.bindVariable(QLatin1String("inputDocument"), QVariant(QString::fromLatin1("dummString")));

    /* This is needed for literalsAndOperators.xq. */
    query.bindVariable(QLatin1String("date"), QVariant(QDate::currentDate()));

    /* These are needed for introExample2.xq. */
    query.bindVariable(QLatin1String("file"), QVariant(QLatin1String("dummy")));
    query.bindVariable(QLatin1String("publisher"), QVariant(QLatin1String("dummy")));
    query.bindVariable(QLatin1String("year"), QVariant(2000));

    /* and filetree/ needs this. */
    TestSimpleNodeModel nodeModel(query.namePool());
    query.bindVariable(QLatin1String("exampleDirectory"), nodeModel.root());

    query.setQuery(&file, queryFile);

    QVERIFY2(query.isValid(), QString::fromLatin1("%1 failed to compile").arg(queryFile).toLatin1().constData());
}

void tst_PatternistExamples::checkQueries_data() const
{
    QTest::addColumn<QString>("queryFile");

    const QStringList queryExamples(listFiles(QStringList(QLatin1String("*.xq"))));

    QCOMPARE(queryExamples.count(), int(XQueryFileCount));

    foreach(QString q, queryExamples)
        QTest::newRow(q.toLocal8Bit().constData()) << q;
}

void tst_PatternistExamples::checkXMLFiles() const
{
    QFETCH(QString, file);

    QXmlQuery query;
    /* Wrapping in QUrl ensures it gets formatted as a URI on all platforms. */
    query.setQuery(QLatin1String("doc('") + QUrl::fromLocalFile(file).toString() + QLatin1String("')"));
    QVERIFY(query.isValid());

    /* We don't care about the result, we only want to ensure the files can be parsed. */
    QByteArray dummy;
    QBuffer buffer(&dummy);
    QVERIFY(buffer.open(QIODevice::WriteOnly));

    QXmlSerializer serializer(query, &buffer);

    /* This is the important one. */
    QVERIFY(query.evaluateTo(&serializer));
}

void tst_PatternistExamples::checkXMLFiles_data() const
{
    QTest::addColumn<QString>("file");
    QStringList patterns;
    patterns.append(QLatin1String("*.xml"));
    patterns.append(QLatin1String("*.gccxml"));
    patterns.append(QLatin1String("*.svg"));
    patterns.append(QLatin1String("*.ui"));
    patterns.append(QLatin1String("*.html"));

    const QStringList xmlFiles(listFiles(patterns));

    if(xmlFiles.count() != XMLFileCount)
        qDebug() << "These files were encountered:" << xmlFiles;

    QCOMPARE(xmlFiles.count(), int(XMLFileCount));

    foreach(QString q, xmlFiles)
        QTest::newRow(q.toLocal8Bit().constData()) << q;
}

/*!
 Below, we include all the examples and ensure that they build, such that we rule
 out syntax error and that API changes has propagated into examples.

 An improvement could be to run them, to ensure that they behave as they intend
 to.
 */

static QUrl abstractURI()
{
    QUrl baseURI;
    QUrl relative;
#include "../../doc/src/snippets/code/src_xmlpatterns_api_qabstracturiresolver.cpp"
}

class MyValue
{
public:
    MyValue parent() const
    {
        return MyValue();
    }
};

static MyValue toMyValue(const QXmlNodeModelIndex &)
{
    return MyValue();
}

static QXmlNodeModelIndex toNodeIndex(const MyValue &)
{
    return QXmlNodeModelIndex();
}

class MyTreeModel : public QSimpleXmlNodeModel
{
public:
    MyTreeModel(const QXmlNamePool &np, const QFile &f);

    virtual QUrl documentUri(const QXmlNodeModelIndex&) const
    {
        return QUrl();
    }

    virtual QXmlNodeModelIndex::NodeKind kind(const QXmlNodeModelIndex&) const
    {
        return QXmlNodeModelIndex::Element;
    }

    virtual QXmlNodeModelIndex::DocumentOrder compareOrder(const QXmlNodeModelIndex&, const QXmlNodeModelIndex&) const
    {
        return QXmlNodeModelIndex::Is;
    }

    virtual QXmlNodeModelIndex root(const QXmlNodeModelIndex&) const
    {
        return QXmlNodeModelIndex();
    }

    virtual QXmlName name(const QXmlNodeModelIndex&) const
    {
        return QXmlName();
    }

    virtual QVariant typedValue(const QXmlNodeModelIndex&) const
    {
        return QVariant();
    }

    virtual QVector<QXmlNodeModelIndex> attributes(const QXmlNodeModelIndex&) const
    {
        return QVector<QXmlNodeModelIndex>();
    }

    QXmlNodeModelIndex nodeFor(const QString &) const
    {
        return QXmlNodeModelIndex();
    }

    virtual QXmlNodeModelIndex nextFromSimpleAxis(SimpleAxis axis, const QXmlNodeModelIndex &origin) const;
};

/*
 Exists for linking with at least msvc-2005.
*/
MyTreeModel::MyTreeModel(const QXmlNamePool &np, const QFile &) : QSimpleXmlNodeModel(np)
{
}

#include "../../doc/src/snippets/code/src_xmlpatterns_api_qsimplexmlnodemodel.cpp"

class MyMapper
{
public:
    class InputType;
    enum OutputType
    {
    };
#include "../../doc/src/snippets/code/src_xmlpatterns_api_qabstractxmlforwarditerator.cpp"
};

#include "../../doc/src/snippets/code/src_xmlpatterns_api_qxmlname.cpp"

void tst_PatternistExamples::buildSnippets() const
{
    /* We don't run this code, see comment above. */
    return;

    /* We place a call to this function, such that GCC doesn't emit a warning. */
    abstractURI();

    {
    }

    {
#include "../../doc/src/snippets/code/src_xmlpatterns_api_qxmlresultitems.cpp"
    }

    {
    }

    {
        QIODevice *myOutputDevice = 0;
#include "../../doc/src/snippets/code/src_xmlpatterns_api_qxmlformatter.cpp"
    }

    {
        QIODevice *myOutputDevice = 0;
#include "../../doc/src/snippets/code/src_xmlpatterns_api_qxmlserializer.cpp"
    }

    {
        QXmlNodeModelIndex myInstance;
        const char **argv = 0;
        typedef MyTreeModel ChemistryNodeModel;
#include "../../doc/src/snippets/code/src_xmlpatterns_api_qabstractxmlnodemodel.cpp"
    }

    {
    }

    {
        QIODevice *myOutputDevice = 0;
#include "../../doc/src/snippets/code/src_xmlpatterns_api_qabstractxmlreceiver.cpp"
    }

    {
        QXmlQuery query;
        QString localName;
        QVariant value;
#include "../../doc/src/snippets/code/src_xmlpatterns_api_qxmlquery.cpp"
    }
}

QTEST_MAIN(tst_PatternistExamples)

#include "tst_patternistexamples.moc"
#else
QTEST_NOOP_MAIN
#endif

// vim: et:ts=4:sw=4:sts=4
