/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.broker.pop;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.apache.rocketmq.broker.BrokerController;
import org.apache.rocketmq.broker.offset.ConsumerOffsetManager;
import org.apache.rocketmq.broker.pop.PopConsumerKVStore;
import org.apache.rocketmq.broker.pop.PopConsumerLockService;
import org.apache.rocketmq.broker.pop.PopConsumerRecord;
import org.apache.rocketmq.common.BrokerConfig;
import org.apache.rocketmq.common.ServiceThread;
import org.apache.rocketmq.common.utils.ConcurrentHashMapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PopConsumerCache
extends ServiceThread {
    private static final Logger log = LoggerFactory.getLogger((String)"RocketmqPop");
    private static final long OFFSET_NOT_EXIST = -1L;
    private final BrokerController brokerController;
    private final PopConsumerKVStore consumerRecordStore;
    private final PopConsumerLockService consumerLockService;
    private final Consumer<PopConsumerRecord> reviveConsumer;
    private final AtomicInteger estimateCacheSize;
    private final ConcurrentMap<String, ConsumerRecords> consumerRecordTable;

    public PopConsumerCache(BrokerController brokerController, PopConsumerKVStore consumerRecordStore, PopConsumerLockService popConsumerLockService, Consumer<PopConsumerRecord> reviveConsumer) {
        this.reviveConsumer = reviveConsumer;
        this.brokerController = brokerController;
        this.consumerRecordStore = consumerRecordStore;
        this.consumerLockService = popConsumerLockService;
        this.estimateCacheSize = new AtomicInteger();
        this.consumerRecordTable = new ConcurrentHashMap<String, ConsumerRecords>();
    }

    public String getKey(String groupId, String topicId, int queueId) {
        return groupId + "@" + topicId + "@" + queueId;
    }

    public String getKey(PopConsumerRecord consumerRecord) {
        return consumerRecord.getGroupId() + "@" + consumerRecord.getTopicId() + "@" + consumerRecord.getQueueId();
    }

    public int getCacheKeySize() {
        return this.consumerRecordTable.size();
    }

    public int getCacheSize() {
        return this.estimateCacheSize.intValue();
    }

    public boolean isCacheFull() {
        return this.estimateCacheSize.intValue() > this.brokerController.getBrokerConfig().getPopCkMaxBufferSize();
    }

    public long getMinOffsetInCache(String groupId, String topicId, int queueId) {
        ConsumerRecords consumerRecords = (ConsumerRecords)this.consumerRecordTable.get(this.getKey(groupId, topicId, queueId));
        return consumerRecords != null ? consumerRecords.getMinOffsetInBuffer() : -1L;
    }

    public long getPopInFlightMessageCount(String groupId, String topicId, int queueId) {
        ConsumerRecords consumerRecords = (ConsumerRecords)this.consumerRecordTable.get(this.getKey(groupId, topicId, queueId));
        return consumerRecords != null ? (long)consumerRecords.getInFlightRecordCount() : 0L;
    }

    public void writeRecords(List<PopConsumerRecord> consumerRecordList) {
        this.estimateCacheSize.addAndGet(consumerRecordList.size());
        consumerRecordList.forEach(consumerRecord -> {
            ConsumerRecords consumerRecords = (ConsumerRecords)ConcurrentHashMapUtils.computeIfAbsent(this.consumerRecordTable, (Object)this.getKey((PopConsumerRecord)consumerRecord), k -> new ConsumerRecords(this.brokerController.getBrokerConfig(), consumerRecord.getGroupId(), consumerRecord.getTopicId(), consumerRecord.getQueueId()));
            assert (consumerRecords != null);
            consumerRecords.write((PopConsumerRecord)consumerRecord);
        });
    }

    public List<PopConsumerRecord> deleteRecords(List<PopConsumerRecord> consumerRecordList) {
        int total = consumerRecordList.size();
        ArrayList<PopConsumerRecord> remain = new ArrayList<PopConsumerRecord>();
        consumerRecordList.forEach(consumerRecord -> {
            ConsumerRecords consumerRecords = (ConsumerRecords)this.consumerRecordTable.get(this.getKey((PopConsumerRecord)consumerRecord));
            if (consumerRecords == null || !consumerRecords.delete((PopConsumerRecord)consumerRecord)) {
                remain.add((PopConsumerRecord)consumerRecord);
            }
        });
        this.estimateCacheSize.addAndGet(remain.size() - total);
        return remain;
    }

    public int cleanupRecords(Consumer<PopConsumerRecord> consumer) {
        int remain = 0;
        Iterator iterator = this.consumerRecordTable.entrySet().iterator();
        while (iterator.hasNext()) {
            ConsumerRecords records = (ConsumerRecords)iterator.next().getValue();
            boolean timeout = this.consumerLockService.isLockTimeout(records.getGroupId(), records.getTopicId());
            if (timeout) {
                List<PopConsumerRecord> removeExpiredRecords = records.removeExpiredRecords(Long.MAX_VALUE);
                if (removeExpiredRecords != null) {
                    this.consumerRecordStore.writeRecords(removeExpiredRecords);
                }
                log.info("PopConsumerOffline, so clean expire records, groupId={}, topic={}, queueId={}, records={}", new Object[]{records.getGroupId(), records.getTopicId(), records.getQueueId(), removeExpiredRecords != null ? removeExpiredRecords.size() : 0});
                iterator.remove();
                continue;
            }
            long currentTime = System.currentTimeMillis();
            ArrayList<PopConsumerRecord> writeConsumerRecords = new ArrayList<PopConsumerRecord>();
            List<PopConsumerRecord> consumerRecords = records.removeExpiredRecords(currentTime);
            if (consumerRecords != null) {
                consumerRecords.forEach(consumerRecord -> {
                    if (consumerRecord.getVisibilityTimeout() <= currentTime) {
                        consumer.accept((PopConsumerRecord)consumerRecord);
                    } else {
                        writeConsumerRecords.add((PopConsumerRecord)consumerRecord);
                    }
                });
            }
            this.consumerRecordStore.writeRecords(writeConsumerRecords);
            long offset = records.getMinOffsetInBuffer();
            if (offset > -1L) {
                this.commitOffset("PopConsumerCache", records.getGroupId(), records.getTopicId(), records.getQueueId(), offset);
            }
            remain += records.getInFlightRecordCount();
        }
        return remain;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void commitOffset(String clientHost, String groupId, String topicId, int queueId, long offset) {
        if (!this.consumerLockService.tryLock(groupId, topicId)) {
            return;
        }
        try {
            ConsumerOffsetManager consumerOffsetManager = this.brokerController.getConsumerOffsetManager();
            long commit = consumerOffsetManager.queryOffset(groupId, topicId, queueId);
            if (commit != -1L && offset < commit) {
                log.info("PopConsumerCache, consumer offset less than store, groupId={}, topicId={}, queueId={}, offset={}", new Object[]{groupId, topicId, queueId, offset});
            }
            consumerOffsetManager.commitOffset(clientHost, groupId, topicId, queueId, offset);
        }
        finally {
            this.consumerLockService.unlock(groupId, topicId);
        }
    }

    public void removeRecords(String groupId, String topicId, int queueId) {
        this.consumerRecordTable.remove(this.getKey(groupId, topicId, queueId));
    }

    public String getServiceName() {
        return PopConsumerCache.class.getSimpleName();
    }

    public void run() {
        while (!this.isStopped()) {
            try {
                this.waitForRunning(TimeUnit.SECONDS.toMillis(1L));
                int cacheSize = this.cleanupRecords(this.reviveConsumer);
                this.estimateCacheSize.set(cacheSize);
            }
            catch (Exception e) {
                log.error("PopConsumerCacheService revive error", (Throwable)e);
            }
        }
    }

    protected static class ConsumerRecords {
        private final Lock lock;
        private final String groupId;
        private final String topicId;
        private final int queueId;
        private final BrokerConfig brokerConfig;
        private final TreeMap<Long, PopConsumerRecord> recordTreeMap;

        public ConsumerRecords(BrokerConfig brokerConfig, String groupId, String topicId, int queueId) {
            this.groupId = groupId;
            this.topicId = topicId;
            this.queueId = queueId;
            this.lock = new ReentrantLock();
            this.brokerConfig = brokerConfig;
            this.recordTreeMap = new TreeMap();
        }

        public void write(PopConsumerRecord record) {
            this.lock.lock();
            try {
                this.recordTreeMap.put(record.getOffset(), record);
            }
            finally {
                this.lock.unlock();
            }
        }

        public boolean delete(PopConsumerRecord record) {
            PopConsumerRecord popConsumerRecord;
            this.lock.lock();
            try {
                popConsumerRecord = this.recordTreeMap.remove(record.getOffset());
            }
            finally {
                this.lock.unlock();
            }
            return popConsumerRecord != null;
        }

        public long getMinOffsetInBuffer() {
            Map.Entry<Long, PopConsumerRecord> entry = this.recordTreeMap.firstEntry();
            return entry != null ? entry.getKey() : -1L;
        }

        public int getInFlightRecordCount() {
            return this.recordTreeMap.size();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public List<PopConsumerRecord> removeExpiredRecords(long currentTime) {
            ArrayList<PopConsumerRecord> result = null;
            this.lock.lock();
            try {
                Iterator<Map.Entry<Long, PopConsumerRecord>> iterator = this.recordTreeMap.entrySet().iterator();
                while (iterator.hasNext()) {
                    Map.Entry<Long, PopConsumerRecord> entry = iterator.next();
                    if (entry.getValue().getVisibilityTimeout() > currentTime && entry.getValue().getPopTime() + (long)this.brokerConfig.getPopCkStayBufferTime() > currentTime) continue;
                    if (result == null) {
                        result = new ArrayList<PopConsumerRecord>();
                    }
                    result.add(entry.getValue());
                    iterator.remove();
                }
            }
            finally {
                this.lock.unlock();
            }
            return result;
        }

        public String getGroupId() {
            return this.groupId;
        }

        public String getTopicId() {
            return this.topicId;
        }

        public int getQueueId() {
            return this.queueId;
        }

        public String toString() {
            return "ConsumerRecords{lock=" + this.lock + ", topicId=" + this.topicId + ", groupId=" + this.groupId + ", queueId=" + this.queueId + ", recordTreeMap=" + this.recordTreeMap.size() + '}';
        }
    }
}

