Object subclass: #FactBase
	instanceVariableNames: 'facts '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Logic_Processing'!
FactBase comment:
'FactBase implements logical data 
(in fact sequenceable collection of sequenceable collections).

FactBase may be used for building of expert system database.

(C) 	Vojtech Merunka
	Smetanova 647
	CZ-286 01 CASLAV

	merunka@pef.czu.cz
	http://omega.pef.czu.cz/pef/ki/merunka'!


!FactBase methodsFor: 'fact accessing'!

add: aFact 
	facts add: aFact!

addFirst: aFact 
	facts addFirst: aFact!

addLast: aFact 
	facts addLast: aFact!

facts
	^facts!

remove: aFact 
	facts remove: aFact ifAbsent: []!

size
	^facts size! !

!FactBase methodsFor: 'printing'!

printOn: aStream 
	aStream nextPutAll: 'FactBase('.
	facts
		do: 
			[:f | 
			aStream cr; tab.
			f printOn: aStream].
	aStream nextPut: $)! !

!FactBase methodsFor: 'instance initialization'!

initialize
	facts := OrderedCollection new: 16! !

!FactBase methodsFor: 'querying'!

call
	"Return every fact in factbase"

	| f |
	1 to: facts size
		do: 
			[:i | 
			f := facts at: i.
			self succeed: f].
	^nil!

call: aFactPattern 
	"Return the fact, which responds to the pattern."

	| result f |
	1 to: facts size
		do: 
			[:i | 
			f := facts at: i.
			result := true.
			1 to: f size do: [:j | (aFactPattern at: j)
					= (f at: j) ifFalse: [result := false]].
			result ifTrue: [self succeed: f]].
	^nil!

call: aFactPattern andReturnAt: anIndex 
	"From fact return the item given by anIndex. Fact must respond to the pattern."

	| result f |
	1 to: facts size
		do: 
			[:i | 
			f := facts at: i.
			result := true.
			1 to: f size do: [:j | (aFactPattern at: j)
					= (f at: j) ifFalse: [result := false]].
			result ifTrue: [self succeed: (f at: anIndex)]].
	^nil!

callAndReturnAt: anIndex 
	"Return anIndex item of every fact in factbase"

	| f |
	1 to: facts size
		do: 
			[:i | 
			f := facts at: i.
			self succeed: (f at: anIndex)].
	^nil!

callValue: aValue at: aCallIndex andReturnAt: aReturnIndex 
	"From fact return the item given by aReturnIndex."
	"aCallIndexValue in fact must be equal to"

	| f |
	1 to: facts size
		do: 
			[:i | 
			f := facts at: i.
			aValue = (f at: aCallIndex) ifTrue: [self succeed: (f at: aReturnIndex)]].
	^nil! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

FactBase class
	instanceVariableNames: ''!


!FactBase class methodsFor: 'instance creation'!

new
	^super new initialize!

with: aCollectionOfFacts 
	| new |
	new := self new.
	aCollectionOfFacts do: [:f | new addLast: f].
	^new! !

Object subclass: #AnyValue
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Logic_Processing'!
AnyValue comment:
'I represent object which returns to any comparing =
the value of true.

Instances may be used in query patterns sent to FactBase objects.

(C) 	Vojtech Merunka
	Smetanova 647
	CZ-286 01 CASLAV

	merunka@pef.czu.cz
	http://omega.pef.czu.cz/pef/ki/merunka

'!


!AnyValue methodsFor: 'printing'!

printOn: aStream 
	aStream nextPutAll: ' anyValue '! !

!AnyValue methodsFor: 'comparing'!

= anObject 
	^true! !

Object subclass: #ValueStream
	instanceVariableNames: 'goal result position '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Logic_Processing'!
ValueStream comment:
'ValueStream is a device, which sequently creates outgoing values from the BlockClosure
from which it has been created. The sequence must end by the nil value.

- see class method examples for better understanding

(C) 	Vojtech Merunka
	Smetanova 647
	CZ-286 01 CASLAV

	merunka@pef.czu.cz
	http://omega.pef.czu.cz/pef/ki/merunka'!


!ValueStream methodsFor: 'private - instance creation'!

on: aBlock 
	"create and initialize the stream - called by the class method 'on:'"

	position := 0.
	goal := aBlock.
	result := goal value! !

!ValueStream methodsFor: 'accessing'!

contents
	"Answer with a copy of the receiver's readable information."

	| coll |
	coll := self contentsSpecies new: 16.
	self do: [:each | coll addLast: each].
	^coll!

flush
	"ReadStreams do not need to flush."

	^self shouldNotImplement!

next
	| return |
	self atEnd ifTrue: [^nil].
	return := result.
	result := goal value.
	position := position + 1.
	^return!

next: anInteger 
	"Answer the next anInteger elements of the receiver"

	| newCollection |
	newCollection := self contentsSpecies new: anInteger.
	anInteger timesRepeat: [newCollection addLast: self next].
	^newCollection!

nextAvailable: anInteger 
	"Answer the next anInteger (or less if at end) elements of the receiver"

	| newCollection |
	newCollection := self contentsSpecies new: anInteger.
	anInteger timesRepeat: [self atEnd ifFalse: [newCollection addLast: self next]].
	^newCollection!

nextMatchFor: anObject 
	"Read the next element and answer whether it is equal to anObject."

	^anObject = self next!

peek
	"Answer what would be returned with a self next, without 
	changing position. If the receiver is at the end, answer nil."

	self atEnd ifTrue: [^nil].
	^result!

peekFor: anObject 
	"Answer false and do not move the position if self next ~= anObject or if the 
	receiver is at the end. Answer true and increment position if self next = anObject."

	| return |
	self atEnd ifTrue: [^false].
	(return := self peek = anObject) ifTrue: [self next].
	^return!

size
	"this stream has unknown size"

	^self shouldNotImplement!

through: anObject 
	"Answer a subcollection from the current position to the occurrence (if any, inclusive) 
	of anObject. If not there, answer everything."

	| newColl element |
	newColl := self contentsSpecies new: 16.
	[self atEnd]
		whileFalse: 
			[newColl addLast: (element := self next).
			element = anObject ifTrue: [^newColl]].
	^newColl!

upTo: anObject 
	"Answer a subcollection from position to the occurrence (if any, exclusive) of anObject. 
	The stream is left positioned after anObject. 
	If anObject is not found answer everything."

	| newColl element |
	newColl := self contentsSpecies new: 16.
	[self atEnd]
		whileFalse: 
			[(element := self next) = anObject ifTrue: [^newColl].
			newColl addLast: element].
	^newColl!

upToEnd
	"Answer a collection consisting of all the elements from the current position 
	to the end of stream."

	^self contents! !

!ValueStream methodsFor: 'testing'!

atEnd
	"Answer whether the position is greater than or equal to the limit."

	^result isNil!

isReadable
	"Answer whether the stream implements #next."

	^true!

isWritable
	"Answer whether the stream implements #nextPut:."

	^false!

needsFileLineEndConversion
	"Test whether a wrapper stream on the receiver
	may wish to do line end conversion."

	^false! !

!ValueStream methodsFor: 'enumerating'!

do: aBlock 
	"Evaluate aBlock for each of the elements of the receiver."

	[self atEnd]
		whileFalse: [aBlock value: self next]! !

!ValueStream methodsFor: 'status'!

close
	"Sets the status of the stream to be closed."

	"This message does nothing at this level, but is included for ExternalStream 
	compatibility."

	^self! !

!ValueStream methodsFor: 'positioning'!

position
	"Answer the current position of accessing the stream."

	^position!

position: anInteger 
	"Set position to anInteger as long as anInteger is within the bounds of the 
	receiver's contents. If it is not, provide an error notification."

	self skip: anInteger - position!

reset
	"Set the receiver's position to 0."

	self position: 0!

setToEnd
	"Set the position of the receiver to the end of its stream of elements."

	[self atEnd]
		whileFalse: [self next]!

skip: anInteger 
	"Move forward my position by anInteger"

	anInteger < 0 ifTrue: [^self error: 'Skip backward is not allowed!!'].
	anInteger timesRepeat: [self next]!

skipThrough: anObject
	"Skip forward to the occurrence of anObject.
	 Leaves positioned after anObject. 
	If the object is not found the stream is positioned at the end and nil is returned."

	[self atEnd ifTrue: [^nil].
	self next = anObject ifTrue: [^self]] repeat!

skipUpTo: anObject 
	"skip forward to the occurrence (if any, not inclusive) of anObject. If not there, answer nil. 
	Leaves positioned before anObject."

	[self atEnd not and: [self peek ~= anObject]]
		whileTrue: [self next]! !

!ValueStream methodsFor: 'private'!

contentsSpecies

	^OrderedCollection! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ValueStream class
	instanceVariableNames: ''!


!ValueStream class methodsFor: 'instance creation'!

on: aBlock 
	^self basicNew on: aBlock! !

!ValueStream class methodsFor: 'examples'!

example
	"do it: self example"

	| stream aBlock |
	aBlock := 
			[| seconds |
			seconds := Time now seconds.
			seconds \\ 20 = 0
				ifTrue: [nil]
				ifFalse: [seconds]].
	Transcript clear.
	stream := ValueStream on: aBlock.
	[stream atEnd]
		whileFalse: [Transcript show: stream next printString , ' ']! !

ValueStream subclass: #LogicValueStream
	instanceVariableNames: 'senderSemaphore solverSemaphore '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Logic_Processing'!
LogicValueStream comment:
'LogicValueStream is a device, which sequently creates outgoing values from the logical
expression from which it has been created. The sequence of values ends when the logical expression failed.

- see class method examples for better understanding

(C) 	Vojtech Merunka
	Smetanova 647
	CZ-286 01 CASLAV

	merunka@pef.czu.cz
	http://omega.pef.czu.cz/pef/ki/merunka'!


!LogicValueStream methodsFor: 'private - instance creation'!

on: aBlock 
	"create and initialize the stream - called by the class method 'on:'"

	solverSemaphore := Semaphore new.
	senderSemaphore := Semaphore new.
	position := 0.
	goal := aBlock asGoal.
	
	[goal call.
	result := goal result.
	senderSemaphore signal.
	goal failed
		ifFalse: 
			[solverSemaphore wait.
			self fail]] fork.
	senderSemaphore wait! !

!LogicValueStream methodsFor: 'accessing - logic specific'!

next
	"Answer the next object in the receiver."

	| return |
	self atEnd
		ifTrue: [return := nil]
		ifFalse: 
			[return := result.
			position := position + 1.
			solverSemaphore signal.
			senderSemaphore wait].
	^return! !

!LogicValueStream methodsFor: 'testing - logic specific'!

atEnd
	"Answer whether the position is greater than or equal to the limit."

	^goal failed! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

LogicValueStream class
	instanceVariableNames: ''!


!LogicValueStream class methodsFor: 'examples'!

example1
	"do it: self example1"

	| stream |
	Transcript clear.
	stream := LogicValueStream on: [FamilyExample default grandChildOf: #Jan].
	[stream atEnd]
		whileFalse: [Transcript show: stream next printString; cr]!

example2
	"inspect: self example2"

	^(LogicValueStream on: [FamilyExample default grandChildOf: #Jan]) contents!

example3
	"print it: self example3"

	^(LogicValueStream on: [FamilyExample default grandChildOf: #Jan]) skipThrough: #Vojta ; position! !

Object subclass: #Goal
	instanceVariableNames: 'result block '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Logic_Processing'!
Goal comment:
'This class represents objects which are ready to logical evaluation.
Goals are created from BlockClosures by the message send "asGoal"

(C) 	Vojtech Merunka
	Smetanova 647
	CZ-286 01 CASLAV

	merunka@pef.czu.cz
	http://omega.pef.czu.cz/pef/ki/merunka'!


!Goal methodsFor: 'public interface'!

block: anObject
	"set the value of instance variable"

	block := anObject!

result
	"get the value of instance variable"

	^result! !

!Goal methodsFor: 'goal processing'!

& aBlock 
	| nextGoal |
	self failed ifTrue: [^self].
	nextGoal := aBlock asGoal.
	nextGoal call.
	nextGoal failed ifTrue: [^self fail].
	^nextGoal!

call
	result := block value!

failed
	^(result isKindOf: Goal)
		ifTrue: [result failed]
		ifFalse: [result isNil]!

| aBlock 
	| aGoal |
	self failed ifFalse: [^self].
	aGoal := aBlock asGoal.
	aGoal call.
	^aGoal! !

!Goal methodsFor: 'printing'!

printOn: aStream 
	aStream nextPutAll: (self failed
			ifTrue: [' a failed Goal ']
			ifFalse: [' an exited Goal '])! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Goal class
	instanceVariableNames: ''!


!Goal class methodsFor: 'instance creation'!

with: aBlock 
	| new |
	new := self new.
	new block: aBlock.
	^new! !

