Controller subclass: #ToolbarController
	instanceVariableNames: 'pressing pressingItem '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Toolbar'!
ToolbarController comment:
'ToolbarController is the controller for the ToolbarView.

Instance Variables:
	pressing	<Boolean>	are we pressing a button?
	pressingItem	<ButtonDescription | nil>	the items we''re pressing

'!


!ToolbarController methodsFor: 'basic control sequence'!

controlTerminate
	pressing ifFalse: [view over: nil].
	^super controlTerminate! !

!ToolbarController methodsFor: 'control defaults'!

controlActivity
	self viewHasCursor 
		ifTrue: 
			[self sensor redButtonPressed 
				ifTrue: [self redButtonActivity]
				ifFalse: [self mouseMovedOver: self sensor cursorPoint]].
	^super controlActivity!

endSelectionAt: point 
	(pressingItem notNil and: [pressingItem == (view itemUnder: point)])
		ifTrue: [self requestValueChange ifTrue: [model selection: pressingItem]].
	pressingItem := nil.
	view selecting: nil.
	pressing := false!

mouseMovedOver: point 
	| item |
	item := view itemUnder: point.
	(item notNil and: [item isEnabled])
		ifTrue: [view over: item]
		ifFalse: [view over: nil]!

moveSelectionTo: point 
	pressingItem == (view itemUnder: point)
		ifTrue: [view selecting: pressingItem]
		ifFalse: [view selecting: nil]!

redButtonActivity
	self startSelectionAt: self sensor cursorPoint.
	[self sensor redButtonPressed] 
		whileTrue: 
			[self moveSelectionTo: self sensor cursorPoint.
			view repairDamage].
	self endSelectionAt: self sensor cursorPoint!

startSelectionAt: point 
	pressing := true.
	pressingItem := view itemUnder: point.
	(pressingItem notNil and: [pressingItem isEnabled])
		ifTrue: [view over: pressingItem]
		ifFalse: [pressingItem := nil].
	pressingItem notNil ifTrue: [view selecting: pressingItem]! !

!ToolbarController methodsFor: 'events'!

mouseMovedEvent: anEvent 
	pressing
		ifTrue: [self moveSelectionTo: (self sensor mousePointFor: anEvent)]
		ifFalse: [self mouseMovedOver: (self sensor mousePointFor: anEvent)]!

redButtonPressedEvent: event 
	self startSelectionAt: (self sensor mousePointFor: event)!

redButtonReleasedEvent: event 
	self endSelectionAt: (self sensor mousePointFor: event)! !

!ToolbarController methodsFor: 'initialize-release'!

initialize
	super initialize.
	pressing := false! !

View subclass: #ToolbarView
	instanceVariableNames: 'horizontal over selecting '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Toolbar'!
ToolbarView comment:
'ToolbarView is the view that displays the toolbar.

Instance Variables:
	horizontal	<Boolean>	description of horizontal
	over	<ButtonDescription>	the item the mouse cursor is over
	selecting	<ButtonDescription>	the item we''re tentatively selecting

'!


!ToolbarView methodsFor: 'accessing'!

buttons
	^self model list!

horizontal
	^horizontal!

horizontal: aBoolean 
	horizontal := aBoolean.
	self invalidate!

invalidateOverIcon
	over notNil 
		ifTrue: 
			[self
				invalidateRectangle: ((self boundsFor: over) expandedBy: self spacer)
				repairNow: false]!

invalidateSelectedIcon
	selecting notNil 
		ifTrue: 
			[self 
				invalidateRectangle: ((self boundsFor: selecting) expandedBy: self spacer)
				repairNow: false]!

over
	^over!

over: anObject 
	over == anObject ifTrue: [^self].
	self invalidateOverIcon.
	over := anObject.
	self invalidateOverIcon!

selecting: anObject 
	selecting == anObject ifTrue: [^self].
	self invalidateSelectedIcon.
	selecting := anObject.
	self invalidateSelectedIcon! !

!ToolbarView methodsFor: 'controller accessing'!

defaultControllerClass
	^ToolbarController!

itemUnder: aPoint 
	self buttonsAndPositionsDo: 
			[:each :point | 
			((point extent: each icon extent) containsPoint: aPoint) ifTrue: [^each]].
	^nil!

mouseMovedOver: aPoint 
	self over: (self itemUnder: aPoint)! !

!ToolbarView methodsFor: 'displaying'!

display: each at: point on: aGraphicsContext 
	| inset |
	inset := selecting == each or: [self isSelected: each].
	each icon 
		displayOn: aGraphicsContext
		at: point + (inset ifTrue: [1] ifFalse: [0]).
	(each == over and: [inset not]) 
		ifTrue: 
			[self overBorder 
				displayOn: aGraphicsContext
				forDisplayBox: ((point extent: each icon extent) expandedBy: 3 @ 3)
				using: self].
	inset 
		ifTrue: 
			[self selectedBorder 
				displayOn: aGraphicsContext
				forDisplayBox: ((point extent: each icon extent) expandedBy: 3 @ 3)
				using: self]!

displayBackgroundOn: aGraphicsContext 
	aGraphicsContext paint: self backgroundColor.
	aGraphicsContext displayRectangle: aGraphicsContext clippingBounds!

displayForegroundOn: aGraphicsContext 
	self buttonsAndPositionsDo: 
			[:each :point | 
			self 
				display: each
				at: point
				on: aGraphicsContext]!

displayOn: aGraphicsContext 
	self displayBackgroundOn: aGraphicsContext.
	self displayForegroundOn: aGraphicsContext! !

!ToolbarView methodsFor: 'initialize-release'!

initialize
	super initialize.
	horizontal := true! !

!ToolbarView methodsFor: 'private'!

boundsFor: aButton 
	self buttonsAndPositionsDo: 
			[:each :point | 
			each == aButton ifTrue: [^point extent: each icon extent]].
	^1 @ 1 extent: 0 @ 0!

buttonsAndPositionsDo: aBlock 
	self buttons inject: self startPosition
		into: 
			[:pos :each | 
			aBlock value: each value: pos.
			pos + (horizontal
						ifTrue: [(each icon extent x + self spacer) @ 0]
						ifFalse: [0 @ (each icon extent y + self spacer)])]!

isSelected: anObject 
	^model selection == anObject!

overBorder
	^BeveledBorder raised!

selectedBorder
	^BeveledBorder inset!

spacer
	^5!

startPosition
	^self spacer @ self spacer! !

!ToolbarView methodsFor: 'visual properties'!

backgroundColor
	^ColorValue 
		red: 198 / 256.0
		green: 195 / 256.0
		blue: 198 / 256.0! !

Object subclass: #ButtonDescription
	instanceVariableNames: 'icon enabled value '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Toolbar'!
ButtonDescription comment:
'ButtonDescription represents a button in the Toolbar. It contains both the icon that is displayed as well as its value.

Instance Variables:
	enabled	<Boolean>	are we enabled, disabled buttons can be greyed out
	icon	<Image>	our image to display
	value	<Object>	some data that we represent (can be anything, it isn''t interpreted by us)

'!


!ButtonDescription methodsFor: 'accessing'!

disable
	^self enabled: false!

enable
	^self enabled: true!

enabled: aBoolean 
	enabled := aBoolean.
	(icon respondsTo: #isEnabled:) ifTrue: [icon isEnabled: aBoolean]!

icon
	^icon!

icon: anObject
	icon := anObject!

value
	^value!

value: anObject
	value := anObject! !

!ButtonDescription methodsFor: 'initialize-release'!

initialize
	enabled := false! !

!ButtonDescription methodsFor: 'testing'!

isEnabled
	^enabled! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

ButtonDescription class
	instanceVariableNames: ''!


!ButtonDescription class methodsFor: 'instance creation'!

icon: anIcon value: anObject 
	^(self new)
		icon: anIcon;
		enable;
		value: anObject;
		yourself!

new
	^super new initialize!

spacer
	^self spacer: 12 @ 12!

spacer: extent 
	^(self new)
		icon: ((VisualRegion new)
					extent: extent;
					lineWidth: 0;
					yourself);
		disable;
		yourself! !

