Controller subclass: #DrawingController
	instanceVariableNames: 'tool keyboardProcessor '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
DrawingController comment:
'DrawingController is a Controller for a Drawing view.

Instance Variables:
	keyboardProcessor	<KeyboardProcessor>	the keyboard processor for our window
	tool	<Tool>	the current tool selected

'!


!DrawingController methodsFor: 'accessing'!

currentTool
	^tool value! !

!DrawingController methodsFor: 'changing'!

changedTool
	view clearSelections.
	(self currentTool)
		controller: self;
		selected.
	self currentTool cursor show.
	self setAsCurrentConsumer! !

!DrawingController methodsFor: 'event driven'!

desiresFocus
	^true!

handleEvent: anEvent 
	anEvent key == #enter ifTrue: [self currentTool cursor show].
	anEvent key == #exit ifTrue: [Cursor normal show].
	(anEvent isMouseEvent or: [anEvent isKeyboard]) 
		ifTrue: 
			[(self currentTool handleEvent: anEvent) 
				ifTrue: [self setAsCurrentConsumer]]!

keyboardProcessor
	^keyboardProcessor!

keyboardProcessor: aKeyboardProcessor 
	keyboardProcessor := aKeyboardProcessor!

setAsCurrentConsumer
	| ctrl |
	keyboardProcessor isNil ifTrue: [^self].
	ctrl := keyboardProcessor currentConsumer.
	ctrl ~= self ifTrue: [keyboardProcessor requestActivationFor: self]! !

!DrawingController methodsFor: 'initialize-release'!

initialize
	super initialize.
	tool := Tool selectionTool asValue.
	tool value controller: self!

tool: aValueModel 
	tool := aValueModel.
	tool onChangeSend: #changedTool to: self.
	self changedTool! !

!DrawingController methodsFor: 'menu processing'!

localMenuItem: aSymbol 
	"Answer an Array of Symbols that represent the menu messages that should be sent to my
	View as opposed to my Model."

	^#(#cut #paste #copySelection #group #ungroup) includes: aSymbol!

processMenuAt: globalPoint local: localPoint for: aFigure 
	| valueResult |
	valueResult := self trackMenu: (aFigure menuAt: localPoint)
				at: globalPoint
				centered: true.
	valueResult isNil ifTrue: [^self].
	valueResult isSymbol
		ifTrue: 
			[(self localMenuItem: valueResult)
				ifTrue: 
					[aFigure isDrawing ifFalse: [view selection: aFigure].
					view perform: valueResult]
				ifFalse: 
					[(aFigure model notNil and: [aFigure model respondsTo: valueResult])
						ifTrue: [aFigure model perform: valueResult]
						ifFalse: [aFigure perform: valueResult]]]
		ifFalse: [valueResult value]! !

Object subclass: #FigureAttributes
	instanceVariableNames: 'lineWidth lineColor fillColor '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
FigureAttributes comment:
'FigureAttributes is a record that holds some attibutes of figures. These attributes could be incorporated into the figures, but this saves space since most figures don''t specialize these attributes.

Instance Variables:
	fillColor	<ColorValue>	if our figure is filled, this is the color to use
	lineColor	<ColorValue>	our outline''s color
	lineWidth	<Integer>	how wide is our outline
'!


!FigureAttributes methodsFor: 'accessing'!

fillColor
	^fillColor!

fillColor: aColorValue 
	fillColor := aColorValue!

lineColor
	^lineColor!

lineColor: aColorValue 
	lineColor := aColorValue!

lineWidth
	^lineWidth!

lineWidth: anInteger 
	lineWidth := anInteger! !

VisualPart subclass: #Figure
	instanceVariableNames: 'dependents state attributes model bounds '
	classVariableNames: 'Connectable Filled Invisible Moveable Removeable Selectable Selected '
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
Figure comment:
'Figure is an abstract class. It represents a graphical item that is contained in a drawing.

Subclasses must implement the following messages:
	displaying
		displayFigureOn:	displays the figure on a graphics context

Instance Variables:
	attributes	<FigureAttributes | nil>	my attributes such as line color, fill color, etc. If nil, I inherit my container''s attributes
	bounds	<Rectangle>	my bounds in the drawing
	dependents	<DependentsCollection | nil>	objects that depend on me, they will be notified when I change
	model	<Object>	the object I''m a view for, normally not assigned, (can be used to store extra data for all figures)
	state	<SmallInteger>	some flag for attributes such as selectable, moveable, etc.

Class Variables:
	Connectable	<SmallInteger>	mask for the connection state
	Filled	<SmallInteger>	mask
	Invisible	<SmallInteger>	mask
	Moveable	<SmallInteger>	mask
	Removeable	<SmallInteger>	mask
	Selectable	<SmallInteger>	mask
	Selected	<SmallInteger>	mask

'!


!Figure methodsFor: 'accessing'!

constrain: aSelector to: aMessageSend 
	"Constrain yourself to aMessageSend's value. If aSelector is a setter, then
	send the message directly. If its a getter, then use the translateBy: message
	to move to the new position. See PositionConstraint for details."

	^PositionConstraint send: aSelector
		to: self
		with: aMessageSend!

container: aContainer 
	aContainer isNil ifTrue: [self delete].
	super container: aContainer!

drawing
	"Return the drawing that you are located in or nil if you are not in any drawing right now."

	container isNil ifTrue: [^nil].
	^container drawing!

figureAt: aPoint 
	self isVisible ifFalse: [^nil].
	^(self containsPoint: aPoint) ifTrue: [self] ifFalse: [nil]!

handles
	| handles |
	handles := OrderedCollection withAll: (TrackHandle allCornersOf: self).
	handles add: (Handle connectionOn: self at: #center).
	^handles!

menuAt: aPoint 
	"Figures have a menu that pops-up when clicked on by the yellow mouse button."

	| mb |
	mb := MenuBuilder new.
	mb
		addCopyCutPaste;
		line;
		add: 'inspect...' -> #inspect;
		line.
	self addLineColorMenuTo: mb.
	self addFillColorMenuTo: mb.
	self addLineWidthMenuTo: mb.
	self addModelMenuTo: mb at: aPoint.
	^mb menu!

model
	^model!

model: anObject
	model := anObject.
	model addDependent: self!

owner
	^self! !

!Figure methodsFor: 'attribute accessing'!

attributes
	^attributes isNil
		ifTrue: [attributes := FigureAttributes new]
		ifFalse: [attributes]!

beInvisible
	self setBit: Invisible to: true.
	self invalidate!

beVisible
	self setBit: Invisible to: false.
	self invalidate!

connectable: aBoolean 
	self setBit: Connectable to: aBoolean!

deselect
	self setBit: Selected to: false!

fillColor
	^(attributes isNil or: [attributes fillColor isNil])
		ifTrue: 
			[container isNil ifTrue: [self backgroundColor] ifFalse: [container fillColor]]
		ifFalse: [attributes fillColor]!

fillColor: aColorValue 
	self attributes fillColor: aColorValue.
	self invalidate!

isSelected
	^(state bitAnd: Selected) ~~ 0!

lineColor
	^(attributes isNil or: [attributes lineColor isNil])
		ifTrue: 
			[container isNil ifTrue: [self foregroundColor] ifFalse: [container lineColor]]
		ifFalse: [attributes lineColor]!

lineColor: aColorValue 
	self attributes lineColor: aColorValue.
	self invalidate!

lineWidth
	^(attributes isNil or: [attributes lineWidth isNil])
		ifTrue: [container isNil ifTrue: [1] ifFalse: [container lineWidth]]
		ifFalse: [attributes lineWidth]!

lineWidth: anInteger 
	self attributes lineWidth: anInteger.
	self recomputePreferredBounds!

moveable: aBoolean 
	self setBit: Moveable to: aBoolean not!

opaque
	self setBit: Filled to: true.
	self invalidate!

removeable: aBoolean 
	self setBit: Removeable to: aBoolean not!

select
	self setBit: Selected to: true!

selectable: aBoolean 
	self setBit: Selectable to: aBoolean not!

transparent
	self setBit: Filled to: false.
	self invalidate! !

!Figure methodsFor: 'bounding box accessing'!

bottomCenter
	^self bounds bottomCenter!

bottomLeft
	^self bounds bottomLeft!

bottomLeft: aPoint 
	self setBoundsTo: (Rectangle vertex: aPoint x @ self origin y
				vertex: self corner x @ aPoint y)!

bottomRight
	^self bounds bottomRight!

bottomRight: aPoint 
	self setBoundsTo: (Rectangle vertex: self origin vertex: aPoint)!

center
	^self bounds center!

corner
	^self bounds corner!

corner: aPoint 
	self bottomRight: aPoint!

extent
	^self bounds extent!

left
	^self bounds left!

leftCenter
	^self bounds leftCenter!

offCenter: deltaPoint 
	^self center + deltaPoint!

offCorner: deltaPoint 
	^self corner + deltaPoint!

offOrigin: deltaPoint 
	^self origin + deltaPoint!

origin
	^self bounds origin!

origin: aPoint 
	self topLeft: aPoint!

position: aPoint offset: offsetPoint 
	^self origin + (self extent * aPoint) + offsetPoint!

right
	^self bounds right!

rightCenter
	^self bounds rightCenter!

top
	^self bounds top!

topCenter
	^self bounds topCenter!

topLeft
	^self bounds topLeft!

topLeft: aPoint 
	self setBoundsTo: (Rectangle vertex: aPoint vertex: self corner)!

topRight
	^self bounds topRight!

topRight: aPoint 
	self setBoundsTo: (Rectangle vertex: self origin x @ aPoint y
				vertex: aPoint x @ self corner y)! !

!Figure methodsFor: 'bounds accessing'!

computePreferredBounds
	"Compute the preferredBounds of the figure. For some figures, the preferredBounds
	is always set so this method is never called. For the other figures, this should
	return a rectangle of the bounds."

	^bounds!

preferredBounds
	^bounds isNil
		ifTrue: [bounds := self computePreferredBounds]
		ifFalse: [bounds]!

recomputePreferredBounds
	^self setBoundsTo: self computePreferredBounds! !

!Figure methodsFor: 'changing'!

changed: anAspectSymbol with: aParameter 
	super changed: anAspectSymbol with: aParameter.
	self invalidate!

delete
	"This message is sent when a figure is being deleted. Default behavior is to notify dependents. For
	example, this will let a line delete itself when the figures it is attached to are deleted."

	self changed: #deleted with: nil!

deletionUpdateFrom: aFigure 
	"aFigure was removed from the drawing. I don't know what you want to do, so I just remove myself as a
	deletion dependent of aFigure. For lines the default action is to remove the line."

	aFigure removeDependent: self!

update: anAspectSymbol with: aParameter from: aSender 
	"Somebody has justed changed. If it is as a result of a deletion dependent then it will be #deleted."

	anAspectSymbol = #deleted ifTrue: [^self deletionUpdateFrom: aSender].
	super update: anAspectSymbol
		with: aParameter
		from: aSender! !

!Figure methodsFor: 'connection'!

canConnectFromFigure: aFigure 
	"Can we connect from aFigure. We might want some additional checking to verify that
	this is a legal operation for our drawing's sematics."

	^true!

canConnectFromPoint: aPoint 
	"Should we allow outgoing connections from aPoint?"

	^self canBeConnected!

canConnectFromPoint: myPoint to: aFigure at: figurePoint 
	^(self canConnectFromPoint: myPoint) and: 
			[(aFigure canConnectToPoint: figurePoint) and: 
					[(self canConnectToFigure: aFigure) 
						and: [aFigure canConnectFromFigure: self]]]!

canConnectToFigure: aFigure 
	"Can we connect to aFigure. We might want some additional checking to verify that
	this is a legal operation for our drawing's sematics."

	^self canBeConnected and: [self ~~ aFigure]!

canConnectToPoint: aPoint 
	"Can we allow a connection to aPoint? If we have several 'ports' we might only allow
	certain connections to some ports."

	^self canBeConnected!

connectFromPoint: myPoint to: aFigure at: figurePoint 
	"Return a figure that represents the connection from self to aFigure (using myPoint
	and figurePoint as the connection points)."

	| newFigure |
	newFigure := self connectionFigureClass connect: self to: aFigure.
	self createStartConnectionConstraintFor: newFigure at: myPoint.
	aFigure createStopConnectionConstraintFor: newFigure at: figurePoint.
	^newFigure!

connectionFigureClass
	"The type of figure that we should use for connections"

	^LineFigure!

connectionPositions
	"If we don't use a boundary constraint, try to guess a good location for the end point of the line."

	^#(#origin #corner #topRight #bottomLeft #center #topCenter #rightCenter #bottomCenter #leftCenter)!

createStartConnectionConstraintFor: newFigure at: myPoint 
	"If we are a filled figure, then try connecting to our boundary (our center
	must be inside us). If not filled, then try to find what part we are connecting
	to/from. Subclasses can override to connect to figure parts."

	self isOpaque
		ifTrue: [BoundaryConstraint forFigure: self startLine: newFigure]
		ifFalse: 
			[| position |
			position := self shortestPositionTo: myPoint.
			PositionConstraint send: #startPoint:
				to: newFigure
				with: (MessageSend receiver: self selector: position)]!

createStopConnectionConstraintFor: newFigure at: myPoint 
	"If we are a filled figure, then try connecting to our boundary (our center
	must be inside us). If not filled, then try to find what part we are connecting
	to/from. Subclasses can override to connect to figure parts."

	self isOpaque
		ifTrue: [BoundaryConstraint forFigure: self stopLine: newFigure]
		ifFalse: 
			[| position |
			position := self shortestPositionTo: myPoint.
			PositionConstraint send: #stopPoint:
				to: newFigure
				with: (MessageSend receiver: self selector: position)]!

shortestPositionTo: myPoint 
	"Return a symbol from #connectionPositions that is the shortest distance from myPoint."

	| min position positions |
	positions := self connectionPositions.
	positions isEmpty ifTrue: [^#center].
	min := myPoint dist: (self perform: positions first).
	position := positions first.
	positions do: 
			[:each | 
			| dist |
			dist := myPoint dist: (self perform: each).
			dist < min 
				ifTrue: 
					[min := dist.
					position := each]].
	^position! !

!Figure methodsFor: 'copying'!

postCopy
	super postCopy.
	container := nil.
	dependents := nil.
	bounds := self computePreferredBounds.
	self deselect! !

!Figure methodsFor: 'displaying'!

displayFigureOn: aGraphicsContext 
	^self subclassResponsibility!

displayOn: aGraphicsContext 
	self isVisible ifTrue: 
			[self isSelected
				ifTrue: [self displaySelectedFigureOn: aGraphicsContext]
				ifFalse: [self displayFigureOn: aGraphicsContext]]!

displaySelectedFigureOn: aGraphicsContext 
	^self displayFigureOn: aGraphicsContext! !

!Figure methodsFor: 'initialize-release'!

initialize
	super initialize.
	model := nil.
	state := Connectable!

release
	model removeDependent: self.
	^super release! !

!Figure methodsFor: 'private'!

addFillColorMenuTo: aMenuBuilder 
	aMenuBuilder beginSubMenuLabeled: 'fill color'.
	ColorValue constantNames do: 
			[:each | 
			aMenuBuilder 
				add: each asString -> [self fillColor: (ColorValue perform: each)]].
	aMenuBuilder endSubMenu!

addLineColorMenuTo: aMenuBuilder 
	aMenuBuilder beginSubMenuLabeled: 'line color'.
	ColorValue constantNames do: 
			[:each | 
			aMenuBuilder 
				add: each asString -> [self lineColor: (ColorValue perform: each)]].
	aMenuBuilder endSubMenu!

addLineWidthMenuTo: aMenuBuilder 
	aMenuBuilder beginSubMenuLabeled: 'line width'.
	1 to: 5
		do: 
			[:i | 
			aMenuBuilder 
				add: (i printString , (i == 1 ifTrue: [' pixel'] ifFalse: [' pixels'])) 
						-> [self lineWidth: i]].
	aMenuBuilder endSubMenu!

addModelMenuTo: aMenuBuilder at: aPoint 
	(model respondsTo: #menuAt:) 
		ifTrue: 
			[aMenuBuilder
				line;
				add: 'Model' -> (model menuAt: aPoint)]
		ifFalse: 
			[(model respondsTo: #menu) 
				ifTrue: 
					[aMenuBuilder
						line;
						add: 'Model' -> model menu]]!

addVisibilityMenuTo: aMenuBuilder 
	aMenuBuilder
		beginSubMenuLabeled: 'visibility';
		add: 'transparent' -> #transparent;
		add: 'opaque' -> #opaque;
		endSubMenu!

myDependents
	"Answer the receiver's dependents or nil. Copied down from Model to make dependency checking faster."

	^dependents!

myDependents: dependentsOrNil
	"Set the receivers dependents. Copied down from Model to make dependency checking faster."

	dependents := dependentsOrNil!

setBit: mask to: aBoolean 
	state := state bitOr: mask.
	aBoolean ifFalse: [state := state bitXor: mask]! !

!Figure methodsFor: 'testing'!

canBeConnected
	^(state bitAnd: Connectable) ~~ 0!

containedBy: aRectangle 
	"Are you contained in aRectangle?"

	^aRectangle contains: self bounds!

isComposite
	^false!

isDrawing
	^false!

isHandle
	^false!

isMoveable
	^(state bitAnd: Moveable) == 0!

isOpaque
	^(state bitAnd: Filled) ~~ 0!

isRemoveable
	^(state bitAnd: Removeable) == 0!

isSelectable
	^(state bitAnd: Selectable) == 0!

isVisible
	^(state bitAnd: Invisible) == 0! !

!Figure methodsFor: 'transforming'!

align: alignmentPoint with: relativePoint 
	self translateBy: relativePoint - alignmentPoint!

basicTranslateBy: aPoint 
	"This method is called by #translateBy:. The #translateBy: method has already moved our
	bounds. If we have other points that need to be moved, we need to move them also."!

scaleBy: aPoint 
	| center |
	center := self center.
	self align: center with: (center scaledBy: aPoint)!

setBoundsTo: aRectangle 
	"We have moved to aRectangle. Update our bounds, and invalidate our old and new positions."

	| oldBounds |
	oldBounds := self preferredBounds.
	bounds := aRectangle.
	self changedPreferredBounds: oldBounds.
	self changed!

translateBy: aPoint 
	"Move ourself by aPoint. Instead of overriding this method, subclasses will probably just
	need to override basicTranslateBy:"

	| oldBounds |
	self isMoveable ifFalse: [^self].
	oldBounds := self preferredBounds.
	self basicTranslateBy: aPoint.
	self translatePreferredBoundsBy: aPoint.
	self changedPreferredBounds: oldBounds.
	self changed!

translatePreferredBoundsBy: aPoint 
	bounds := self preferredBounds translatedBy: aPoint!

translateTo: aPoint 
	"Move origin to aPoint. This used to move corner to aPoint, which seemed wierd."

	self translateBy: aPoint - self origin! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Figure class
	instanceVariableNames: ''!


!Figure class methodsFor: 'class initialization'!

initialize
	"Figure initialize"

	Selectable := 1.
	Moveable := 2.
	Invisible := 4.
	Removeable := 8.
	Filled := 16.
	Selected := 32.
	Connectable := 64! !

!Figure class methodsFor: 'instance creation'!

createAt: aPoint 
	^(self new)
		translateTo: aPoint;
		yourself! !

ApplicationModel subclass: #DrawingEditor
	instanceVariableNames: 'buttons drawing '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
DrawingEditor comment:
'DrawingEditor is the ApplicationModel for a Drawing.

Instance Variables:
	buttons	<SelectionInList of: ButtonDescription>	the selection in list of all our tools
	drawing	<Drawing>	our drawing

'!


!DrawingEditor methodsFor: 'accessing'!

drawing
	drawing isNil 
		ifTrue: 
			[drawing := self buildDrawing.
			self initializeDrawingController].
	^drawing!

initializeDrawingController
	drawing controller tool: (BlockValue block: [:selection | selection value]
				arguments: (Array with: self toolbarButtons selectionHolder))!

toolbar
	^ToolbarView model: self toolbarButtons! !

!DrawingEditor methodsFor: 'drawing description'!

activeToolNames
	"Return the tool names that should pass input down to their views. Override to include more/less tools."

	^#('Selection Tool')!

drawingClass
	"What type of drawing should we create?"

	^Drawing!

iconNames
	"If the icon names don't follow the convention of removing white space from the tool name, 
	making the first letter lowercase, and then replacing 'Tool' with 'Icon'; you might need to 
	override this.

	Normally the icon names can be converted from the tool names:
		Selection Tool -> selectionIcon
		Bring To Front Tool -> bringToFrontIcon"

	^self toolNames collect: [:each | self iconNameFor: each]!

toolNames
	"Return the list of names for the tools. 'nil' represents a space between tools in the icon bar."

	^#('Selection Tool' 
	'Hand Tool'
	nil 
	'Delete Tool' 
	'Bring To Front Tool' 
	'Send To Back Tool' 
	nil 
	'Polyline Tool' 
	'Bezier Tool' 
	'Spline Tool' 
	'Rectangle Tool' 
	'Rounded Rectangle Tool'
	'Ellipse Tool' 
	'Arc Tool'
	'Image Tool'
	'Text Figure Creation Tool')!

windowName
	"What should our window label be?"

	^'Drawing Editor'! !

!DrawingEditor methodsFor: 'interface opening'!

postBuildWith: aBuilder 
	super postBuildWith: aBuilder.
	(builder window)
		damageRepairPolicy: DoubleBufferingWindowDisplayPolicy new;
		label: self windowName! !

!DrawingEditor methodsFor: 'private'!

buildButtonDescriptionForTool: aString andIcon: iconSymbol 
	| tool icon |
	tool := Tool toolFor: aString.
	tool passInputDown: (self activeToolNames includes: aString).
	icon := (self class respondsTo: iconSymbol) 
				ifTrue: [self class perform: iconSymbol]
				ifFalse: [self class selectionIcon].
	^ButtonDescription icon: icon value: tool!

buildDrawing
	drawing := self drawingClass new.
	drawing model: self.
	^drawing!

drawing: aDrawing 
	drawing := aDrawing.
	self initializeDrawingController!

iconNameFor: aString 
	| iconName |
	aString isNil ifTrue: [^nil].
	iconName := aString select: [:each | each isAlphaNumeric].
	iconName := iconName copyFrom: 1 to: (iconName size - 4 max: 1).
	iconName at: 1 put: iconName first asLowercase.
	^(iconName , 'Icon') asSymbol!

toolbarButtons
	buttons isNil 
		ifTrue: 
			[| list |
			list := List new.
			self toolNames with: self iconNames
				do: 
					[:tool :icon | 
					list add: (tool isNil 
								ifTrue: [ButtonDescription spacer]
								ifFalse: [self buildButtonDescriptionForTool: tool andIcon: icon])].
			buttons := SelectionInList new.
			buttons list: list.
			buttons selection: list first].
	^buttons! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

DrawingEditor class
	instanceVariableNames: ''!


!DrawingEditor class methodsFor: 'examples'!

example1
	"Open a drawing editor with another drawing inside."

	"self example1"

	| editor |
	editor := self new.
	editor open.
	editor drawing add: (ViewAdapterFigure 
				view: (DrawingEditor new allButOpenInterface: #windowSpec) window component
				in: (10 @ 10 corner: 450 @ 350))!

example2
	"Open a drawing editor with a browser inside."

	"self example2"

	| editor |
	editor := self new.
	editor open.
	editor drawing add: (ViewAdapterFigure 
				view: (Browser new allButOpenInterface: #windowSpec) window component
				in: (10 @ 10 corner: 450 @ 350))! !

!DrawingEditor class methodsFor: 'interface opening'!

openOnDrawing: aDrawing
	| editor |
	editor := self new.
	editor drawing: aDrawing.
	editor open! !

!DrawingEditor class methodsFor: 'interface specs'!

windowSpec
	"UIPainter new openOnClass: self andSelector: #windowSpec"

	<resource: #canvas>
	^#(#FullSpec 
		#window: 
		#(#WindowSpec 
			#label: 'Drawing Editor' 
			#bounds: #(#Rectangle 133 548 688 1028 ) 
			#colors: 
			#(#LookPreferences 
				#setForegroundColor: nil 
				#setBackgroundColor: nil 
				#setSelectionForegroundColor: nil 
				#setSelectionBackgroundColor: nil 
				#setBorderColor: nil ) 
			#isEventDriven: true ) 
		#component: 
		#(#SpecCollection 
			#collection: #(
				#(#ArbitraryComponentSpec 
					#layout: #(#LayoutFrame 0 0 28 0 0 1 0 1 ) 
					#flags: 11 
					#component: #drawing ) 
				#(#ArbitraryComponentSpec 
					#layout: #(#LayoutFrame 0 0 0 0 0 1 28 0 ) 
					#flags: 0 
					#component: #toolbar ) ) ) )! !

!DrawingEditor class methodsFor: 'resources'!

arcIcon
	^OpaqueImage figure: self arcIconImage shape: self arcIconMask!

arcIconImage
	"UIMaskEditor new openOnClass: self andSelector: #arcIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0 248 126 0 0 231 158 0 0 223 190 0 0 223 126 0 0 190 254 0 0 190 254 0 0 223 126 0 0 223 190 0 0 231 158 0 0 248 126 0 0 255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0])!

arcIconMask
	"UIMaskEditor new openOnClass: self andSelector: #arcIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 7 128 0 0 31 224 0 0 63 192 0 0 63 128 0 0 127 0 0 0 127 0 0 0 63 128 0 0 63 192 0 0 31 224 0 0 7 128 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0])!

bezierIcon
	^OpaqueImage figure: self bezierIconImage shape: self bezierIconMask!

bezierIconImage
	"UIMaskEditor new openOnClass: self andSelector: #bezierIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 255 254 0 0 255 242 0 0 255 242 0 0 255 242 0 0 255 230 0 0 255 206 0 0 255 30 0 0 252 126 0 0 241 254 0 0 231 254 0 0 207 254 0 0 159 254 0 0 159 254 0 0 159 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0])!

bezierIconMask
	"UIMaskEditor new openOnClass: self andSelector: #bezierIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 0 0 0 0 12 0 0 0 12 0 0 0 12 0 0 0 24 0 0 0 48 0 0 0 224 0 0 3 128 0 0 14 0 0 0 24 0 0 0 48 0 0 0 96 0 0 0 96 0 0 0 96 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0])!

bringToFrontIcon
	^OpaqueImage figure: self bringToFrontIconImage shape: self bringToFrontIconMask!

bringToFrontIconImage
	"UIMaskEditor new openOnClass: self andSelector: #bringToFrontIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 128 126 0 0 191 126 0 0 187 126 0 0 179 126 0 0 160 62 0 0 128 30 0 0 160 14 0 0 179 6 0 0 187 70 0 0 191 102 0 0 191 102 0 0 191 102 0 0 191 78 0 0 191 62 0 0 128 126 0 0 255 254 0 0 255 254 0 0])!

bringToFrontIconMask
	"UIMaskEditor new openOnClass: self andSelector: #bringToFrontIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 127 128 0 0 127 128 0 0 127 128 0 0 127 128 0 0 127 192 0 0 127 224 0 0 127 240 0 0 127 248 0 0 127 184 0 0 127 152 0 0 127 152 0 0 127 152 0 0 127 176 0 0 127 192 0 0 127 128 0 0 0 0 0 0 0 0 0 0])!

deleteIcon
	^OpaqueImage figure: self deleteIconImage shape: self deleteIconMask!

deleteIconImage
	"UIMaskEditor new openOnClass: self andSelector: #deleteIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 255 190 0 0 255 94 0 0 254 238 0 0 253 246 0 0 251 250 0 0 247 242 0 0 239 234 0 0 223 218 0 0 191 182 0 0 127 110 0 0 62 222 0 0 93 190 0 0 107 126 0 0 182 254 0 0 213 254 0 0 227 254 0 0 247 254 0 0])!

deleteIconMask
	"UIMaskEditor new openOnClass: self andSelector: #deleteIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 64 0 0 0 224 0 0 1 240 0 0 3 248 0 0 7 252 0 0 15 252 0 0 31 252 0 0 63 252 0 0 127 248 0 0 255 240 0 0 255 224 0 0 255 192 0 0 255 128 0 0 127 0 0 0 62 0 0 0 28 0 0 0 8 0 0 0])!

ellipseIcon
	^OpaqueImage figure: self ellipseIconImage shape: self ellipseIconMask!

ellipseIconImage
	"UIMaskEditor new openOnClass: self andSelector: #ellipseIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0 248 126 0 0 231 158 0 0 223 238 0 0 223 238 0 0 191 246 0 0 191 246 0 0 223 238 0 0 223 238 0 0 231 158 0 0 248 126 0 0 255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0])!

ellipseIconMask
	"UIMaskEditor new openOnClass: self andSelector: #ellipseIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 7 128 0 0 31 224 0 0 63 240 0 0 63 240 0 0 127 248 0 0 127 248 0 0 63 240 0 0 63 240 0 0 31 224 0 0 7 128 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0])!

handIcon
	^OpaqueImage figure: self handIconImage shape: self handIconMask!

handIconImage
	"UIMaskEditor new openOnClass: self andSelector: #handIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 16@16 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[254 127 0 0 229 143 0 0 217 183 0 0 217 181 0 0 237 178 0 0 237 182 0 0 151 246 0 0 103 254 0 0 119 253 0 0 191 253 0 0 223 253 0 0 239 251 0 0 239 251 0 0 247 247 0 0 251 247 0 0 248 7 0 0])!

handIconMask
	"UIMaskEditor new openOnClass: self andSelector: #handIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 16@16 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[1 128 0 0 27 240 0 0 63 248 0 0 63 250 0 0 31 255 0 0 31 255 0 0 111 255 0 0 255 255 0 0 255 254 0 0 127 254 0 0 63 254 0 0 31 252 0 0 31 252 0 0 15 248 0 0 7 248 0 0 7 248 0 0])!

imageIcon
	^OpaqueImage figure: self imageIconImage shape: self imageIconMask!

imageIconImage
	"UIMaskEditor new openOnClass: self andSelector: #imageIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0 143 226 0 0 191 250 0 0 191 250 0 0 255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0 191 250 0 0 191 250 0 0 143 226 0 0 255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0])!

imageIconMask
	"UIMaskEditor new openOnClass: self andSelector: #imageIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 112 28 0 0 64 4 0 0 64 4 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 64 4 0 0 64 4 0 0 112 28 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0])!

polylineIcon
	^OpaqueImage figure: self polylineIconImage shape: self polylineIconMask!

polylineIconImage
	"UIMaskEditor new openOnClass: self andSelector: #polylineIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 255 126 0 0 254 62 0 0 252 158 0 0 249 206 0 0 243 230 0 0 231 206 0 0 207 158 0 0 159 62 0 0 199 126 0 0 241 254 0 0 252 126 0 0 255 30 0 0 255 198 0 0 255 206 0 0 255 158 0 0 255 62 0 0 255 254 0 0])!

polylineIconMask
	"UIMaskEditor new openOnClass: self andSelector: #polylineIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 128 0 0 1 192 0 0 3 96 0 0 6 48 0 0 12 24 0 0 24 48 0 0 48 96 0 0 96 192 0 0 56 128 0 0 14 0 0 0 3 128 0 0 0 224 0 0 0 56 0 0 0 48 0 0 0 96 0 0 0 192 0 0 0 0 0 0])!

rectangleIcon
	^OpaqueImage figure: self rectangleIconImage shape: self rectangleIconMask!

rectangleIconImage
	"UIMaskEditor new openOnClass: self andSelector: #rectangleIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0 128 2 0 0 191 250 0 0 191 250 0 0 191 250 0 0 191 250 0 0 191 250 0 0 191 250 0 0 191 250 0 0 191 250 0 0 128 2 0 0 255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0])!

rectangleIconMask
	"UIMaskEditor new openOnClass: self andSelector: #rectangleIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 252 0 0 127 252 0 0 127 252 0 0 127 252 0 0 127 252 0 0 127 252 0 0 127 252 0 0 127 252 0 0 127 252 0 0 127 252 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0])!

roundedRectangleIcon
	^OpaqueImage figure: self roundedRectangleIconImage shape: self roundedRectangleIconMask!

roundedRectangleIconImage
	"UIMaskEditor new openOnClass: self andSelector: #roundedRectangleIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0 240 30 0 0 207 230 0 0 223 246 0 0 191 250 0 0 191 250 0 0 191 250 0 0 191 250 0 0 223 246 0 0 207 230 0 0 240 30 0 0 255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0])!

roundedRectangleIconMask
	"UIMaskEditor new openOnClass: self andSelector: #roundedRectangleIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 15 224 0 0 63 248 0 0 63 248 0 0 127 252 0 0 127 252 0 0 127 252 0 0 127 252 0 0 63 248 0 0 63 248 0 0 15 224 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0])!

selectionIcon
	^OpaqueImage figure: self selectionIconImage shape: self selectionIconMask!

selectionIconImage
	"UIMaskEditor new openOnClass: self andSelector: #selectionIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 239 254 0 0 231 254 0 0 235 254 0 0 237 254 0 0 238 254 0 0 239 126 0 0 239 190 0 0 239 222 0 0 239 14 0 0 237 126 0 0 233 190 0 0 230 190 0 0 238 222 0 0 255 94 0 0 255 62 0 0 255 254 0 0 255 254 0 0])!

selectionIconMask
	"UIMaskEditor new openOnClass: self andSelector: #selectionIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 16 0 0 0 24 0 0 0 28 0 0 0 30 0 0 0 31 0 0 0 31 128 0 0 31 192 0 0 31 224 0 0 31 240 0 0 31 128 0 0 31 192 0 0 25 192 0 0 17 224 0 0 0 224 0 0 0 192 0 0 0 0 0 0 0 0 0 0])!

sendToBackIcon
	^OpaqueImage figure: self sendToBackIconImage shape: self sendToBackIconMask!

sendToBackIconImage
	"UIMaskEditor new openOnClass: self andSelector: #sendToBackImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 128 126 0 0 191 126 0 0 184 62 0 0 184 14 0 0 184 6 0 0 191 98 0 0 191 112 0 0 191 120 0 0 191 104 0 0 191 72 0 0 191 0 0 0 191 2 0 0 191 78 0 0 191 110 0 0 128 126 0 0 255 254 0 0 255 254 0 0])!

sendToBackIconMask
	"UIMaskEditor new openOnClass: self andSelector: #sendToBackIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 127 128 0 0 127 128 0 0 127 192 0 0 127 240 0 0 127 248 0 0 127 156 0 0 127 142 0 0 127 134 0 0 127 150 0 0 127 182 0 0 127 254 0 0 127 252 0 0 127 176 0 0 127 144 0 0 127 128 0 0 0 0 0 0 0 0 0 0])!

splineIcon
	^OpaqueImage figure: self splineIconImage shape: self splineIconMask!

splineIconImage
	"UIMaskEditor new openOnClass: self andSelector: #splineIconImage"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 254 0 0 255 254 0 0 255 254 0 0 255 254 0 0 193 254 0 0 128 254 0 0 28 118 0 0 62 102 0 0 62 78 0 0 62 30 0 0 28 62 0 0 128 254 0 0 193 254 0 0 243 254 0 0 231 254 0 0 207 254 0 0 159 254 0 0 255 254 0 0])!

splineIconMask
	"UIMaskEditor new openOnClass: self andSelector: #splineIconMask"

	<resource: #image>
	^CachedImage on: (Image extent: 15@18 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 62 0 0 0 127 0 0 0 227 136 0 0 193 152 0 0 193 176 0 0 193 224 0 0 227 192 0 0 127 0 0 0 62 0 0 0 12 0 0 0 24 0 0 0 48 0 0 0 96 0 0 0 0 0 0 0])!

textFigureCreationIcon
	^OpaqueImage figure: self textFigureCreationImage
		shape: self textFigureCreationMask!

textFigureCreationImage
	"UIMaskEditor new openOnClass: self andSelector: #textFigureCreationImage"

	<resource: #image>
	^CachedImage on: (Image extent: 16@16 depth: 1 bitsPerPixel: 1 palette: MonoMappedPalette blackWhite usingBits: #[255 255 0 0 255 255 0 0 255 255 0 0 192 3 0 0 192 3 0 0 222 123 0 0 254 127 0 0 254 127 0 0 254 127 0 0 254 127 0 0 254 127 0 0 254 127 0 0 254 127 0 0 254 123 0 0 252 49 0 0 255 238 0 0])!

textFigureCreationMask
	"UIMaskEditor new openOnClass: self andSelector: #textFigureCreationMask"

	<resource: #image>
	^CachedImage on: (Image extent: 16@16 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 0 0 0 0 0 0 0 63 252 0 0 63 252 0 0 33 132 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 132 0 0 3 206 0 0 0 17 0 0])!

textIcon
	^OpaqueImage figure: self textFigureCreationImage shape: self textMask!

textMask
	"UIMaskEditor new openOnClass: self andSelector: #textMask"

	<resource: #image>
	^CachedImage on: (Image extent: 16@16 depth: 1 bitsPerPixel: 1 palette: CoveragePalette monoMaskPalette usingBits: #[0 0 0 0 0 0 0 0 0 0 0 0 63 252 0 0 63 252 0 0 33 132 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 128 0 0 1 128 0 0 3 192 0 0 0 0 0 0])! !

Figure subclass: #CompositeFigure
	instanceVariableNames: 'components '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
CompositeFigure comment:
'CompositeFigure is a Figure that groups other figures into a single figure (composite pattern).

Instance Variables:
	components	<SequenceableCollection of: Figure>	the figures I''m grouping

'!


!CompositeFigure methodsFor: 'accessing'!

components
	^components! !

!CompositeFigure methodsFor: 'adding'!

add: aFigure 
	| figureBounds |
	components add: aFigure.
	aFigure container: self.
	figureBounds := aFigure bounds.
	self invalidateRectangle: figureBounds.
	self mergeBounds: figureBounds.
	^aFigure!

addAll: aCollection 
	aCollection do: [:each | self add: each].
	^aCollection!

addLast: aFigure 
	components addFirst: aFigure.
	aFigure container: self.
	self invalidateRectangle: aFigure bounds.
	self mergeBounds: aFigure bounds.
	^aFigure! !

!CompositeFigure methodsFor: 'broadcast'!

downcastEvent: aKey with: aParameter from: anInitiator 
	"This is a message passed down the view structure to all subparts."

	super downcastEvent: aKey
		with: aParameter
		from: anInitiator.
	components do: 
			[:each | 
			each downcastEvent: aKey
				with: aParameter
				from: anInitiator]!

downcastLocalEvent: aKey with: aParameter at: aPoint from: anInitiator 
	"This is a message passed down the view structure to some single 
	part. Answer true if we accepted the event, or false if it should be 
	passed on to whatever's behind us."

	components size to: 1
		by: -1
		do: 
			[:i | 
			((components at: i) downcastLocalEvent: aKey
				with: aParameter
				at: aPoint
				from: anInitiator) ifTrue: [^true]].
	^super downcastLocalEvent: aKey
		with: aParameter
		at: aPoint
		from: anInitiator!

flushCoordinateCaches
	"Flush caches that relate to coordinate translations between this component 
	and its container"

	super flushCoordinateCaches.
	components do: [:each | each flushCoordinateCaches]!

newGraphicsDevice: aGraphicsDevice 
	"Forward the newGraphicsDevice: aGraphicsDevice messages
	to the receiver's components."

	super newGraphicsDevice: aGraphicsDevice.
	components do: [:component | component newGraphicsDevice: aGraphicsDevice]! !

!CompositeFigure methodsFor: 'control'!

componentWantingControl
	"Answer the first component that wishes to take control.  Answer nil if there is
	no component wanting control."

	components size to: 1
		by: -1
		do: 
			[:i | 
			| obj |
			obj := (components at: i) objectWantingControl.
			obj notNil ifTrue: [^obj]].
	^nil! !

!CompositeFigure methodsFor: 'copying'!

postCopy
	super postCopy.
	components := components collect: [:each | each copy]! !

!CompositeFigure methodsFor: 'displaying'!

displayFigureOn: aGraphicsContext 
	| rect |
	rect := (aGraphicsContext clippingBounds intersect: self preferredBounds) 
				expandedBy: 1 @ 1.
	aGraphicsContext clippingRectangle: rect.
	components 
		do: [:each | (each intersects: rect) ifTrue: [each displayOn: aGraphicsContext copy]]! !

!CompositeFigure methodsFor: 'initialize-release'!

flushCaches
	super flushCaches.
	components do: [:each | each flushCaches]!

initialize
	super initialize.
	components := OrderedCollection new!

release
	components do: [:each | each release].
	super release! !

!CompositeFigure methodsFor: 'private'!

compositionBoundsFor: aComponent 
	^aComponent preferredBounds!

computePreferredBounds
	| rect |
	components isEmpty ifTrue: [^0 @ 0 extent: 0 @ 0].
	rect := components first bounds.
	2 to: components size
		do: [:i | rect := rect merge: (components at: i) preferredBounds].
	^rect!

mergeBounds: aRectangle 
	bounds isNil ifTrue: [^self].
	bounds := bounds merge: aRectangle! !

!CompositeFigure methodsFor: 'removing'!

remove: aFigure 
	| rect |
	rect := aFigure bounds.
	aFigure container: nil.
	components remove: aFigure ifAbsent: [].
	self invalidateRectangle: rect.
	^aFigure!

removeAll: aCollection 
	aCollection do: [:each | self remove: each].
	^aCollection! !

!CompositeFigure methodsFor: 'sorting'!

bringToFront: aFigure 
	| index |
	index := components indexOf: aFigure ifAbsent: [^self].
	index to: components size - 1
		do: [:i | components at: i put: (components at: i + 1)].
	components at: components size put: aFigure.
	self invalidateRectangle: aFigure bounds!

sendToBack: aFigure 
	| index |
	index := components indexOf: aFigure ifAbsent: [^self].
	index to: 2
		by: -1
		do: [:i | components at: i put: (components at: i - 1)].
	components at: 1 put: aFigure.
	self invalidateRectangle: aFigure bounds! !

!CompositeFigure methodsFor: 'testing'!

containsPoint: aPoint 
	^components contains: [:each | each containsPoint: aPoint]!

isComposite
	^true! !

!CompositeFigure methodsFor: 'transforming'!

basicTranslateBy: aPoint 
	components do: [:each | each translateBy: aPoint]! !

CompositeFigure subclass: #Drawing
	instanceVariableNames: 'selections handles controller '
	classVariableNames: 'CopiedFigures '
	poolDictionaries: ''
	category: 'HotDraw-Framework'!
Drawing comment:
'Drawing is a view that allows the user to interact with the figure. In addition to the composite figure behavior, it cooperates with a controller to allow the user to manipulate the figures.

Instance Variables:
	controller	<DrawingController>	my controller
	handles	<SequenceableCollection of: Handle>	handles that the user can interact with, only selections will have their handles displayed
	selections	<Collection of: Figure>	the figures that are currently selected by the user. Selected figures may be treated differently by different tools

Class Variables:
	CopiedFigures	<Collection of: Figure>	a collection of figures that were copied or cut from a drawing (for cut/paste operations)

'!


!Drawing methodsFor: 'accessing'!

drawing
	^self!

edit
	^DrawingEditor openOnDrawing: self!

figureAt: aPoint 
	"Drawings contain all points, if no figures contain this point, return self"

	| figure |
	handles size to: 1
		by: -1
		do: 
			[:i | 
			figure := (handles at: i) figureAt: aPoint.
			figure notNil ifTrue: [^figure]].
	components size to: 1
		by: -1
		do: 
			[:i | 
			figure := (components at: i) figureAt: aPoint.
			figure notNil ifTrue: [^figure]].
	^self!

figures
	^self components!

figuresIn: aRectangle 
	^components select: [:each | aRectangle contains: each]!

menuAt: aPoint 
	| mb |
	mb := MenuBuilder new.
	mb
		addCopyCutPaste;
		line;
		add: 'save...' -> #saveDrawing;
		add: 'load...' -> #loadDrawing;
		line;
		add: 'print' -> #print;
		add: 'save as postscript...' -> #savePostscript;
		line;
		add: 'inspect...' -> #inspect;
		line;
		add: 'group' -> #group;
		add: 'ungroup' -> #ungroup;
		line.
	self addLineColorMenuTo: mb.
	self addFillColorMenuTo: mb.
	self addLineWidthMenuTo: mb.
	self addModelMenuTo: mb at: aPoint.
	^mb menu! !

!Drawing methodsFor: 'broadcast'!

downcastEvent: aKey with: aParameter from: anInitiator 
	"This is a message passed down the view structure to all subparts."

	super 
		downcastEvent: aKey
		with: aParameter
		from: anInitiator.
	handles do: 
			[:each | 
			each 
				downcastEvent: aKey
				with: aParameter
				from: anInitiator]!

downcastLocalEvent: aKey with: aParameter at: aPoint from: anInitiator 
	"This is a message passed down the view structure to some single 
	part. Answer true if we accepted the event, or false if it should be 
	passed on to whatever's behind us."

	handles size to: 1
		by: -1
		do: 
			[:i | 
			((handles at: i) 
				downcastLocalEvent: aKey
				with: aParameter
				at: aPoint
				from: anInitiator) ifTrue: [^true]].
	^super 
		downcastLocalEvent: aKey
		with: aParameter
		at: aPoint
		from: anInitiator!

flushCoordinateCaches
	"Flush caches that relate to coordinate translations between this component 
	and its container"

	super flushCoordinateCaches.
	handles do: [:each | each flushCoordinateCaches]!

newGraphicsDevice: aGraphicsDevice 
	"Forward the newGraphicsDevice: aGraphicsDevice messages
	to the receiver's components."

	super newGraphicsDevice: aGraphicsDevice.
	handles do: [:component | component newGraphicsDevice: aGraphicsDevice]! !

!Drawing methodsFor: 'control'!

componentWantingControl
	"Answer the first component that wishes to take control.  Answer nil if there is
	no component wanting control."

	| obj |
	handles size to: 1
		by: -1
		do: 
			[:i | 
			obj := (handles at: i) objectWantingControl.
			obj notNil ifTrue: [^obj]].
	^super componentWantingControl!

objectWantingControl
	"The receiver is in a control hierarchy and the container is asking
	for an object that wants control.  If no control is desired then the receiver
	answers nil.  If control is wanted then the receiver answers the
	control object."

	| obj |
	obj := super objectWantingControl.
	obj notNil ifTrue: [^obj].
	^self componentWantingControl! !

!Drawing methodsFor: 'controller accessing'!

controller
	"Answer the receiver`s current controller. If the receiver's controller 
	is nil (the default case), an initialized instance of the receiver's 
	default controller is installed and returned."

	^self getController!

controller: aController 
	"Set the receiver's controller to aController.  An instance of NoController can be
	specified to indicate that the receiver will not have a controller.  The model of
	aController is set to the receiver's model."

	self setController: aController!

defaultController
	"Answer an initialized instance of the receiver's default controller.
	Subclasses should redefine this message only if the default controller 
	instances need to be initialized in a nonstandard way."

	^self defaultControllerClass new!

defaultControllerClass
	^DrawingController! !

!Drawing methodsFor: 'copying'!

postCopy
	super postCopy.
	selections := nil.
	handles := #().
	controller := nil! !

!Drawing methodsFor: 'displaying'!

displayOn: aGraphicsContext 
	| clipRect |
	super displayOn: aGraphicsContext.
	clipRect := aGraphicsContext clippingBounds.
	handles do: 
			[:each | 
			(each intersects: clipRect)
				ifTrue: [each displayOn: aGraphicsContext copy]]!

invalidateRectangle: aRectangle repairNow: aBoolean 
	"Hacked to avoid some off by one errors. Different OS's have different definitions of how to draw 
	rectangles. Instead of trying to do something smart, just expand the rectangle..."

	^super invalidateRectangle: (aRectangle expandedBy: 1 @ 1) repairNow: aBoolean!

invalidateRectangle: aRectangle repairNow: aBoolean forComponent: aFigure 
	"Hacked to avoid some off by one errors. Different OS's have different definitions of how to draw 
	rectangles. Instead of trying to do something smart, just expand the rectangle..."

	^super 
		invalidateRectangle: (aRectangle expandedBy: 1 @ 1)
		repairNow: aBoolean
		forComponent: aFigure! !

!Drawing methodsFor: 'event driven'!

handlerForMouseEvent: aMouseEvent 
	"The receiver is in a control hierarchy and the container is asking 
	for an object that wants control. If no control is desired then the 
	receiver answers nil. If control is wanted then the receiver answers the 
	control object."

	self controller currentTool passInputDown 
		ifTrue: 
			[| obj |
			components size to: 1
				by: -1
				do: 
					[:i | 
					obj := (components at: i) handlerForMouseEvent: aMouseEvent.
					obj notNil ifTrue: [^obj]].
			^super handlerForMouseEvent: aMouseEvent]
		ifFalse: [^self]!

updateSpot: type 
	"Forward this message to all components."

	components do: [:cmp | cmp updateSpot: type]! !

!Drawing methodsFor: 'fileIn/Out'!

representBinaryOn: binWriter 
	"Don't try to store our whole representation (just store our figures)."

	^MessageSend 
		receiver: self class
		selector: #figures:
		argument: components! !

!Drawing methodsFor: 'initialize-release'!

initialize
	super initialize.
	selections := OrderedCollection new.
	handles := OrderedCollection new.
	self
		lineWidth: 1;	"Set the default properties of figures, someone must do this..."
		lineColor: ColorValue black;
		fillColor: ColorValue white;
		selectable: false;
		moveable: false;
		removeable: false;
		connectable: false!

release
	Cursor normal show.
	super release! !

!Drawing methodsFor: 'menu actions'!

copySelection
	CopiedFigures := self selections collect: [:each | each copy]!

cut
	CopiedFigures := self selections.
	self removeAll: self selections!

group
	| figures newFigure |
	figures := selections copy.
	figures isEmpty ifTrue: [^self].
	self removeAll: figures.
	newFigure := CompositeFigure new.
	newFigure addAll: figures.
	self
		add: newFigure;
		selection: newFigure!

loadDrawing
	| name boss drawing |
	name := Dialog requestFileName: 'Load drawing from:'.
	name isEmpty ifTrue: [^self].
	name asFilename exists ifFalse: [^Dialog warn: 'File does not exist'].
	boss := BinaryObjectStorage onOld: name asFilename readStream.
	drawing := Cursor read 
				showWhile: [[boss next] valueNowOrOnUnwindDo: [boss close]].
	drawing
		container: container;
		model: model;
		privateSetController: controller.
	self become: drawing.
	self invalidate!

paste
	CopiedFigures isNil ifTrue: [^self].
	self addAll: (CopiedFigures collect: [:each | each copy])!

print
	| ps parent scale |
	parent := container.
	bounds := self computePreferredBounds expandedBy: 10 @ 10.
	ps := Printer startPrintJobNamed: 'HotDraw.postscript'.
	ps setLandscape: self extent x > self extent y.
	scale := (ps extent x / bounds extent x min: ps extent y / bounds extent y) 
				min: 1.
	
	[(MockMedium 
		on: ((ScalingWrapper on: self) scale: scale)
		in: ps bounds
		for: ps)
		newGraphicsDevice: ps graphicsDevice;
		display] 
			valueNowOrOnUnwindDo: 
				[container := parent.
				self flushCaches.
				ps print]!

saveDrawing
	| name boss oldSelections |
	name := Dialog requestFileName: 'Save drawing to:'.
	name isEmpty ifTrue: [^self].
	oldSelections := self selections.
	self
		clearSelections;
		flushCaches.
	boss := BinaryObjectStorage onNew: name asFilename writeStream.
	Cursor write showWhile: 
			[[boss nextPut: self] valueNowOrOnUnwindDo: 
					[boss close.
					self selections: oldSelections]]!

savePostscript
	| scale filename file gc bounds |
	filename := Dialog request: 'Enter filename for output:'
				initialAnswer: 'drawing.ps'.
	filename isEmpty ifTrue: [^self].
	bounds := self computePreferredBounds expandedBy: 10 @ 10.
	file := PostScriptFile named: filename.
	file setLandscape: bounds extent x > bounds extent y.
	gc := file graphicsContext.
	scale := gc clippingBounds extent x / bounds extent x 
				min: gc clippingBounds extent y / bounds extent y.
	gc scaleBy: (scale min: 1).
	self displayOn: gc.
	file close!

ungroup
	| figures compositeFigures |
	compositeFigures := selections select: [:each | each isComposite].
	compositeFigures isEmpty ifTrue: [^self].
	figures := OrderedCollection new.
	compositeFigures do: [:each | figures addAll: each components].
	self
		removeAll: compositeFigures;
		addAll: figures;
		selections: figures! !

!Drawing methodsFor: 'private'!

changedPreferredBounds: oldPreferredBounds forComponent: aVisualComponent 
	| newBounds newRect |
	newBounds := aVisualComponent bounds.
	self updatePreferredBoundsWith: newBounds.
	newRect := oldPreferredBounds isNil
				ifTrue: [newBounds]
				ifFalse: [oldPreferredBounds merge: newBounds].
	newRect origin: newRect origin - 1.
	newRect corner: newRect corner + 1.
	self invalidateRectangle: newRect repairNow: false!

getController
	"Answer the receiver`s current controller. If the receiver's controller 
	is nil (the default case), an initialized instance of the receiver's 
	default controller is installed and returned."

	controller isNil ifTrue: [self setController: self defaultController].
	^controller!

privateSetController: aController
	controller := aController!

setComponents: aCollection
	components := aCollection.
	bounds := nil!

setController: aController 
	"Set the receiver's controller to aController. If aController is not nil, its view is set to the receiver 
	and its model is set to the receiver's model."

	aController notNil 
		ifTrue: 
			[aController view: self.
			aController model: model].
	controller := aController!

updatePreferredBoundsWith: aRectangle 
	| oldBounds |
	oldBounds := bounds.
	bounds := bounds isNil 
				ifTrue: [aRectangle]
				ifFalse: [bounds merge: aRectangle].
	oldBounds = bounds ifFalse: [self changedPreferredBounds: oldBounds].
	self invalidateRectangle: (aRectangle expandedBy: 1 @ 1) repairNow: false! !

!Drawing methodsFor: 'removing'!

remove: aFigure 
	aFigure isRemoveable ifFalse: [^self].
	self deselectFigure: aFigure.
	super remove: aFigure.
	^aFigure! !

!Drawing methodsFor: 'selection'!

clearSelections
	selections do: [:each | each deselect].
	selections := OrderedCollection new.
	handles do: 
			[:each | 
			each
				invalidate;
				container: nil].
	handles := OrderedCollection new!

deselectFigure: aFigure 
	selections remove: aFigure ifAbsent: [^self].
	aFigure deselect.
	handles := handles reject: 
					[:each | 
					| value |
					(value := each owner == aFigure) 
						ifTrue: 
							[each
								invalidate;
								container: nil].
					value]!

isSelected: aFigure 
	^aFigure isSelected!

selectFigure: aFigure 
	| newHandles |
	aFigure isSelectable ifFalse: [^self].
	aFigure select.
	selections add: aFigure.
	newHandles := aFigure handles.
	handles addAll: newHandles.
	newHandles do: 
			[:each | 
			self mergeBounds: each bounds.
			each
				container: self;
				invalidate]!

selection
	^selections isEmpty 
		ifTrue: [nil] 
		ifFalse: [selections first]!

selection: aFigure 
	self clearSelections.
	self selectFigure: aFigure!

selections
	^selections!

selections: aCollection 
	self clearSelections.
	aCollection do: [:each | self selectFigure: each]!

toggleSelection: aFigure 
	(self isSelected: aFigure) 
		ifTrue: [self deselectFigure: aFigure]
		ifFalse: [self selectFigure: aFigure]!

toggleSelections: aCollection 
	aCollection do: [:each | self toggleSelection: each]! !

!Drawing methodsFor: 'testing'!

containsPoint: aPoint 
	"Drawings contain everything"

	^true!

isDrawing
	^true! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

Drawing class
	instanceVariableNames: ''!


!Drawing class methodsFor: 'accessing'!

clearCutBuffer
	CopiedFigures := nil! !

!Drawing class methodsFor: 'private'!

figures: aCollection 
	^(self new)
		setComponents: aCollection;
		yourself! !
Figure initialize!


