Object subclass: #BoundaryConstraint
	instanceVariableNames: 'line isStart filledFigure isUpdating '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Constraints'!
BoundaryConstraint comment:
'BoundaryConstraint represents a constraint that is placed between a filled figure and a line figure. It computes the lines position so that it is attached to the edge of the filled figure. This constraint only works if the filled figure answers true when sent the containsPoint: message with the filled figures center.

Instance Variables:
	filledFigure	<Figure>	the filled figure that answers true to "filledFigure containsPoint: filledFigure center"
	isStart	<Boolean>	setting the start or stop point of the line
	isUpdating	<Boolean>	are we currently setting the lines values (avoids infinite recursion)
	line	<PolylineFigure>	the line that is attached'!


!BoundaryConstraint methodsFor: 'initialize-release'!

figure: aFigure line: aLineFigure 
	filledFigure := aFigure.
	line := aLineFigure.
	isUpdating := false.
	filledFigure addDependent: self.
	line addDependent: self!

isStart: aBoolean 
	isStart := aBoolean!

release
	filledFigure removeDependent: self.
	line removeDependent: self.
	super release! !

!BoundaryConstraint methodsFor: 'private'!

currentPoint
	^isStart 
		ifTrue: [line startPoint] 
		ifFalse: [line stopPoint]!

findPointOnBoundaryBetweenInside: aPoint andOutside: anotherPoint 
	| newPoint insidePoint outsidePoint |
	insidePoint := aPoint.
	outsidePoint := anotherPoint.
	[outsidePoint between: insidePoint - 1 and: insidePoint + 1] whileFalse: 
			[newPoint := (insidePoint + outsidePoint) / 2.0.
			(filledFigure containsPoint: newPoint) 
				ifTrue: [insidePoint := newPoint]
				ifFalse: [outsidePoint := newPoint]].
	^outsidePoint!

movePointTo: aPoint 
	isStart 
		ifTrue: [line startPoint: aPoint] 
		ifFalse: [line stopPoint: aPoint]!

previousPoint
	^isStart
		ifTrue: [line pointAt: 2]
		ifFalse: [line pointAt: line pointsSize - 1]! !

!BoundaryConstraint methodsFor: 'updating'!

update: anAspectSymbol with: aParameter from: aSender 
	anAspectSymbol == #deleted ifTrue: [^self release].
	aSender == line ifTrue: [^self updateFromLineWith: anAspectSymbol].
	aSender == filledFigure ifTrue: [^self updateLinePoint].
	super 
		update: anAspectSymbol
		with: aParameter
		from: aSender!

updateFromLineWith: anAspectSymbol 
	isUpdating ifTrue: [^self].
	isStart 
		ifTrue: [anAspectSymbol == #stopPoint ifTrue: [^self]]
		ifFalse: [anAspectSymbol == #startPoint ifTrue: [^self]].
	self updateLinePoint!

updateLinePoint
	isUpdating := true.
	
	[| current previous direction inside outside |
	current := self currentPoint.
	previous := self previousPoint.
	(filledFigure containsPoint: current) 
		ifTrue: 
			[(filledFigure containsPoint: previous) 
				ifTrue: 
					[direction := current - previous.
					direction := direction = (0 @ 0) 
								ifTrue: [1 @ 0]
								ifFalse: [direction unitVector].
					outside := previous + (direction * (filledFigure extent r + 1)).
					inside := previous]
				ifFalse: 
					[inside := current.
					outside := previous]]
		ifFalse: 
			[inside := filledFigure center.
			outside := current].
	self movePointTo: (self findPointOnBoundaryBetweenInside: inside
				andOutside: outside)] 
			valueNowOrOnUnwindDo: [isUpdating := false]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

BoundaryConstraint class
	instanceVariableNames: ''!


!BoundaryConstraint class methodsFor: 'instance creation'!

forFigure: aFigure startLine: aLineFigure 
	^(self new) figure: aFigure line: aLineFigure;
		isStart: true;
		updateLinePoint;
		yourself!

forFigure: aFigure stopLine: aLineFigure 
	^(self new) figure: aFigure line: aLineFigure;
		isStart: false;
		updateLinePoint;
		yourself! !

Object subclass: #PositionConstraint
	instanceVariableNames: 'receiver position message isUpdating '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'HotDraw-Constraints'!
PositionConstraint comment:
'PositionConstraint represents a constraint between a position (a message send) and a message (symbol) of a receiver (figure). Whenever either figure changes, the constraint is re-evaluated, and the receiver is moved to the position.

Instance Variables:
	isUpdating	<Boolean>	are we currently updating the figure (avoid infinite recursion)
	message	<Symbol>	the symbol that is performed on receiver to set its position
	position	<MessageSend>	a message send that when evaluated, returns the new position for receiver
	receiver	<Figure>	the figure to be constrained

'!


!PositionConstraint methodsFor: 'initialize-release'!

location: aMessageSend receiver: anObject sending: aSelector 
	position := aMessageSend.
	(position receiver isKindOf: Figure)
		ifTrue: [position receiver addDependent: self].
	receiver := anObject.
	receiver addDependent: self.
	message := aSelector.
	isUpdating := false!

release
	position receiver removeDependent: self.
	receiver removeDependent: self.
	super release! !

!PositionConstraint methodsFor: 'updating'!

evaluate
	message last == $:
		ifTrue: [receiver perform: message with: position value]
		ifFalse: [receiver translateBy: position value - (receiver perform: message)]!

update: anAspectSymbol with: aParameter from: aSender 
	"If one of our figures was removed from the drawing, disconnect ourself from the figures."

	anAspectSymbol == #deleted ifTrue: [^self release].
	aSender == receiver 
		ifTrue: 
			[isUpdating 
				ifFalse: 
					[isUpdating := true.
					[self evaluate] valueNowOrOnUnwindDo: [isUpdating := false]]]
		ifFalse: [self evaluate]! !
"-- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- "!

PositionConstraint class
	instanceVariableNames: ''!


!PositionConstraint class methodsFor: 'instance creation'!

send: aSelector to: anObject with: aMessageSend 
	^(self new) location: aMessageSend
			receiver: anObject
			sending: aSelector;
		evaluate;
		yourself! !

