"	This Squeak format file is an implementation of the exception handling system (EHS) based on a minimal reading of the standard.
	  The EHS can be downloaded from the following web site:  The Fourth Estate, Inc. web address is http://www.4thEstate.com.  Send email to squeak@4thEstate.com.  Originally named TFEI-Exceptions-core.cs.

 ******
 * $Name:  $
 * $Id: ExcpCore.cs 1.1 1999-10-12 07:47:09-05 Harmon Alpha $
 * $Log: ExcpCore.cs $
 * Revision 1.1  1999-10-12 07:47:09-05  Harmon
 * Initial revision
 *
 ******"!

'From Squeak 2.3 of January 14, 1999 on 13 June 1999 at 2:40:28 am'!
Object subclass: #Exception
	instanceVariableNames: 'messageText initialContext resignalException handlerContext tag '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!
Exception subclass: #BlockCannotReturn
	instanceVariableNames: 'result '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!
Exception subclass: #Error
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!
Object subclass: #ExceptionSet
	instanceVariableNames: 'exceptions '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!
Exception subclass: #IllegalResumeAttempt
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!
Error subclass: #MessageNotUnderstood
	instanceVariableNames: 'message '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!
Exception subclass: #Notification
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!
Notification subclass: #ExceptionAboutToReturn
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!
Notification subclass: #Warning
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!
Error subclass: #ZeroDivide
	instanceVariableNames: 'dividend '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'TFEI-Exceptions'!

!ContextPart methodsFor: 'controlling' stamp: 'crl 2/26/1999 15:34'!
terminate
	"Make myself unresumable."

	sender _ nil! !

!ContextPart methodsFor: 'controlling' stamp: 'crl 2/26/1999 15:34'!
terminateTo: previousContext
	"Terminate all the Contexts between me and previousContext, if previousContext is on my Context stack. Make previousContext my sender."

	| currentContext sendingContext |

	(self hasSender: previousContext) ifTrue: [
		currentContext _ sender.
		[currentContext == previousContext] whileFalse: [
			sendingContext _ currentContext sender.
			currentContext terminate.
			currentContext _ sendingContext]].
	sender _ previousContext! !

!ContextPart methodsFor: 'private-exceptions' stamp: 'tfei 4/1/1999 18:02'!
isHandlerContext

	^self method == (BlockContext compiledMethodAt: #on:do:)! !

!ContextPart methodsFor: 'private-exceptions' stamp: 'tfei 4/1/1999 18:17'!
isUnwindContext

	| m |
	^(m := self method) == (BlockContext compiledMethodAt: #ensure:)
		or: [m == (BlockContext compiledMethodAt: #ifCurtailed:)]! !

!ContextPart methodsFor: 'private-exceptions' stamp: 'tfei 4/13/1999 19:02'!
unwindTo: aContext

	| ctx returnValue aContextSender unwindBlock |
	ctx := self.
	returnValue := nil.
	aContext == nil
		ifTrue: [aContextSender := nil]
		ifFalse: [aContextSender := aContext sender]. "if aContext itself is marked for unwind, then need to use sender for whileFalse: loop check"
	[ctx == aContextSender or: [ctx == nil]]
		whileFalse:
			[ctx isUnwindContext
				ifTrue:
					[unwindBlock := ctx tempAt: 1.
					ctx tempAt: 1 put: nil.	"see comment in #ensure:"
					unwindBlock == nil
						ifFalse: [returnValue := unwindBlock value]].
			ctx := ctx sender].
	^returnValue! !


!BlockContext methodsFor: 'private' stamp: 'tfei 3/31/1999 17:40'!
cannotReturn: result
	"The receiver tried to return result to a method context that no longer exists."

	| ex newResult |
	ex := BlockCannotReturn new.
	ex result: result.
	newResult := ex signal.
	^newResult! !

!BlockContext methodsFor: 'exceptions' stamp: 'tfei 6/5/1999 18:54'!
ensure: aBlock
	"Evaluate a termination block after evaluating the receiver, regardless of whether the receiver's evaluation completes."

	| returnValue |
	returnValue := self valueUninterruptably.
	"aBlock wasn't nil when execution of this method began; it is nil'd out by the unwind machinery, and that's how we know it's already been evaluated ... otherwise, obviously, it needs to be evaluated"
	aBlock == nil
		ifFalse: [aBlock value].
	^returnValue! !

!BlockContext methodsFor: 'exceptions' stamp: 'tfei 6/5/1999 18:53'!
ifCurtailed: aBlock
	"Evaluate the receiver with an abnormal termination action."

	^self valueUninterruptably! !

!BlockContext methodsFor: 'exceptions' stamp: 'tfei 6/5/1999 16:15'!
on: exception do: handlerAction
	"Evaluate the receiver in the scope of an exception handler."

	^self value! !

!BlockContext methodsFor: 'private-exceptions' stamp: 'tfei 6/9/1999 16:39'!
valueUninterruptably
	"Temporarily make my home Context unable to return control to its sender, to guard against circumlocution of the ensured behavior."

	| sendingContext result homeSender |
	sendingContext := thisContext sender sender.
	homeSender _ home swapSender: nil.
	[[result := self
				on: BlockCannotReturn
				do:
					[:ex |
					thisContext unwindTo: sendingContext.
					sendingContext home answer: ex result.
					ex return: ex result]]
						on: ExceptionAboutToReturn
						do:
							[:ex |
							home sender == nil
								ifTrue:
									[home swapSender: homeSender.
									ex resume: homeSender]
								ifFalse: [ex resume: nil]]]
		on: Exception
		do:
			[:ex |
			home swapSender: homeSender.
			ex pass].
	home swapSender: homeSender.
	^result! !


!Exception commentStamp: '<historical>' prior: 0!
This is the main class used to implement the exception handling system (EHS).  It plays two distinct roles:  that of the exception, and that of the exception handler.  More specifically, it implements the bulk of the protocols laid out in the ANSI specification - those protocol names are reflected in the message categories.

Exception is an abstract class.  Instances should neither be created nor trapped.  In most cases, subclasses should inherit from Error or Notification rather than directly from Exception.

In implementing this EHS, The Fourth Estate Inc. incorporated some ideas and code from Craig Latta's EHS.  His insights were crucial in allowing us to implement BlockContext>>valueUninterruptably (and by extension, #ensure: and #ifCurtailed:), and we imported the following methods with little or no modification:

ContextPart>>terminateTo:
ContextPart>>terminate
MethodContext>>receiver:
MethodContext>>answer:

Thanks, Craig!!!

!Exception methodsFor: 'private' stamp: 'tfei 4/1/1999 17:58'!
findHandlerFrom: startCtx

	| ctx handler |
	ctx := startCtx.
	[ctx == nil]
		whileFalse:
			[ctx isHandlerContext
				ifTrue:
					[handler := ctx tempAt: 1. "the first argument"
					(handler handles: self)
						ifTrue:
							[handlerContext := ctx.
							^ctx]].
			ctx := ctx sender].
	^nil! !

!Exception methodsFor: 'private' stamp: 'tfei 6/13/1999 02:16'!
handlerAction

	| na handler returnValue |
	handler := handlerContext tempAt: 2.	"the second argument"
	na := handler numArgs.
	returnValue := na == 0
						ifTrue: [handler value]
						ifFalse: [handler value: self].
	resignalException == nil ifFalse: [^returnValue].
	"Execution will only continue beyond this point if the handler did not specify a handler action."
	self isResumable
		ifTrue: [self resume: returnValue]
		ifFalse: [self return: returnValue]! !

!Exception methodsFor: 'private' stamp: 'tfei 3/22/1999 10:43'!
initialContext: aContext

	initialContext := aContext! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/4/1999 17:49'!
isNested
	"Determine whether the current exception handler is within the scope of another handler for the same exception."

	| outer |
	outer := self findHandlerFrom: handlerContext sender.
	^outer ~~ nil! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/4/1999 17:51'!
outer
	"Evaluate the enclosing exception action for the receiver and return."

	| outer |
	self isResumable
		ifTrue:
			[outer := self findHandlerFrom: handlerContext sender.
			outer == nil
				ifTrue: [self defaultAction]
				ifFalse: [self handlerAction]]
		ifFalse: [self pass]! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/9/1999 16:33'!
pass
	"Yield control to the enclosing exception action for the receiver."

	| handler |
	handler := self findHandlerFrom: handlerContext sender.
	^handler == nil
		ifTrue: [self defaultAction]
		ifFalse: [self handlerAction]! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/13/1999 02:05'!
resignalAs: replacementException
	"Signal an alternative exception in place of the receiver."

	thisContext unwindTo: initialContext.
	replacementException initialContext: initialContext.
	resignalException := replacementException! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/4/1999 17:54'!
resume
	"Return from the message that signaled the receiver."


	^self resume: nil! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/7/1999 13:39'!
resume: resumptionValue
	"Return the argument as the value of the message that signaled the receiver."


	| tc |
	self isResumable ifFalse: [IllegalResumeAttempt signal].
	tc := thisContext.
	tc unwindTo: initialContext.
	tc terminateTo: initialContext.
	^resumptionValue! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/4/1999 17:55'!
retry
	
"Abort an exception handler and re-evaluate its protected block."

	thisContext unwindTo: handlerContext.
	thisContext terminateTo: handlerContext.
	handlerContext restart! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/5/1999 00:34'!
retryUsing: alternativeBlock
	
"Abort an exception handler and evaluate a new block in place of the handler's protected block."

	handlerContext receiver: alternativeBlock.
	self retry! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/4/1999 17:57'!
return
	"Return nil as the value of the block protected by the active exception handler."

	self return: nil! !

!Exception methodsFor: 'signaledException' stamp: 'tfei 6/9/1999 16:35'!
return: returnValue
	"Return the argument as the value of the block protected by the active exception handler."

	| handlerHomeContext |
	handlerHomeContext := ExceptionAboutToReturn signal.
	initialContext unwindTo: handlerContext.
	thisContext terminateTo: handlerContext.
	handlerHomeContext == nil
		ifFalse: [handlerContext sender swapSender: handlerHomeContext].
	^returnValue! !

!Exception methodsFor: 'exceptionBuilder' stamp: 'tfei 6/4/1999 17:47'!
messageText: signalerText
	
"Set an exception's message text."

	messageText := signalerText! !

!Exception methodsFor: 'exceptionBuilder' stamp: 'tfei 6/6/1999 23:06'!
tag: t
	"TFEI - This message is not specified in the ANSI protocol, but that looks like an oversight because #tag is specified, and states that the signaler may store the tag value."

	tag := t! !

!Exception methodsFor: 'exceptionDescription' stamp: 'tfei 6/6/1999 23:06'!
defaultAction
	"The default action taken if the exception is signaled."


	self subclassResponsibility! !

!Exception methodsFor: 'exceptionDescription' stamp: 'tfei 6/6/1999 23:09'!
description
	
"Return a textual description of the exception."

	| desc mt |
	desc := self class name asString.
	^(mt := self messageText) == nil
		ifTrue: [desc]
		ifFalse: [desc, ': ', mt]! !

!Exception methodsFor: 'exceptionDescription' stamp: 'tfei 6/4/1999 17:36'!
isResumable
	
"Determine whether an exception is resumable."

	^false! !

!Exception methodsFor: 'exceptionDescription' stamp: 'tfei 6/4/1999 17:40'!
messageText
	
"Return an exception's message text."

	^messageText! !

!Exception methodsFor: 'exceptionDescription' stamp: 'tfei 6/4/1999 17:44'!
tag
	"Return an exception's tag value."

	^tag == nil
		ifTrue: [tag]
		ifFalse: [self messageText]! !

!Exception methodsFor: 'exceptionSignaler' stamp: 'tfei 6/13/1999 02:06'!
signal
	"Signal the occurrence of an exceptional condition."

	| result |
	initialContext == nil ifTrue: [initialContext := thisContext sender].
	resignalException := nil.
	(self findHandlerFrom: initialContext) == nil
		ifTrue: [^self defaultAction].
	result := self handlerAction.
	^resignalException == nil
		ifTrue: [result]
		ifFalse: [resignalException signal]! !

!Exception methodsFor: 'exceptionSignaler' stamp: 'tfei 6/4/1999 17:46'!
signal: signalerText
	"Signal the occurrence of an exceptional condition with a specified textual description."

	self messageText: signalerText.
	initialContext == nil ifTrue: [initialContext := thisContext sender].
	^self signal! !


!BlockCannotReturn commentStamp: '<historical>' prior: 0!
This class is private to the EHS implementation.  Its use allows for ensured execution to survive code such as:

[self doThis.
^nil]
	ensure: [self doThat]

Signaling or handling this exception is not recommended.!

!BlockCannotReturn methodsFor: 'accessing' stamp: 'tfei 3/30/1999 12:54'!
result

	^result! !

!BlockCannotReturn methodsFor: 'accessing' stamp: 'tfei 3/30/1999 12:54'!
result: r

	result := r! !

!BlockCannotReturn methodsFor: 'exceptionDescription' stamp: 'tfei 3/30/1999 12:55'!
defaultAction

	self messageText: 'Block cannot return'.
	^super defaultAction! !

!BlockCannotReturn methodsFor: 'exceptionDescription' stamp: 'tfei 4/2/1999 15:49'!
isResumable

	^true! !


!Error commentStamp: '<historical>' prior: 0!
From the ANSI standard:
This protocol describes the behavior of instances of class Error. These are used to represent error conditions that prevent the normal continuation of processing. Actual error exceptions used by an application may be subclasses of this class.
As Error is explicitly specified  to be subclassable, conforming implementations must implement its behavior in a non-fragile manner.

Additional notes:
Error>defaultAction uses an explicit test for the presence of the Debugger class to decide whether or not it is in development mode.  In the future, TFEI hopes to enhance the semantics of #defaultAction to improve support for pluggable default handlers.!

!Error methodsFor: 'private' stamp: 'tfei 6/5/1999 00:10'!
devDefaultAction

	Debugger
		openContext: thisContext
		label: self description
		contents: thisContext shortStack! !

!Error methodsFor: 'private' stamp: 'tfei 6/5/1999 00:09'!
isDevelopmentEnvironmentPresent

	^Smalltalk includesKey: #Debugger! !

!Error methodsFor: 'private' stamp: 'tfei 6/5/1999 00:26'!
runtimeDefaultAction
	"Dump the stack trace to a log file, then exit the program (image)."

	| file |
	file := FileStream newFileNamed: ('error', Utilities dateTimeSuffix, FileDirectory dot, 'log') asFileName.
	Smalltalk timeStamp: file.
	(thisContext sender stackOfSize: 20) do: [:ctx | file cr. ctx printOn: file].
	file close.
	Smalltalk snapshot: false andQuit: true! !

!Error methodsFor: 'exceptionDescription' stamp: 'tfei 6/5/1999 00:10'!
defaultAction
	"The current computation is terminated. The cause of the error should be logged or reported to the user. If the program is operating in an interactive debugging environment the computation should be suspended and the debugger activated."

	self isDevelopmentEnvironmentPresent
		ifTrue: [self devDefaultAction]
		ifFalse: [self runtimeDefaultAction]! !


!Exception class methodsFor: 'exceptionInstantiator' stamp: 'tfei 6/4/1999 18:12'!
signal
	"Signal the occurrence of an exceptional condition."

	| ex |
	ex := self new.
	ex initialContext: thisContext sender.
	^ex signal! !

!Exception class methodsFor: 'exceptionInstantiator' stamp: 'tfei 6/4/1999 18:15'!
signal: signalerText
	"TFEI - Signal the occurrence of an exceptional condition with a specified textual description."

	| ex |
	ex := self new.
	ex initialContext: thisContext sender.
	^ex signal: signalerText! !

!Exception class methodsFor: 'exceptionSelector' stamp: 'tfei 6/4/1999 18:08'!
, anotherException
	"Create an exception set."

	^ExceptionSet new
		add: self;
		add: anotherException;
		yourself! !

!Exception class methodsFor: 'exceptionSelector' stamp: 'tfei 6/4/1999 18:08'!
handles: exception
	"Determine whether an exception handler will accept a signaled exception."

	^exception isKindOf: self! !


!ExceptionSet commentStamp: '<historical>' prior: 0!
An ExceptionSet is a grouping of exception handlers which acts as a single handler.  Within the group, the most recently added handler will be the last handler found during a handler search (in the case where more than one handler in the group is capable of handling a given exception). !

!ExceptionSet methodsFor: 'private' stamp: 'tfei 3/23/1999 14:07'!
initialize

	exceptions := OrderedCollection new! !

!ExceptionSet methodsFor: 'private' stamp: 'tfei 7/16/1999 1:07'!
add: anException

	exceptions add: anException! !

!ExceptionSet methodsFor: 'exceptionSelector' stamp: 'tfei 6/4/1999 18:37'!
, anException
	"Return an exception set that contains the receiver and the argument exception. This is commonly used to specify a set of exception selectors for an exception handler."

	self add: anException.
	^self! !

!ExceptionSet methodsFor: 'exceptionSelector' stamp: 'tfei 6/4/1999 18:37'!
handles: anException
	"Determine whether an exception handler will accept a signaled exception."


	exceptions do:
		[:ex |
		(ex handles: anException)
			ifTrue: [^true]].
	^false! !


!ExceptionSet class methodsFor: 'exceptionInstantiator' stamp: 'tfei 3/23/1999 14:06'!
new

	^super new initialize! !


!IllegalResumeAttempt commentStamp: '<historical>' prior: 0!
This class is private to the EHS implementation.  An instance of it is signaled whenever an attempt is made to resume from an exception which answers false to #isResumable.!

!IllegalResumeAttempt methodsFor: 'comment' stamp: 'tfei 6/2/1999 14:59'!
readMe

	"Never handle this exception!!"! !


!MessageNotUnderstood commentStamp: '<historical>' prior: 0!
This exception is provided to support Object>>doesNotUnderstand:.!

!MessageNotUnderstood methodsFor: 'exceptionBuilder' stamp: 'tfei 6/13/1999 02:31'!
message: aMessage
	"TFEI extension"

	message := aMessage! !

!MessageNotUnderstood methodsFor: 'exceptionDescription' stamp: 'tfei 6/4/1999 18:30'!
isResumable
	"Determine whether an exception is resumable."

	^true! !

!MessageNotUnderstood methodsFor: 'exceptionDescription' stamp: 'tfei 6/4/1999 18:27'!
message
	"Answer the selector and arguments of the message that failed."

	^message! !

!MessageNotUnderstood methodsFor: 'exceptionDescription' stamp: 'tfei 6/5/1999 17:24'!
messageText
	
"Return an exception's message text."

	^messageText == nil
		ifTrue:
			[message == nil
				ifTrue: [super messageText]
				ifFalse: [message selector asString]]
		ifFalse:
			[messageText]! !

!MessageNotUnderstood methodsFor: 'exceptionDescription' stamp: 'tfei 6/4/1999 18:31'!
receiver
	"Answer the receiver of the message that failed."

	^initialContext receiver! !


!MethodContext methodsFor: 'private-exceptions' stamp: 'tfei 6/7/1999 20:46'!
cannotReturn: result

	Debugger
		openContext: thisContext
		label: 'computation has been terminated'
		contents: thisContext printString! !

!MethodContext methodsFor: 'private-exceptions' stamp: 'tfei 3/23/1999 13:00'!
receiver: r

	receiver := r! !

!MethodContext methodsFor: 'controlling' stamp: 'crl 3/1/1999 12:23'!
answer: anObject
	"Modify my code, from the current program counter value, to answer anObject."
	| methodSize |
	self push: anObject.
	(method at: pc) = 124 ifFalse: [
		methodSize _ method size.
		method _ (
			(method copyFrom: 1 to: pc + 4)
				at: pc + 1 put: 124;
				at: pc + 2 put: (method at: methodSize - 2);
				at: pc + 3 put: (method at: methodSize - 1);
				at: pc + 4 put: (method at: methodSize);
				yourself)]! !


!Notification commentStamp: '<historical>' prior: 0!
A Notification is an indication that something interesting has occurred.  If it is not handled, it will pass by without effect.!

!Notification methodsFor: 'exceptionDescription' stamp: 'tfei 6/4/1999 18:18'!
defaultAction
	"No action is taken. The value nil is returned as the value of the message that signaled the exception."


	^nil! !

!Notification methodsFor: 'exceptionDescription' stamp: 'tfei 6/4/1999 18:17'!
isResumable
	"Answer true. Notification exceptions by default are specified to be resumable."

	^true! !


!ExceptionAboutToReturn commentStamp: '<historical>' prior: 0!
This class is private to the EHS implementation.  Its use allows for ensured execution to survive code such as:

[self doThis.
^nil]
	ensure: [self doThat]

Signaling or handling this exception is not recommended.  Not even slightly.!

!Process methodsFor: 'changing process state' stamp: 'tfei 4/1/1999 18:23'!
terminate 
	"Stop the process that the receiver represents forever."

	| context |
	Processor activeProcess == self
		ifTrue: 
			[thisContext unwindTo: nil.
			thisContext sender == nil ifFalse:
				[thisContext sender release].
			thisContext removeSelf suspend]
		ifFalse: 
			[myList == nil
				ifFalse: 
					[myList remove: self ifAbsent: [].
					myList _ nil].
			context _ suspendedContext.
			suspendedContext _ nil.
			context == nil ifFalse: [context unwindTo: nil].
			(context ~~ nil and: [context sender ~~ nil])
				ifTrue: [context sender release]]! !


!Warning commentStamp: '<historical>' prior: 0!
A Warning is a Notification which by default should be brought to the attention of the user.!

!Warning methodsFor: 'exceptionDescription' stamp: 'tfei 6/7/1999 13:31'!
defaultAction
	"The user should be notified of the occurrence of an exceptional occurrence and given an option of continuing or aborting the computation. The description of the occurrence should include any text specified as the argument of the #signal: message."

	| continue |
	continue := self confirm: 'Warning: ', self messageText, ' Continue?'.
	continue ifTrue: [self resume]! !

!Warning methodsFor: 'private-exceptionSignaler' stamp: 'tfei 6/7/1999 20:39'!
signal
	"Signal the occurrence of an exceptional condition."
	"Warning is overriding this and marking it private in order to get around an apparent contradiction within the ANSI specification:
		Warning>>defaultAction must give the user the option to 'abort the computation'.
		Warning is resumable.
		Exception>>signal must return to its caller if it reaches a defaultAction in the case of a resumable exception.
		Therefore, Warning needs to invoke a handler action from its defaultAction; that handler action would seem to be return:, but according to the spec, 'It is erroneous to directly or indirectly send this message from within a #defaultAction method  to the receiver of the #defaultAction method.'

We have chosen to support the apparently intended behavior of Warning, at the expense of introducing a mildly unpleasant hack - in case of an unhandled Warning, we determine whether the user has elected to continue or not.  In the latter case, this process needs to terminate immediately, without returning to the sender of #signal.  To accomplish this, we nil out the sender on the call stack, and then use MethodContext>>cannotReturn: to open a notifier and allow the environment to continue in use."

	| result |
	initialContext == nil ifTrue: [initialContext := thisContext sender].
	resignalException := nil.
	(self findHandlerFrom: initialContext) == nil
		ifTrue:
			[self defaultAction.
			"if user elected to continue the computation, execution did not reach this line"
			initialContext unwindTo: nil.
			thisContext terminateTo: nil]
		ifFalse:
			[result := self handlerAction.
			^resignalException == nil
				ifTrue: [result]
				ifFalse: [resignalException signal]]! !


!ZeroDivide commentStamp: '<historical>' prior: 0!
ZeroDivide may be signaled when a mathematical division by 0 is attempted.!

!ZeroDivide methodsFor: 'exceptionDescription' stamp: 'tfei 6/5/1999 17:29'!
dividend
	"Answer the number that was being divided by zero."

	^dividend! !

!ZeroDivide methodsFor: 'exceptionDescription' stamp: 'tfei 6/5/1999 17:26'!
isResumable
	
"Determine whether an exception is resumable."

	^true! !

!ZeroDivide methodsFor: 'exceptionBuilder' stamp: 'tfei 6/5/1999 17:34'!
dividend: argument
	"TFEI - Specify the number that was being divided by zero."

	dividend := argument! !


!ZeroDivide class methodsFor: 'exceptionInstantiator' stamp: 'tfei 6/5/1999 00:38'!
dividend: argument
	"Signal the occurance of a division by zero."

	| err |
	err := self new dividend: argument; yourself.
	err signal! !


