/****************************************************************************
*   PROJECT: Squeak port for Win32 (NT / Win95)
*   FILE:    sqWin32Window.c
*   CONTENT: Window management
*
*   AUTHOR:  Andreas Raab (ar)
*   ADDRESS: University of Magdeburg, Germany
*   EMAIL:   raab@isg.cs.uni-magdeburg.de
*   RCSID:   $Id: sqWin32Window.c,v 2.0.1.12 1998/10/02 22:28:01 raab Exp raab $
*
*   NOTES:
*    1) Currently supported Squeak color depths include 1,4,8,16,32 bits
*    2) To speed up drawing a slight update delay has been added (Toggle with F2)
*    3) The modifier keys are mapped as follows:
*
*        Mac    |  Win32
*       --------------------
*       Shift   -> Shift
*       Ctrl    -> Ctrl
*       Command -> Left ALT
*       Option  -> Right ALT
*
*****************************************************************************/
#include <windows.h>
#include <commdlg.h>
#include <excpt.h>

#include "sq.h"

static TCHAR RCSID[]= TEXT("$Id: sqWin32Window.c,v 2.0.1.12 1998/10/02 22:28:01 raab Exp raab $");

/****************************************************************************/
/* General Squeak declarations and definitions                              */
/****************************************************************************/

/*** Variables -- Imported from Virtual Machine ***/
extern int interruptPending;
extern int interruptCheckCounter;
extern int interruptKeycode;
extern int fullScreenFlag;
extern int deferDisplayUpdates; /* Is the Interpreter doing defered updates for us?! */


/*** Variables -- image and path names ***/
#define IMAGE_NAME_SIZE MAX_PATH

char imageName[MAX_PATH+1];		  /* full path and name to image */
TCHAR vmPath[MAX_PATH+1];		    /* full path to interpreter's directory */
TCHAR vmName[MAX_PATH+1];		    /* name of the interpreter's executable */

int		 savedWindowSize= 0;	/* initial size of window */
POINT		 mousePosition;		/* position at last PointerMotion event */

/*** Variables -- Event Recording ***/
#define KEYBUF_SIZE 64

int keyBuf[KEYBUF_SIZE];	/* circular buffer */
int keyBufGet= 0;		/* index of next item of keyBuf to read */
int keyBufPut= 0;		/* index of next item of keyBuf to write */
int keyBufOverflows= 0;	        /* number of characters dropped */

int buttonState= 0;		/* mouse button and modifier state when mouse
				   button went down or 0 if not pressed */

/*** Win32-related Variables (declared in sqWin32.h) ***/
HWND stWindow = NULL;      /*	the squeak window */
HINSTANCE hInstance;	     /*	the instance of squeak running */
HCURSOR currentCursor=0;	 /*	current cursor displayed by squeak */
HPALETTE palette;	         /*	the palette (might be unused) */
LOGPALETTE *logPal;	       /*	the logical palette definition */
BITMAPINFO *bmi1;	         /*	1 bit depth bitmap info */
BITMAPINFO *bmi4;	         /*	4 bit depth bitmap info */
BITMAPINFO *bmi8;	         /*	8 bit depth bitmap info */
BITMAPINFO *bmi16;	       /*	16 bit depth bitmap info */
BITMAPINFO *bmi32;	       /*	32 bit depth bitmap info */
BOOL fWindows95;           /* Are we running on Win95 or NT? */

/* Preference values */
BOOL fDeferredUpdate = 1; /* I prefer the deferred update*/
BOOL fShowConsole = 0;    /* do we show the console window?*/
BOOL fDynamicConsole = 1; /* Should we show the console if any errors occur? */
BOOL fReduceCPUUsage = 1; /* Should we reduce CPU usage? */
BOOL f3ButtonMouse = 0;   /* Should we use a real 3 button mouse mapping? */

/* Startup options */
BOOL  fHeadlessImage = 0;      /* Do we run headless? */
BOOL  fRunService = 0;         /* Do we run as NT service? */
DWORD dwMemorySize = 0;        /* How much memory do we use? */

HANDLE vmSemaphoreMutex = 0; /* the mutex for synchronization of access to the
                                external semaphores in the VM */

/* variables for cached display */
RECT updateRect;		     /*	the rectangle to update */
HRGN updateRgn;	     	     /*	the region to update (more accurate) */
BOOL updateRightNow;	     /*	update flag */
HWND  consoleWindow;       /* console */

/* VM preference variables */
#ifndef NO_PREFERENCES
TCHAR iniName[MAX_PATH+1]; /* full path and name to ini file */
HMENU vmPrefsMenu;         /* preferences menu */
#endif

#ifndef NO_PRINTER
/* printer settings */
PRINTDLG printValues;
#endif

/* misc declarations */
extern int byteSwapped(int);
extern int convertToSqueakTime(SYSTEMTIME);
int recordMouseDown(WPARAM, LPARAM);
int recordModifierButtons();
int recordKeystroke(UINT,WPARAM,LPARAM);
int recordVirtualKey(UINT,WPARAM,LPARAM);
void SetSystemTrayIcon(BOOL on);


/* UNICODE stuff */
const TCHAR U_ON[]  = TEXT("1");
const TCHAR U_OFF[] = TEXT("0");
const TCHAR U_GLOBAL[] = TEXT("Global");
const TCHAR U_SLASH[] = TEXT("/");
const TCHAR U_BACKSLASH[] = TEXT("\\");

/****************************************************************************/
/*                      Synchronization functions                           */
/****************************************************************************/

/* NOTE: Why do we need this? When running multi-threaded code such as in
         the networking code and in midi primitives
         we will signal the interpreter several semaphores. If one of those
         threads gets interrupted by any other thread trying to signal
         the interpreter we might have a problem. Even with the current
         synchronization we live somewhat dangerous, since the main thread
         does not use this synchronization when accessing the external
         semaphores. */

int synchronizedSignalSemaphoreWithIndex(int semaIndex)
{ int result;

  /* wait until we have access */
  WaitForSingleObject(vmSemaphoreMutex, INFINITE);
  /* do our job */
  result = signalSemaphoreWithIndex(semaIndex);
  /* and release access */
  ReleaseMutex(vmSemaphoreMutex);
  return result;
}

/****************************************************************************/
/*                   Preference functions                                   */
/****************************************************************************/

#ifndef NO_PREFERENCES
void TrackMenu(HMENU menu)
{ POINT p;

  GetCursorPos(&p);
  TrackPopupMenu(menu, TPM_LEFTALIGN, p.x, p.y, 0, stWindow, NULL);
}

void SetDeferredUpdate()
{
  CheckMenuItem(vmPrefsMenu, 0x0020, MF_BYCOMMAND | (fDeferredUpdate ? MF_CHECKED : MF_UNCHECKED));
  fullDisplayUpdate();
  WritePrivateProfileString(U_GLOBAL,TEXT("DeferUpdate"),fDeferredUpdate ? U_ON : U_OFF,iniName);
}

void SetShowConsole()
{
  CheckMenuItem(vmPrefsMenu, 0x0030, MF_BYCOMMAND | (fShowConsole ? MF_CHECKED : MF_UNCHECKED));
  if(IsWindow(stWindow)) ShowWindow(consoleWindow, fShowConsole ? SW_SHOW : SW_HIDE);
  WritePrivateProfileString(U_GLOBAL,TEXT("ShowConsole"),fShowConsole ? U_ON:U_OFF,iniName);
}


void SetDynamicConsole()
{
  CheckMenuItem(vmPrefsMenu, 0x0050, MF_BYCOMMAND | (fDynamicConsole ? MF_CHECKED : MF_UNCHECKED));
  fullDisplayUpdate();
  WritePrivateProfileString(U_GLOBAL,TEXT("DynamicConsole"),fDynamicConsole ? U_ON:U_OFF,iniName);
}

void SetReduceCPUUsage()
{
  CheckMenuItem(vmPrefsMenu, 0x0060, MF_BYCOMMAND | (fReduceCPUUsage ? MF_CHECKED : MF_UNCHECKED));
  WritePrivateProfileString(U_GLOBAL,TEXT("ReduceCPUUsage"),fReduceCPUUsage ? U_ON:U_OFF,iniName);
}

void Set3ButtonMouse()
{
  CheckMenuItem(vmPrefsMenu, 0x0070, MF_BYCOMMAND | (f3ButtonMouse ? MF_CHECKED : MF_UNCHECKED));
  WritePrivateProfileString(U_GLOBAL,TEXT("3ButtonMouse"),f3ButtonMouse ? U_ON : U_OFF,iniName);
}

void SetDefaultPrinter()
{
#ifndef NO_PRINTER
  printValues.Flags = PD_PRINTSETUP;
  PrintDlg(&printValues);
#endif /* NO_PRINTER */
}
#endif /* NO_PREFERENCES */

/* SetupPreferences might be called from outside; so leave a stub in */
void SetupPreferences()
{
#ifndef NO_PREFERENCES
  /* Set preferences */
  fDeferredUpdate = GetPrivateProfileInt(U_GLOBAL,TEXT("DeferUpdate"),1,iniName);
  fShowConsole    = GetPrivateProfileInt(U_GLOBAL,TEXT("ShowConsole"),0,iniName);
  fDynamicConsole = GetPrivateProfileInt(U_GLOBAL,TEXT("DynamicConsole"),1,iniName);
  fReduceCPUUsage = GetPrivateProfileInt(U_GLOBAL,TEXT("ReduceCPUUsage"),1,iniName);
  f3ButtonMouse   = GetPrivateProfileInt(U_GLOBAL,TEXT("3ButtonMouse"),0,iniName);

  SetDeferredUpdate();
  SetShowConsole();
  SetDynamicConsole();
  SetReduceCPUUsage();
  Set3ButtonMouse();
#endif /* NO_PREFERENCES */
}

/****************************************************************************/
/*                   Message Processing                                     */
/****************************************************************************/

/* The entry to the message hooks called from the window procedure.
   If another module requires to process messages by itself, it should 
   put its message procedure in this place. */
messageHook firstMessageHook = 0;

/* The entry to a pre-message hook. Can be used to intercept any messages
   to the squeak main window.
   Note: This is meant for development purposes only! */
messageHook preMessageHook = 0;

/* main window procedure */
LRESULT CALLBACK MainWndProc (HWND hwnd,
                              UINT message,
                              WPARAM wParam,
                              LPARAM lParam)
{ PAINTSTRUCT ps;

  /* Intercept any messages if wanted */
  if(preMessageHook)
    if((*preMessageHook)(hwnd, message,wParam, lParam))
       return 1;
  switch(message)
    {
      case WM_SYSCOMMAND:
      case WM_COMMAND:
        switch (wParam & 0xFFF0) {
#if !defined(_WIN32_WCE)
          case SC_MINIMIZE:
            if(fHeadlessImage)
              ShowWindow(stWindow, SW_HIDE);
            else
              return DefWindowProc(hwnd, message, wParam, lParam);
            break;
#endif /* defined(_WIN32_WCE) */
          case SC_CLOSE:
            /* do not attempt to close the window directly ! */
            if(MessageBox(stWindow,TEXT("Quit Squeak without saving?"),TEXT("Squeak"),MB_YESNO) == IDYES)
              {
                DestroyWindow(stWindow);
                exit(1);
                return 1;
              }
            else return 0;
            break;
#ifndef NO_PREFERENCES
          case 0x0010: 
            MessageBox(stWindow,RCSID,TEXT("About Squeak VM on Win32"), MB_OK);
            break;
          case 0x0020:
            fDeferredUpdate = !fDeferredUpdate;
            SetDeferredUpdate();
            break;
          case 0x0030:
            fShowConsole = !fShowConsole;
            SetShowConsole();
            break;
          case 0x0040:
            break;
          case 0x0050:
            fDynamicConsole = !fDynamicConsole;
            SetDynamicConsole();
            break;
          case 0x0060:
            fReduceCPUUsage = !fReduceCPUUsage;
            SetReduceCPUUsage();
            break;
          case 0x0070:
            f3ButtonMouse = !f3ButtonMouse;
            Set3ButtonMouse();
            break;
          case 0x0080:
            SetDefaultPrinter();
            break;
#endif /* NO_PREFERENCES */
          default:
            return DefWindowProc(hwnd,message,wParam,lParam);
        }
      break;
      /*  mousing */
      case WM_MOUSEMOVE: 
        mousePosition.x = LOWORD(lParam);
	      mousePosition.y = HIWORD(lParam);
	      break;
      case WM_LBUTTONDOWN:
      case WM_RBUTTONDOWN:
      case WM_MBUTTONDOWN:
        mousePosition.x = LOWORD(lParam);
	      mousePosition.y = HIWORD(lParam);
        /* check for console focus */
        if(GetFocus() == consoleWindow) SetFocus(stWindow);
        recordMouseDown(wParam, lParam);
        recordModifierButtons();
        /* capture the mouse as long as the button is pressed so we can scroll outside */
        SetCapture(stWindow);
        break;
      case WM_LBUTTONUP:
      case WM_RBUTTONUP:
      case WM_MBUTTONUP:
        mousePosition.x = LOWORD(lParam);
	      mousePosition.y = HIWORD(lParam);
        /* check for console focus */
        if(GetFocus() == consoleWindow) SetFocus(stWindow);
        recordMouseDown(wParam,lParam);
        recordModifierButtons();
        /* release capture */
        ReleaseCapture();
        break;
      /* virtual key codes */
      case WM_KEYDOWN:
        if(GetFocus() == consoleWindow) return DefWindowProc(hwnd, message, wParam, lParam);
        recordModifierButtons();
        recordVirtualKey(message,wParam,lParam);
        break;
      case WM_KEYUP:
        if(GetFocus() == consoleWindow) return DefWindowProc(hwnd, message, wParam, lParam);
        recordModifierButtons();
        break;
      /* character codes */
      case WM_CHAR:
      case WM_SYSCHAR:
        if(GetFocus() == consoleWindow) return DefWindowProc(hwnd, message, wParam, lParam);
        recordModifierButtons();
        recordKeystroke(message,wParam,lParam);
        break;
      case WM_DEADCHAR:
      case WM_SYSDEADCHAR:
        break;
      /* expose events */
      case WM_PAINT:
        /* retrieve update region _before_ calling BeginPaint() !!! */
        GetUpdateRgn(hwnd,updateRgn,FALSE);
        BeginPaint(hwnd,&ps);
        /* store the rectangle required for image bit reversal */
        updateRect = ps.rcPaint;
        EndPaint(hwnd,&ps);
       /* Avoid repaint unless absolutely necessary (null update rects are quite common) */
       if (!IsRectEmpty(&updateRect))
          {
            /* force redraw the next time ioShowDisplay() is called */
            updateRightNow = TRUE;
            fullDisplayUpdate();  /* this makes VM call ioShowDisplay */
          }
       break;
     case WM_SIZE:
       if(hwnd == stWindow)
          { RECT r;
            GetClientRect(stWindow,&r);
            MoveWindow(consoleWindow, 0, r.bottom-100, r.right, 100, 1);
          }
        else return DefWindowProc(hwnd,message,wParam,lParam);

#if defined(_WIN32_WCE)
     /* All erasing is done in the display function and the window is always full screen*/
     case WM_ERASEBKGND:
       return TRUE;
#endif

#if !defined(_WIN32_WCE)	/* Don't change the cursor or system tray on WinCE */
      /* cursor redraw */
      case WM_SETCURSOR:
        /* keep currentCursor */
        if((LOWORD(lParam) == HTCLIENT) && currentCursor)
          {
            SetCursor(currentCursor);
            break;
          }
        else return DefWindowProc(hwnd,message,wParam,lParam);
      case WM_USER+42:
        /* system tray notification */
        if(wParam != (UINT)hInstance) return 0;
        /* if right button, show system menu */
        /* if(lParam == WM_RBUTTONUP)
          TrackMenu(GetSystemMenu(stWindow,0)); */
        /* if double clicked, show main window */
        if(lParam == WM_LBUTTONDBLCLK)
          {
            if(!IsWindowVisible(stWindow))
              ShowWindow(stWindow, SW_SHOW);
            BringWindowToTop(stWindow);
          }
        return 0;
#endif /* !defined(_WIN32_WCE) */
      default:
        /* Unprocessed messages may be processed outside the current
           module. If firstMessageHook is non-NULL and returns a non
           zero value, the message has been successfully processed */
        if(firstMessageHook)
          if((*firstMessageHook)(hwnd, message, wParam, lParam))
            return 1;
        return DefWindowProc(hwnd,message,wParam,lParam);
    }
  return 1;
}

/****************************************************************************/
/*                      Timer Setup                                         */
/****************************************************************************/
/****************************************************************************/
/* Windows CE does not support waiting for semaphores, change notification  */
/* objects, console input, and timers, as does the Win32 version.           */
/* Also, it does not support waiting for process, thread, and mutex events. */
/****************************************************************************/

static DWORD dwTimerPeriod;

void SetupTimer()
{
#if defined(_WIN32_WCE) || defined(GCC_WORKAROUND)
  dwTimerPeriod = 0;
#else /* defined(_WIN32_WCE) */
  TIMECAPS tCaps;

  dwTimerPeriod = 0;
  if(timeGetDevCaps(&tCaps,sizeof(tCaps)) != 0)
    return;
  dwTimerPeriod = tCaps.wPeriodMin;
  if(timeBeginPeriod(dwTimerPeriod) != 0)
    return;
#endif /* defined(_WIN32_WCE) */
}

void ReleaseTimer()
{
#if !defined(_WIN32_WCE) && !defined(GCC_WORKAROUND)
  timeEndPeriod(dwTimerPeriod);
#endif /* !defined(_WIN32_WCE) */
}

/****************************************************************************/
/*                     Printer Setup                                        */
/****************************************************************************/

void SetupPrinter()
{
#ifndef NO_PRINTER
  ZeroMemory(&printValues, sizeof(printValues));
  printValues.lStructSize = sizeof(PRINTDLG);
  printValues.hInstance = hInstance;
  printValues.nMinPage = 1;
  printValues.nMaxPage = 1;
  printValues.nFromPage = 1;
  printValues.nToPage = 1;
  printValues.hwndOwner = stWindow;

  /* fetch default DEVMODE/DEVNAMES */
  printValues.Flags = PD_RETURNDEFAULT;
  PrintDlg(&printValues);
#endif
}


/****************************************************************************/
/*                   Color and Bitmap setup                                 */
/****************************************************************************/
#ifndef NO_STANDARD_COLORS
static void SetColorEntry(int index, int red, int green, int blue)
{
  logPal->palPalEntry[index].peRed = red / 256;
  logPal->palPalEntry[index].peGreen = green / 256;
  logPal->palPalEntry[index].peBlue = blue / 256;
  logPal->palPalEntry[index].peFlags = 0;
}

/* Generic color maps and bitmap info headers 1,4,8,16,32 bits per pixel */
void SetupPixmaps(void)
{ int i;

  logPal = malloc(sizeof(LOGPALETTE) + 255 * sizeof(PALETTEENTRY));
  logPal->palVersion = 0x300;
  logPal->palNumEntries = 256;

  /* 1-bit colors (monochrome) */
  SetColorEntry(0, 65535, 65535, 65535);	/* white or transparent */
  SetColorEntry(1,     0,     0,     0);	/* black */

  /* additional colors for 2-bit color */
  SetColorEntry(2, 65535, 65535, 65535);	/* opaque white */
  SetColorEntry(3, 32768, 32768, 32768);	/* 1/2 gray */

  /* additional colors for 4-bit color */
  SetColorEntry( 4, 65535,     0,     0);	/* red */
  SetColorEntry( 5,     0, 65535,     0);	/* green */
  SetColorEntry( 6,     0,     0, 65535);	/* blue */
  SetColorEntry( 7,     0, 65535, 65535);	/* cyan */
  SetColorEntry( 8, 65535, 65535,     0);	/* yellow */
  SetColorEntry( 9, 65535,     0, 65535);	/* magenta */
  SetColorEntry(10,  8192,  8192,  8192);	/* 1/8 gray */
  SetColorEntry(11, 16384, 16384, 16384);	/* 2/8 gray */
  SetColorEntry(12, 24576, 24576, 24576);	/* 3/8 gray */
  SetColorEntry(13, 40959, 40959, 40959);	/* 5/8 gray */
  SetColorEntry(14, 49151, 49151, 49151);	/* 6/8 gray */
  SetColorEntry(15, 57343, 57343, 57343);	/* 7/8 gray */


  /* additional colors for 8-bit color */
  /* 24 more shades of gray (does not repeat 1/8th increments) */
  SetColorEntry(16,  2048,  2048,  2048);	/*  1/32 gray */
  SetColorEntry(17,  4096,  4096,  4096);	/*  2/32 gray */
  SetColorEntry(18,  6144,  6144,  6144);	/*  3/32 gray */
  SetColorEntry(19, 10240, 10240, 10240);	/*  5/32 gray */
  SetColorEntry(20, 12288, 12288, 12288);	/*  6/32 gray */
  SetColorEntry(21, 14336, 14336, 14336);	/*  7/32 gray */
  SetColorEntry(22, 18432, 18432, 18432);	/*  9/32 gray */
  SetColorEntry(23, 20480, 20480, 20480);	/* 10/32 gray */
  SetColorEntry(24, 22528, 22528, 22528);	/* 11/32 gray */
  SetColorEntry(25, 26624, 26624, 26624);	/* 13/32 gray */
  SetColorEntry(26, 28672, 28672, 28672);	/* 14/32 gray */
  SetColorEntry(27, 30720, 30720, 30720);	/* 15/32 gray */
  SetColorEntry(28, 34815, 34815, 34815);	/* 17/32 gray */
  SetColorEntry(29, 36863, 36863, 36863);	/* 18/32 gray */
  SetColorEntry(30, 38911, 38911, 38911);	/* 19/32 gray */
  SetColorEntry(31, 43007, 43007, 43007);	/* 21/32 gray */
  SetColorEntry(32, 45055, 45055, 45055);	/* 22/32 gray */
  SetColorEntry(33, 47103, 47103, 47103);	/* 23/32 gray */
  SetColorEntry(34, 51199, 51199, 51199);	/* 25/32 gray */
  SetColorEntry(35, 53247, 53247, 53247);	/* 26/32 gray */
  SetColorEntry(36, 55295, 55295, 55295);	/* 27/32 gray */
  SetColorEntry(37, 59391, 59391, 59391);	/* 29/32 gray */
  SetColorEntry(38, 61439, 61439, 61439);	/* 30/32 gray */
  SetColorEntry(39, 63487, 63487, 63487);	/* 31/32 gray */

  /* The remainder of color table defines a color cube with six steps
     for each primary color. Note that the corners of this cube repeat
     previous colors, but simplifies the mapping between RGB colors and
     color map indices. This color cube spans indices 40 through 255.
     */
  {
    int r, g, b;

    for (r= 0; r < 6; r++)
      for (g= 0; g < 6; g++)
	for (b= 0; b < 6; b++)
	  {
	    int i= 40 + ((36 * r) + (6 * b) + g);
	    if (i > 255) error("index out of range in color table compuation");
	    SetColorEntry(i, (r * 65535) / 5, (g * 65535) / 5, (b * 65535) / 5);
	  }
  }

  palette = CreatePalette(logPal);


  /* generate BITMAPINFOHEADERs */
  /* 1 bit color depth */
  bmi1 = calloc(1,sizeof(BITMAPINFO) + 2 * sizeof(RGBQUAD));
  bmi1->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  bmi1->bmiHeader.biPlanes = 1;
  bmi1->bmiHeader.biBitCount = 1;
  bmi1->bmiHeader.biCompression = BI_RGB;
  for(i=0;i<2;i++)
    {
      bmi1->bmiColors[i].rgbRed = logPal->palPalEntry[i].peRed;
      bmi1->bmiColors[i].rgbGreen = logPal->palPalEntry[i].peGreen;
      bmi1->bmiColors[i].rgbBlue = logPal->palPalEntry[i].peBlue;
    }
  /* 4 bit color depth */
  bmi4 = calloc(1,sizeof(BITMAPINFO) + 16 * sizeof(RGBQUAD));
  bmi4->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  bmi4->bmiHeader.biPlanes = 1;
  bmi4->bmiHeader.biBitCount = 4;
  bmi4->bmiHeader.biCompression = BI_RGB;
  for(i=0;i<16;i++)
    {
      bmi4->bmiColors[i].rgbRed = logPal->palPalEntry[i].peRed;
      bmi4->bmiColors[i].rgbGreen = logPal->palPalEntry[i].peGreen;
      bmi4->bmiColors[i].rgbBlue = logPal->palPalEntry[i].peBlue;
    }
  /* 8 bit color depth */
  bmi8 = calloc(1,sizeof(BITMAPINFO) + 256 * sizeof(RGBQUAD));
  bmi8->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  bmi8->bmiHeader.biPlanes = 1;
  bmi8->bmiHeader.biBitCount = 8;
  bmi8->bmiHeader.biCompression = BI_RGB;
  for(i=0;i<256;i++)
    {
      bmi8->bmiColors[i].rgbRed = logPal->palPalEntry[i].peRed;
      bmi8->bmiColors[i].rgbGreen = logPal->palPalEntry[i].peGreen;
      bmi8->bmiColors[i].rgbBlue = logPal->palPalEntry[i].peBlue;
    }
  /* 16 bit color depth */
  bmi16 = calloc(1,sizeof(BITMAPINFO) + 4 * sizeof(DWORD));
  bmi16->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  bmi16->bmiHeader.biPlanes = 1;
  bmi16->bmiHeader.biBitCount = 16;
  bmi16->bmiHeader.biCompression = BI_BITFIELDS;
  ((DWORD*) bmi16->bmiColors)[0] = (31 << 10); /* red mask */
  ((DWORD*) bmi16->bmiColors)[1] = (31 << 5); /* green mask */
  ((DWORD*) bmi16->bmiColors)[2] = (31 << 0); /* blue mask */

  /* 32 bit color depth */
  bmi32 = calloc(1,sizeof(BITMAPINFO) + 4 * sizeof(DWORD));
  bmi32->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
  bmi32->bmiHeader.biPlanes = 1;
  bmi32->bmiHeader.biBitCount = 32;
  bmi32->bmiHeader.biCompression = BI_BITFIELDS;
  ((DWORD*) bmi32->bmiColors)[0] = 0x00FF0000; /* red mask */
  ((DWORD*) bmi32->bmiColors)[1] = 0x0000FF00; /* green mask */
  ((DWORD*) bmi32->bmiColors)[2] = 0x000000FF; /* blue mask */

}

#endif /* NO_STANDARD_COLORS */

/****************************************************************************/
/*                     Window Setup                                         */
/****************************************************************************/

/* SetWindowTitle(): Set the main window title */
void SetWindowTitle()
{ 
  TCHAR titleString[MAX_PATH+20];

  if(!IsWindow(stWindow)) return;
  wsprintf(titleString,TEXT("Squeak! (%s)"), toUnicode(imageName));
  SetWindowText(stWindow,titleString);
}

void SetupWindows()
{ WNDCLASS wc;

  /* create our update region */
  updateRgn = CreateRectRgn(0,0,1,1);

  /* No windows at all when running as NT service */
  if(fRunService && !fWindows95) return;

  wc.style = CS_OWNDC; /* don't waste resources ;-) */
  wc.lpfnWndProc = (WNDPROC)MainWndProc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon (hInstance, MAKEINTRESOURCE(2));
  wc.hCursor = NULL;
  wc.hbrBackground = GetStockObject (WHITE_BRUSH);
  wc.lpszMenuName = NULL;
  wc.lpszClassName = TEXT("SqueakWindowClass");

  /* Due to Win95 bug we have to ignore the return value
     of RegisterClass(). Win95 returns several strange
     error messages (such as ERR_FUNCTION_NOT_SUPPORTED)
     when the class is already registered (which should
     result in an ERR_CLASS_ALREADY_EXISTS) */
  RegisterClass(&wc);

  stWindow = CreateWindowEx(WS_EX_APPWINDOW | WS_EX_OVERLAPPEDWINDOW,
                       TEXT("SqueakWindowClass"),
                       TEXT("Squeak!"),
                       WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
                       0,
                       0,
                       CW_USEDEFAULT,
                       CW_USEDEFAULT,
                       NULL,
                       NULL,
                       hInstance,
                       NULL);
#if defined(_WIN32_WCE)

	/* WinCE does not support RegisterClassEx(), so we must set 
	   the small icon after creating the window. */
	SendMessage(stWindow,WM_SETICON, FALSE, 
		(LPARAM)LoadImage(hInstance, MAKEINTRESOURCE(1), IMAGE_ICON, 16, 16, 0));

	consoleWindow = NULL; /* We do not use console under WinCE */

#else /* defined(_WIN32_WCE) */

  consoleWindow = CreateWindow (TEXT("EDIT"),
                       TEXT(""),
                       WS_CHILD | WS_BORDER | WS_HSCROLL | WS_VSCROLL | ES_MULTILINE,
                       0,
                       0,
                       CW_USEDEFAULT,
                       CW_USEDEFAULT,
                       stWindow,
                       NULL,
                       hInstance,
                       NULL);
#endif /* defined(_WIN32_WCE) */

  /* Modify the system menu for any VM options */
#ifndef NO_PREFERENCES
  { HMENU hMenu,pMenu;
    pMenu = CreatePopupMenu();
    AppendMenu(pMenu,MF_STRING | MF_DISABLED, 0,TEXT("------- Squeak VM Preferences -------"));
    AppendMenu(pMenu,MF_SEPARATOR, 0,NULL);
    AppendMenu(pMenu,MF_STRING | MF_UNCHECKED , 0x0070, TEXT("Use 3 button mouse mapping"));
    AppendMenu(pMenu,MF_SEPARATOR, 0,NULL);
    AppendMenu(pMenu,MF_STRING | (fDeferredUpdate ? MF_CHECKED : MF_UNCHECKED), 0x0020, TEXT("Defer display update"));
    AppendMenu(pMenu,MF_STRING | (fReduceCPUUsage ? MF_CHECKED : MF_UNCHECKED), 0x0060, TEXT("Reduce CPU usage"));
    AppendMenu(pMenu,MF_SEPARATOR, 0,NULL);
    AppendMenu(pMenu,MF_STRING | (fShowConsole ? MF_CHECKED : MF_UNCHECKED) , 0x0030, TEXT("Show output console"));
    AppendMenu(pMenu,MF_STRING | (fDynamicConsole ? MF_CHECKED : MF_UNCHECKED), 0x0050, TEXT("Show console on errors"));
    AppendMenu(pMenu,MF_SEPARATOR, 0,NULL);
#ifndef NO_PRINTER
    AppendMenu(pMenu,MF_STRING | MF_UNCHECKED , 0x0080, TEXT("Printer configuration ..."));
    AppendMenu(pMenu,MF_SEPARATOR, 0,NULL);
#endif /* NO_PRINTER */
    AppendMenu(pMenu,MF_STRING | MF_UNCHECKED , 0x0010, TEXT("Display version information"));
    vmPrefsMenu = pMenu;
    hMenu = GetSystemMenu(stWindow,false);
    AppendMenu(hMenu,MF_SEPARATOR, 0,NULL);
    AppendMenu(hMenu, MF_POPUP, (UINT) pMenu, TEXT("&VM Preferences"));
  }
#endif
  SetWindowTitle();
}


#if !defined(_WIN32_WCE)  /* Unused under WinCE */

void SetWindowSize(void) 
{ RECT r;
  int width, height, maxWidth, maxHeight;

  if (savedWindowSize != 0) 
    {
      width  = (unsigned) savedWindowSize >> 16;
      height = savedWindowSize & 0xFFFF;
    } 
  else 
    {
      width  = 640;
      height = 480;
    }

  /* minimum size is 64 x 64 */
  width  = ( width > 64) ?   width : 64;
  height = (height > 64) ?  height : 64;

  /* include borders, title bar, scrollbars etc. */
  r.left = 0; r.right = width; r.top = 0; r.bottom = height;
  AdjustWindowRect(&r,GetWindowLong(stWindow,GWL_STYLE),FALSE);
  width = r.right - r.top;
  height = r.bottom - r.top;

  /* maximum size is screen size */
  maxWidth  = GetSystemMetrics(SM_CXFULLSCREEN);
  maxHeight = GetSystemMetrics(SM_CYFULLSCREEN);
  width  = ( width <= maxWidth)  ?  width : maxWidth;
  height = (height <= maxHeight) ? height : maxHeight;

  if(!IsWindow(stWindow)) return; /* might happen if run as NT service */

  SetWindowPos(stWindow, 
  		NULL, 
  		(maxWidth-width) / 2, 
  		(maxHeight-height) / 2, 
  		width, 
  		height, 
  		SWP_NOZORDER | SWP_HIDEWINDOW);
}

#endif /* !defined(_WIN32_WCE) */

/****************************************************************************/
/*              Keyboard and Mouse                                          */
/****************************************************************************/

int recordVirtualKey(UINT message, WPARAM wParam, LPARAM lParam)
{ int keystate = 0;

  switch (wParam) {
    case VK_CANCEL:
      interruptPending= true;
      interruptCheckCounter= 0;
      break;
#ifndef NO_PREFERENCES
    case VK_F2:
      {
        TrackMenu(vmPrefsMenu);
        return 1;
      }
#endif /* NO_PREFERENCES */
    case VK_DELETE: keystate = 127; break;
    case VK_INSERT: keystate = 5; break;
    case VK_PRIOR: keystate = 11; break;
    case VK_NEXT : keystate = 12; break;
    case VK_END  : keystate = 4; break;
    case VK_HOME : keystate = 1; break;
    case VK_LEFT : keystate = 28; break;
    case VK_RIGHT: keystate = 29; break;
    case VK_UP   : keystate = 30; break;
    case VK_DOWN : keystate = 31; break;
    default: return 0;
  }
  keystate = keystate | ((buttonState >> 3) << 8);
  keyBuf[keyBufPut]= keystate;
  keyBufPut= (keyBufPut + 1) % KEYBUF_SIZE;
  if (keyBufGet == keyBufPut)
    {
      /* buffer overflow; drop the last character */
      keyBufGet= (keyBufGet + 1) % KEYBUF_SIZE;
      keyBufOverflows++;
    }
  return 1;
}

/* The following is a mapping to Mac Roman glyphs.
   It is not entirely correct since a number of glyphs are
   different but should be good enough for Squeak.
   More significantly, we can now map in both directions. */
static unsigned char keymap[256] =
{
  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15,
 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
 96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,
112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,
173,176,226,196,227,201,160,224,246,228,178,220,206,179,182,183,
184,212,213,210,211,165,208,209,247,170,185,221,207,186,189,217,
202,193,162,163,219,180,195,164,172,169,187,199,194,197,168,248,
161,177,198,215,171,181,166,225,252,218,188,200,222,223,240,192,
203,231,229,204,128,129,174,130,233,131,230,232,237,234,235,236,
245,132,241,238,239,205,133,249,175,244,242,243,134,250,251,167,
136,135,137,139,138,140,190,141,143,142,144,145,147,146,148,149,
253,150,152,151,153,155,154,214,191,157,156,158,159,254,255,216
};

/* The following is the inverse keymap */
static unsigned char iKeymap[256];

void SetupKeymap()
{ int i;
  for(i=0;i<256;i++)
    iKeymap[keymap[i]] = i;
}

int recordKeystroke(UINT msg, WPARAM wParam, LPARAM lParam)
{
  int keystate=0;
  /* Map from Win32 to Mac */
  keystate = keymap[wParam];
  /* add the modifiers */
  keystate = keystate | ((buttonState >> 3) << 8);
  /* check for interrupt key */
  if(keystate == interruptKeycode)
    {
      /* NOTE: Interrupt key is meta, not recorded as key stroke */
      interruptPending= true;
      interruptCheckCounter= 0;
	  return 1;
    }
  /* Add the key */
  keyBuf[keyBufPut]= keystate;
  keyBufPut= (keyBufPut + 1) % KEYBUF_SIZE;
  if (keyBufGet == keyBufPut)
    {
      /* buffer overflow; drop the last character */
      keyBufGet= (keyBufGet + 1) % KEYBUF_SIZE;
      keyBufOverflows++;
    }
}

/* record mouse events */
int recordMouseDown(WPARAM wParam, LPARAM lParam)
{
  int stButtons= 0;

#if defined(_WIN32_WCE)

	if (wParam & MK_LBUTTON) stButtons |= 4;
	if (stButtons == 4)	/* red button honours the modifiers */
	{				 
		if (wParam & MK_CONTROL) stButtons = 1;	/* blue button if CTRL down */
		else if (GetKeyState(VK_LMENU) & 0x8000) stButtons = 2;	/* yellow button if META down */
	}

#else /* defined(_WIN32_WCE) */

  if(GetKeyState(VK_LBUTTON) & 0x8000) stButtons |= 4;
  if(GetKeyState(VK_MBUTTON) & 0x8000) stButtons |= f3ButtonMouse ? 2 : 1;
  if(GetKeyState(VK_RBUTTON) & 0x8000) stButtons |= f3ButtonMouse ? 1 : 2;

  if (stButtons == 4)	/* red button honours the modifiers */
    {		     
      if (GetKeyState(VK_CONTROL) & 0x8000)
        stButtons= 1;	/* blue button if CTRL down */
      else if (GetKeyState(VK_LMENU) & 0x8000)
        stButtons= 2;	/* yellow button if META down */
    }

#endif /* defined(_WIN32_WCE) */

  buttonState = stButtons & 0x7;
  return 1;
}

/* record the modifier buttons */
int recordModifierButtons()
{ int modifiers=0;

  /* map both shift and caps lock to squeak shift bit */
  if(GetKeyState(VK_SHIFT) & 0x8000)
    modifiers |= 1 << 3;
  if(GetKeyState(VK_CAPITAL) & 0x8000)
    modifiers |= 1 << 3;

  /* Map meta key to command.
     
     NOTE:
     Non US keyboards typically need the right menu key for special mappings.
     However, it seems that hitting this key is interpreted as Alt-Ctrl.
     We therefore ignore the control modifier if the right menu key 
     is pressed (hoping that nobody will use the right-menu / control combination)
     
     NOTE^2:
     Due to another Win95 bug, we can not ask for the left
     or right alt key directly (Win95 returns always 0 if
     asked for VK_LMENU or VK_RMENU). Therefore we're using
     the Ctrl-Alt combination for detecting the right menu key */

  /* if CTRL and not ALT pressed use control modifier */
  if((GetKeyState(VK_CONTROL) & 0x8000) && (GetKeyState(VK_MENU) & 0x8000) == 0)
    modifiers |= 1 << 4;

  /* if ALT and not CTRL pressed use command modifier */
  if((GetKeyState(VK_MENU) & 0x8000) && (GetKeyState(VK_CONTROL) & 0x8000) == 0)
    modifiers |= 1 << 6;

  /* if ALT and CTRL pressed use option modifier */
  if((GetKeyState(VK_MENU) & 0x8000) && (GetKeyState(VK_CONTROL) & 0x8000))
    modifiers |= 1 << 5;

  /* button state: low three bits are mouse buttons; next 4 bits are modifier bits */
  buttonState= modifiers | (buttonState & 0x7);
  return 1;
}

int ioGetButtonState(void)
{
  if(fReduceCPUUsage)
    { MSG msg;
      /* Delay execution of squeak if
         - there is currently no button pressed
         - there is currently no mouse event in the input queue
         by at most 5 msecs. This will reduce cpu load during idle times.
      */
      if((buttonState & 0x7) == 0 &&
        !PeekMessage(&msg, stWindow, WM_MOUSEFIRST, WM_MOUSELAST, PM_NOREMOVE))
          MsgWaitForMultipleObjects(0,NULL,0,5, QS_MOUSE);
    }
  ioProcessEvents();  /* process all pending events */
  return buttonState;
}

int ioGetKeystroke(void)
{
  int keystate;

  ioProcessEvents();  /* process all pending events */
  if (keyBufGet == keyBufPut)
    return -1;  /* keystroke buffer is empty */

  keystate= keyBuf[keyBufGet];
  keyBufGet= (keyBufGet + 1) % KEYBUF_SIZE;
  /* set modifer bits in buttonState to reflect the last keystroke fetched */
  buttonState= ((keystate >> 5) & 0xF8) | (buttonState & 0x7);
  return keystate;
}

/****************************************************************************/
/*              Misc support primitves                                      */
/****************************************************************************/

int ioExit(void)
{
  exit(0);
  /* avoid the warnings here */
  return 0;
}

int ioBeep(void)
{
  MessageBeep(0);
  return 1;
}

int ioMSecs()
{
  /* Make sure the value fits into Squeak SmallIntegers */
  return GetTickCount() &0x3FFFFFFF;
}

/* Note: ioMicroMSecs returns *milli*seconds */
int ioMicroMSecs(void)
{
  /* Make sure the value fits into Squeak SmallIntegers */
  return timeGetTime() &0x3FFFFFFF;
}

/* Note: ioRelinquishProcessorForMicroseconds has *micro*seconds  as argument*/
int ioRelinquishProcessorForMicroseconds(int microSeconds) 
{
  MsgWaitForMultipleObjects(0,NULL,0,microSeconds / 1000, QS_ALLINPUT);
	return microSeconds;
}

int ioMousePoint(void)
{ 
  if(fReduceCPUUsage)
    { MSG msg;
      /* Delay execution of squeak if
         - there is currently no button pressed
         - there is currently no mouse event in the input queue
         by at most 5 msecs. This will reduce cpu load during idle times.
      */
      if((buttonState & 0x7) == 0 &&
        !PeekMessage(&msg, stWindow, WM_MOUSEFIRST, WM_MOUSELAST, PM_NOREMOVE))
          MsgWaitForMultipleObjects(0,NULL,0,5, QS_MOUSE);
    }
  ioProcessEvents();  /* process all pending events */
  /* x is high 16 bits; y is low 16 bits */
  return (mousePosition.x << 16) | (mousePosition.y);
}

int ioPeekKeystroke(void)
{
  int keystate;

  ioProcessEvents();  /* process all pending events */

  if (keyBufGet == keyBufPut)
    return -1;  /* keystroke buffer is empty */

  keystate= keyBuf[keyBufGet];
  /* set modifer bits in buttonState to reflect the last keystroke peeked at */
  buttonState= ((keystate >> 5) & 0xF8) | (buttonState & 0x7);
  return keystate;
}

int ioProcessEvents(void)
{ MSG msg;

  if(fRunService && !fWindows95) return 1;
	/* WinCE doesn't retrieve WM_PAINTs from the queue with PeekMessage,
	   so we won't get anything painted unless we use GetMessage() if there
	   is a dirty rect. */
  while(PeekMessage(&msg,NULL,0,0,PM_NOREMOVE))
    {
      GetMessage(&msg,NULL,0,0);
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }

  return 1;
}

/* returns the size of the Squeak window */
int ioScreenSize(void)
{ RECT r;

  if(!IsWindow(stWindow)) return savedWindowSize;
  GetClientRect(stWindow,&r);
  /* width is high 16 bits; height is low 16 bits */
  return MAKELONG(r.bottom,r.right);
}

/* returns the local wall clock time */
int ioSeconds(void)
{ SYSTEMTIME sysTime;
  GetLocalTime(&sysTime);
  return convertToSqueakTime(sysTime);
}

int ioSetCursorWithMask(int cursorBitsIndex, int cursorMaskIndex, int offsetX, int offsetY)
{
#if !defined(_WIN32_WCE)
	/****************************************************/
	/* Only one cursor is defined under CE...           */
	/* (the wait cursor)                         :-(    */
	/****************************************************/
  static unsigned char *andMask=0,*xorMask=0;
  static int cx=0,cy=0,cursorSize=0;
  int i;

  if(!IsWindow(stWindow)) return 1;
  if(!andMask || !xorMask)
	{ /* Determine the cursor size.
		 NOTE: On NT this is actually not necessary, as NT _can_
		 create cursors of different sizes. However, on Win95 the
		 cursor will be created, BUT WILL NOT BE SHOWN! */
      cx = GetSystemMetrics(SM_CXCURSOR);
      cy = GetSystemMetrics(SM_CYCURSOR);
      cursorSize = cx*cy / 8;
      andMask = malloc(cursorSize);
      xorMask = malloc(cursorSize);
  }

  /* free last used cursor */
  if(currentCursor) DestroyCursor(currentCursor);

  memset(andMask,0xff,cursorSize);
  memset(xorMask,0x00,cursorSize);
  if(cursorMaskIndex)
    { /* New Cursors specify mask and bits:
         Mask    Bit    Result
         0       0      Transparent
         0       1      Invert
         1       0      White
         1       1      Black
      */
      for (i=0; i<16; i++)
        {
          andMask[i*cx/8+0] = ~(checkedLongAt(cursorMaskIndex + (4 * i)) >> 24) & 0xFF;
          andMask[i*cx/8+1] = ~(checkedLongAt(cursorMaskIndex + (4 * i)) >> 16) & 0xFF;
        }
      for (i=0; i<16; i++)
        {
          xorMask[i*cx/8+0] = (~(checkedLongAt(cursorBitsIndex + (4 * i)) >> 24) & 0xFF) ^ (andMask[i*cx/8+0]);
          xorMask[i*cx/8+1] = (~(checkedLongAt(cursorBitsIndex + (4 * i)) >> 16) & 0xFF) ^ (andMask[i*cx/8+1]);
        }
    }
  else /* Old Cursor: Just make all 1-bits black */
    for (i=0; i<16; i++)
      {
        andMask[i*cx/8+0] = ~(checkedLongAt(cursorBitsIndex + (4 * i)) >> 24) & 0xFF;
        andMask[i*cx/8+1] = ~(checkedLongAt(cursorBitsIndex + (4 * i)) >> 16) & 0xFF;
      }

  currentCursor = CreateCursor(hInstance,-offsetX,-offsetY,cx,cy,andMask,xorMask);
  if(currentCursor) 
    {
      SetCursor(0);
      SetCursor(currentCursor);
    }
  else
    {
      printLastError(TEXT("CreateCursor failed"));
      fflush(stderr);
    }
#endif /* !defined(_WIN32_WCE) */
  return 1;
}

int ioSetCursor(int cursorBitsIndex, int offsetX, int offsetY)
{
  return ioSetCursorWithMask(cursorBitsIndex, 0, offsetX, offsetY);
}


int ioSetFullScreen(int fullScreen)
{ static int wasFullScreen = 0;

  if(!IsWindow(stWindow)) return 1;
  if(wasFullScreen == fullScreen) return 1;
  /* NOTE: No modifications if the window is not currently
           visible, else we'll have a number of strange effects ... */
  if(!IsWindowVisible(stWindow)) return 1;
  if(fullScreen)
    {
#if !defined(_WIN32_WCE)
      SetWindowLong(stWindow,GWL_STYLE, WS_POPUP | WS_CLIPCHILDREN);
      ShowWindow(stWindow, SW_SHOWMAXIMIZED);
#else /* !defined(_WIN32_WCE) */
      ShowWindow(stWindow,SW_SHOWNORMAL);
#endif /* !defined(_WIN32_WCE) */
      fullScreenFlag = 1;
    }
  else
    {
#if !defined(_WIN32_WCE)
      ShowWindow(stWindow, SW_RESTORE);
      SetWindowLong(stWindow,GWL_STYLE, WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN);
      SetWindowPos(stWindow,0,0,0,0,0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOZORDER | SWP_FRAMECHANGED | SWP_NOREDRAW);
#endif /* !defined(_WIN32_WCE) */
      ShowWindow(stWindow,SW_SHOWNORMAL);
      fullScreenFlag = 0;
    }
  wasFullScreen = fullScreen;
  return 1;
}


/****************************************************************************/
/*                           Image byte reversal                            */
/****************************************************************************/
#ifndef NO_STD_BYTE_REVERSAL
int reverse_image_bytes(int *ptr,int d, int w,int h,int x0,int y0,int x1,int y1)
{ int scanLine, pixPerWord;
  int startX, stopX, startY, stopY;
  int *linePtr, *pixPtr;
  int log2Depth,pixPerWordShift;
  int j,xLen;
  /* note: this function should only be called for a d<=8 */
  switch (d) {
  case 8: log2Depth=3; break;
  case 4: log2Depth=2; break;
  case 2: log2Depth=1; break;
  case 1: log2Depth=0; break;
  default: return 0;  /* don't do anything distructive if the person calling us is confused */
  }
  pixPerWordShift = 5-log2Depth;
  pixPerWord= 1 << pixPerWordShift;  /* was = 32/depth */
  scanLine= (w+pixPerWord-1) >> pixPerWordShift;  /* words per scan line */
  startX = (x0 >> pixPerWordShift);
  stopX  = (x1 + pixPerWord -1) >> pixPerWordShift;
  xLen = stopX - startX;
  startY = y0 * scanLine;
  stopY  = y1 * scanLine;
  if(stopX <= startX || stopY <= startY) return 1;
  for(j = startY; j < stopY; j += scanLine)
    {
      linePtr = ptr + j;
      pixPtr = linePtr + startX;
      { int i;
        for (i=xLen; i--; pixPtr++)
          *pixPtr = byteSwapped(*pixPtr);
      }
    }
  return 1;
}

int reverse_image_words(int *ptr,int d, int w,int h,int x0,int y0,int x1,int y1)
{ int scanLine, pixPerWord;
  int startX, stopX, startY, stopY;
  int *linePtr,*pixPtr;
  int j,xLen;
  const int log2Depth=4;    /* log2Depth = log2(d) = log2(16) */
  const int pixPerWordShift = 5-log2Depth;

  if (d!=16)
	return 0;  /* The person calling us is confused.  We only deal with 16 bit words */

  pixPerWord= 1 << pixPerWordShift;  /* was = 32/depth */
  scanLine= (w+(pixPerWord-1)) >> pixPerWordShift;  /* words per scan line */
  startX = (x0 >> pixPerWordShift);
  stopX  = (x1 >> pixPerWordShift) + 1;
  xLen = stopX - startX;
  startY = y0 * scanLine;
  stopY  = y1 * scanLine;
  if(stopX <= startX || stopY <= startY) return 1;
  for(j = startY; j < stopY; j += scanLine)
    {
      linePtr = ptr + j;
      pixPtr = linePtr + startX;
#define SWAPWORDS(l) (((((l) & 0xFFFF) << 16) | ((l) >> 16)))
      { int i;
        for (i=xLen; i--; pixPtr++)
          *pixPtr = SWAPWORDS(*pixPtr);
      }
#undef SWAPWORDS
    }
  return 1;
}

#endif /* NO_STD_BYTE_REVERSAL */

/****************************************************************************/
/*              Display and printing                                        */
/****************************************************************************/

BITMAPINFO *BmiForDepth(int depth)
{ BITMAPINFO *bmi = NULL;
  switch(depth) {
    case 1: bmi = bmi1; break;
    case 4: bmi = bmi4; break;
    case 8: bmi = bmi8; break;
    case 16: bmi = bmi16; break;
    case 32: bmi = bmi32; break;
  }
  return bmi;
}

/* force an update of the squeak window if using deferred updates */
int ioForceDisplayUpdate(void) {
  /* Check if
     a) We should do deferred updates at all
     b) The window is valid
     c) The Interpreter does not defer updates by itself
  */
	if(fDeferredUpdate && IsWindow(stWindow) && !deferDisplayUpdates)
    {
      UpdateWindow(stWindow);
    }
}

int ioFormPrint(int bitsAddr, int width, int height, int depth, double hDPI, double vDPI, int landscapeFlag)
	/* print a form with the given bitmap, width, height, and depth at
	   the given horizontal and vertical scales in the given orientation */
{
#ifdef NO_PRINTER
  warnPrintf(TEXT("This VM does not support printing.\n"));
  return success(false);
#else /* !defined(NO_PRINTER) */
  DEVMODE *dmPtr;
  BITMAPINFO *bmi;
  DOCINFO di;
  HDC dc;
  DEVNAMES *devNames;
  TCHAR *namePtr;
  int scWidth;
  int scHeight;

  devNames = GlobalLock(printValues.hDevNames);
  if(!devNames)
    {
      warnPrintf(TEXT("No printer configured\n"));
      return false;
    }
  dmPtr = GlobalLock(printValues.hDevMode);
  if(dmPtr)
    {
      dmPtr->dmOrientation = landscapeFlag ? DMORIENT_LANDSCAPE : DMORIENT_PORTRAIT;
      dmPtr->dmFields |= DM_ORIENTATION;
    }
  namePtr = (TCHAR*) devNames;
  dc = CreateDC(namePtr+devNames->wDriverOffset, 
                namePtr+devNames->wDeviceOffset, 
                namePtr+devNames->wOutputOffset,
                dmPtr);
  GlobalUnlock(printValues.hDevMode);
  GlobalUnlock(printValues.hDevNames);

  if(!dc)
    {
      warnPrintf(TEXT("Unable to open printer.\n"));
      return false;
    }

  bmi = BmiForDepth(depth);
  if(!bmi) 
    {
      warnPrintf(TEXT("Color depth %d not supported"), depth);
      return false;
    }

  di.cbSize      = sizeof(DOCINFO);
  di.lpszDocName = TEXT("Squeak Print Job");
  di.lpszOutput  = NULL;

  StartDoc  (dc, &di);
  StartPage (dc);

  bmi->bmiHeader.biWidth = width;
  bmi->bmiHeader.biHeight = height;
  bmi->bmiHeader.biSizeImage = 0;

  /* set device coords to 0.001 inch per unit */

  SetMapMode(dc, MM_HIENGLISH);
  scWidth = (int)(width * 1000.0 / hDPI);
  scHeight = (int)(height * 1000.0 / vDPI );

  /* reverse the image bits if necessary */
#ifndef NO_BYTE_REVERSAL
  if( depth < 32 )
    if(depth == 16)
      reverse_image_words((int*) bitsAddr, depth, width, height, 0, 0, width, height);
    else
      reverse_image_bytes((int*) bitsAddr, depth, width, height, 0, 0, width, height);
#endif /* NO_BYTE_REVERSAL */

  if(GDI_ERROR == StretchDIBits(dc,
                0,         /* dst_x */
                -scHeight, /* dst_y --- mapping mode changes positive y-axis */
                scWidth,   /* dst_w */
                scHeight,  /* dst_h */
                0,         /* src_x */
                0,         /* src_y */
                width,     /* src_w */
                height,    /* src_h */
                (void*) bitsAddr, /* bits */
                bmi,
                DIB_RGB_COLORS,
                SRCCOPY)) printLastError(TEXT("StretchDIBits failed"));

  /* reverse the image bits if necessary */
#ifndef NO_BYTE_REVERSAL
  if( depth < 32 )
    if(depth == 16)
      reverse_image_words((int*) bitsAddr, depth, width, height, 0, 0, width, height);
    else
      reverse_image_bytes((int*) bitsAddr, depth, width, height, 0, 0, width, height);
#endif /* NO_BYTE_REVERSAL */

  EndPage   (dc);
  EndDoc    (dc);
  DeleteDC  (dc);

	return true;
#endif /* NO_PRINTER */
}


int ioShowDisplay(int dispBitsIndex, int width, int height, int depth,
		  int affectedL, int affectedR, int affectedT, int affectedB)
{ HDC dc;
  BITMAPINFO *bmi;
  int lines;

  if(!IsWindow(stWindow))
    return 0;

  if(affectedR < affectedL || affectedT > affectedB)
    return 1;

  /* Try to accumulate the changes */
  if(!updateRightNow)
    {
      updateRect.left = affectedL;
      updateRect.top = affectedT;
      updateRect.right = affectedR;
      updateRect.bottom = affectedB;
      /* Acknowledge the request for deferred updates only 
         if the interpreter is not deferring these by itself */
      if(fDeferredUpdate && !deferDisplayUpdates)
        {
          /* Wait until the next WM_PAINT gets processed */
          InvalidateRect(stWindow,&updateRect,FALSE);
          return 1;
        }
      else
        /* Set the update region manually */
        SetRectRgn(updateRgn, updateRect.left, updateRect.top,updateRect.right,updateRect.bottom);
    }
  else
    {
      /* After a paint, so get affected area from update rect */
      updateRightNow = FALSE;
    }
  if (IsRectEmpty(&updateRect)) return 1; /* Nothing to do */

  affectedL = updateRect.left;
  affectedT = updateRect.top;
  affectedR = updateRect.right;
  affectedB = updateRect.bottom;

  /* Careful here: 
     After resizing the main window the affected area can
     be larger than the area covered by the display bits ... */
  if (affectedR > width) affectedR= width;
  if (affectedB > height) affectedB= height;
  /* ... and don't forget left and top - else reverse_image_* will crash */
  if (affectedL > width) affectedL= width;
  if (affectedT > height) affectedT= height;

  /* Don't draw empty areas */
  if(affectedL == affectedR || affectedT == affectedB) return 1;

#if defined(_WIN32_WCE)
  /******************************************************/
	/* Windows CE version, using DIBSection               */
	/* (does not support palettes or SetDIBitsToDevice()) */
  /******************************************************/
  {
	void* pBits;
	HDC memDC;

	/* Not after a paint event occurred, so update rect is specified by parameters */
	updateRect.left = affectedL;
	updateRect.top = affectedT;
	updateRect.right = affectedR;
	updateRect.bottom = affectedB;

	dc = GetDC(stWindow);
	if (!dc)
		error("GetDC() failed");

	memDC = createBitmapDC(dc, depth, width, height, &pBits);
	/* Reverse the affected area out of the squeak bitmap, into the DIBSection */

	PROFILE_BEGIN(PROFILE_DISPLAY)
	reverse_image_bytesCE(pBits,(DWORD*)dispBitsIndex, depth, width, &updateRect);
	PROFILE_END(ticksForReversal)

	PROFILE_BEGIN(PROFILE_DISPLAY);
	BitBlt(dc, 
			updateRect.left, 						/* dst_x */
			updateRect.top,							/* dst_y */
			(updateRect.right - updateRect.left),	/* dst_w */
			(updateRect.bottom - updateRect.top),	/* dst_h */
			memDC,
			updateRect.left,						/* src_x */
			updateRect.top,			 				/* src_y */
			SRCCOPY);

	releaseBitmapDC(memDC);
	ReleaseDC(stWindow,dc);
	PROFILE_END(ticksForBlitting);
  }
#else /* defined(_WIN32_WCE) */

  bmi = BmiForDepth(depth);
  if(!bmi)
    {
      abortMessage(TEXT("Aborting Squeak!!!!\nColor depth %d not supported"),depth);
    }

  /* reverse the image bits if necessary */
#ifndef NO_BYTE_REVERSAL
  PROFILE_BEGIN(PROFILE_DISPLAY)
  if( depth < 32 )
    if(depth == 16)
      reverse_image_words((int*) dispBitsIndex, depth, width, height, affectedL, affectedT, affectedR, affectedB);
    else
      reverse_image_bytes((int*) dispBitsIndex, depth, width, height, affectedL, affectedT, affectedR, affectedB);
  PROFILE_END(ticksForReversal)
#endif NO_BYTE_REVERSAL

  bmi->bmiHeader.biWidth = width;
  bmi->bmiHeader.biHeight = -height;
  bmi->bmiHeader.biSizeImage = 0;

  dc = GetDC(stWindow);
  SelectPalette(dc,palette,0);
  RealizePalette(dc);


  PROFILE_BEGIN(PROFILE_DISPLAY)
  /* use the actual affected region rather than the complete recangle for clipping */
  SelectClipRgn(dc,updateRgn);
  lines = SetDIBitsToDevice(dc,
  	    0, 		    /* dst_x */
  	    0,		    /* dst_y */
  	    width,    /* dst_w */
  	    height,   /* dst_h */
  	    0,		    /* src_x */
  	    0, 		    /* src_y */
  	    0, 			  /* start scan line in DIB */
  	    height,		/* num scan lines in DIB */
	    (void*) dispBitsIndex,  /* bits */
	    bmi,
	    DIB_RGB_COLORS);
  SelectClipRgn(dc,NULL);
  PROFILE_END(ticksForBlitting)

  ReleaseDC(stWindow,dc);

  if(lines == 0)
    printLastError(TEXT("SetDIBitsToDevice failed"));

  /* reverse the image bits if necessary */
#ifndef NO_BYTE_REVERSAL
  PROFILE_BEGIN(PROFILE_DISPLAY)
  if( depth < 32 )
    if(depth == 16)
      reverse_image_words((int*) dispBitsIndex, depth, width, height, affectedL, affectedT, affectedR, affectedB);
    else
      reverse_image_bytes((int*) dispBitsIndex, depth, width, height, affectedL, affectedT, affectedR, affectedB);
  PROFILE_END(ticksForReversal)
#endif /* NO_BYTE_REVERSAL */
#endif /* defined(_WIN32_WCE) */
}

/****************************************************************************/
/*                      Clipboard                                           */
/****************************************************************************/

int clipboardSize(void)
{ HANDLE h;
  unsigned char *src;
  int len,tel;

  if(!IsClipboardFormatAvailable(CF_TEXT)) /* check for format CF_TEXT */
    return 0;
  if(!OpenClipboard(stWindow))
    return 0;
  h = GetClipboardData(CF_TEXT);
  src = GlobalLock(h);
  len = src ? strlen(src) : 0;
  /* RvL 17-04-1998
     Look for CrLf sequences */
  tel = len;
  while(tel--)
    if((*src++ == 0x0d) && (*src == 0x0a))
      len--;
  GlobalUnlock(h);
  CloseClipboard();
  return len;
}
    
/* send the given string to the clipboard */
int clipboardWriteFromAt(int count, int byteArrayIndex, int startIndex)
{ HANDLE h;
  unsigned char *dst,*src;
  int adjCount, tel;

  if(!OpenClipboard(stWindow))
    return 0;

  /* RvL 17-04-1998
     adjust count for inserting 0A into clipboard string */
  src = (unsigned char *)byteArrayIndex + startIndex;
  tel = count;
  adjCount = 0;
  while(tel--)
    if ((*src++ == 0x0D) && (*src != 0x0A))
      adjCount++;

  h = GlobalAlloc(GMEM_MOVEABLE | GMEM_DDESHARE, count+1+adjCount);
  dst = GlobalLock(h);
  src = (unsigned char *)byteArrayIndex + startIndex;
  /* Remap from Mac Roman to Ansi Latin */
  while(count--)
    if((*src == 0x0D) && (src[1] != 0x0A))
      { /* special case: crlf translation */
        *dst++ = *src++;
        *dst++ = 0x0A;
      }
    else
      { /* regular case: lookup translation */
        *dst++ = iKeymap[*src++];
      }
  *dst = 0;
  EmptyClipboard();
  SetClipboardData(CF_TEXT, h);
  /* Note: After setting clipboard data, 
     the memory block previously allocated belongs to
     the clipboard - not to the app. */
  CloseClipboard();
  return 1;
}


/* transfer the clipboard data into the given byte array */
int clipboardReadIntoAt(int count, int byteArrayIndex, int startIndex)
{ HANDLE h;
  unsigned char *src,*dst;
  int len;

  if(!IsClipboardFormatAvailable(CF_TEXT)) /* check for format CF_TEXT */
    return 0;
  if(!OpenClipboard(stWindow))
    return 0;
  dst= (unsigned char *)byteArrayIndex + startIndex;
  h = GetClipboardData(CF_TEXT);
  src = GlobalLock(h);
  len = src ? strlen(src) : 0;
  if(len > count) len = count;
  /* Remap from Ansi Latin to Mac Roman */
  while(len--)
    if((*src == 0x0D) && (src[1] == 0x0A))
      { /* RvL 17-04-1998
           drop the line feed after a carriage return
           but leave lone line feeds alone
           may transfer less than 'len' bytes but who cares
           i.e. if they use the clipboardSize there should
           be no problem.*/
        *dst++ = *src++;
        src++;
      }
    else
      { /* regular case: lookup translation */
        *dst++ = keymap[*src++];
      }
  GlobalUnlock(h);
  CloseClipboard();
  return len;
}


/****************************************************************************/
/*                          Profiling                                       */
/****************************************************************************/

int clearProfile(void) { return 1;}
int dumpProfile(void) {return 1;}
int startProfiling(void) {return 1;}
int stopProfiling(void) {return 1;}


/****************************************************************************/
/*                    Image / VM File Naming                                */
/****************************************************************************/

int vmPathSize(void)
{
  return lstrlen(vmPath);
}

int vmPathGetLength(int sqVMPathIndex, int length)
{
  char *stVMPath= (char *)sqVMPathIndex;
  int count, i;

  count= lstrlen(vmPath);
  count= (length < count) ? length : count;

  /* copy the file name into the Squeak string */
  for (i= 0; i < count; i++)
    stVMPath[i]= (char) vmPath[i]; /* will remove leading zeros from unicode */

  return count;
}

int imageNameSize(void)
{
  return strlen(imageName);
}

int imageNameGetLength(int sqImageNameIndex, int length)
{
  char *sqImageName= (char *)sqImageNameIndex;
  int count, i;

  count= strlen(imageName);
  count= (length < count) ? length : count;

  /* copy the file name into the Squeak string */
  for (i= 0; i < count; i++)
    sqImageName[i]= imageName[i];

  return count;
}

int imageNamePutLength(int sqImageNameIndex, int length)
{
  char *sqImageName= (char *)sqImageNameIndex;
  char tmpImageName[MAX_PATH+1];
  char *tmp;
  int count, i;

  count= (IMAGE_NAME_SIZE < length) ? IMAGE_NAME_SIZE : length;

  /* copy the file name into a null-terminated C string */
  for (i= 0; i < count; i++)
    tmpImageName[i]= sqImageName[i];

  tmpImageName[count]= 0;

  /* Note: We have to preserve the fully qualified image path */
  tmp = strrchr(tmpImageName,'\\');
  if(tmp)
    { /* fully qualified */
      strcpy(imageName,tmpImageName);
    }
  else
    { /* not qualified */
      tmp = strrchr(imageName,'\\');
      if(!tmp)
        strcpy(imageName,tmpImageName);
      else
        {
          tmp++; *tmp = 0;
          count = IMAGE_NAME_SIZE - (tmp-imageName);
          if(count < length)
            tmpImageName[count] = 0;
          strcat(imageName,tmpImageName);
        }
    }
  SetWindowTitle();
  return 1;
}

/****************************************************************************/
/*                      System Attributes                                   */
/****************************************************************************/
char * GetAttributeString(int id) {
	/* This is a hook for getting various status strings back from
	   the OS. In particular, it allows Squeak to be passed arguments
	   such as the name of a file to be processed. Command line options
	   could be reported this way as well.
	*/
  /* id == 0 : return the full name of the VM */
  if(id == 0) return fromUnicode(vmName);
  /* 0 < id <= 1000 : return options of the image (e.g. given *after* the image name) */
  if(id > 0 && id <= 1000)
    return GetImageOption(id-1);
  /* id < 0 : return options of the VM (e.g. given *before* the image name) */
  if(id < 0)
    return GetVMOption(0-id-1);
  /* special attributes */
  if(id == 1001) /* Primary OS type */
      return WIN32_NAME;
  if(id == 1002) /* Secondary OS type */
      return WIN32_OS_NAME;
  if(id == 1003) /* Processor type */
      return WIN32_PROCESSOR_NAME;
  return "";
}

int attributeSize(int id) {
	return strlen(GetAttributeString(id));
}

int getAttributeIntoLength(int id, int byteArrayIndex, int length) {
	char *srcPtr, *dstPtr, *end;
	int charsToMove;

	srcPtr = GetAttributeString(id);
	charsToMove = strlen(srcPtr);
	if (charsToMove > length) {
		charsToMove = length;
	}

	dstPtr = (char *) byteArrayIndex;
	end = srcPtr + charsToMove;
	while (srcPtr < end) {
		*dstPtr++ = *srcPtr++;
	}
	return charsToMove;
}


/****************************************************************************/
/*                      File Startup                                        */
/****************************************************************************/

static TCHAR *lstrrchr(TCHAR *source, TCHAR c)
{ TCHAR *tmp;

  /* point to the last char */
  tmp = source + lstrlen(source)-1;
  while(tmp >= source)
    if(*tmp == c) return tmp;
    else tmp--;
  return NULL;
}

#if defined(_WIN32_WCE)
	/* WinCE does not support short file names, and has
	   no concept of a current directory. Space is at a
	   premium, the file system is small, and we are unlikely
	   to have a full sources file anyway (too big). All these
	   factors means that we stick with a simpler scheme, of
	   either requiring the image name to be fully pathed, or
	   if not, popping up a file open dialog */

void SetupFilesAndPath()
{
}

#else /* defined(_WIN32_WCE) */

void
LongFileNameFromPossiblyShortName(TCHAR *nameBuffer)
{ TCHAR oldDir[MAX_PATH+1];
  TCHAR testName[13];
  TCHAR nameBuf[MAX_PATH+1];
  TCHAR *shortName;
  WIN32_FIND_DATA findData;
  HANDLE findHandle;

  GetCurrentDirectory(MAX_PATH,oldDir);
  shortName = lstrrchr(nameBuffer,U_BACKSLASH[0]);
  if(!shortName) shortName = lstrrchr(nameBuffer,U_SLASH[0]);
  if(!shortName) return;
  /* if the file name is longer than 8.3
     this can't be a short name */
  *(shortName++) = 0;
  if(lstrlen(shortName) > 12)
    goto notFound;

  /* back up the old and change to the given directory,
     this makes searching easier */
  lstrcpy(nameBuf, nameBuffer);
  lstrcat(nameBuf,TEXT("\\"));
  SetCurrentDirectory(nameBuf);

  /* now search the directory */
  findHandle = FindFirstFile(TEXT("*.*"),&findData);
  if(findHandle == INVALID_HANDLE_VALUE) goto notFound; /* nothing found */
  do {
    if(lstrcmp(findData.cFileName,TEXT("..")) && lstrcmp(findData.cFileName,TEXT(".")))
      lstrcpy(testName,findData.cAlternateFileName);
    else
      *testName = 0;
    if(lstrcmp(testName,shortName) == 0) /* gotcha! */
      {
        FindClose(findHandle);
        /* recurse down */
        lstrcpy(nameBuf, findData.cFileName);
        goto recurseDown;
      }
  } while(FindNextFile(findHandle,&findData) != 0);
  /* nothing appropriate found */
  FindClose(findHandle);
notFound:
  lstrcpy(nameBuf, shortName);
recurseDown:
  /* recurse down */
  LongFileNameFromPossiblyShortName(nameBuffer);
  lstrcat(nameBuffer,TEXT("\\"));
  lstrcat(nameBuffer,nameBuf);
  SetCurrentDirectory(oldDir);
}

void
SetupFilesAndPath()
{ TCHAR *tmp;
  TCHAR tmpName[MAX_PATH+1];
  TCHAR imageNameW[MAX_PATH+1];

  /* get the full path for the image */
  lstrcpy(tmpName, toUnicode(imageName));
  GetFullPathName(tmpName,MAX_PATH,imageNameW,&tmp);
  /* check if it is a short name and if so convert 
     it to the full name */
  LongFileNameFromPossiblyShortName(imageNameW);

  /* and copy back to a C string */
  strcpy(imageName, fromUnicode(imageNameW));


  /* get the VM directory */
  lstrcpy(vmPath, vmName);
  tmp = lstrrchr(vmPath,U_BACKSLASH[0]);
  if(tmp) *tmp = 0;
  else GetCurrentDirectory(MAX_PATH,vmPath);
  lstrcat(vmPath,U_BACKSLASH);

#ifndef NO_PREFERENCES
  /* make ini file */
  lstrcpy(iniName, vmPath);
  lstrcat(iniName,TEXT("Squeak.ini"));
#endif
}

#endif /* !defined(_WIN32_WCE) */

/* SqueakImageLength(): 
   Return the length of the image if it is a valid Squeak image file.
   Otherwise return 0. */
DWORD SqueakImageLength(TCHAR *fileName)
{ DWORD dwRead, dwSize, magic = 0;
  HANDLE hFile;

  /* open image file */
  hFile = CreateFile(fileName, GENERIC_READ, FILE_SHARE_READ, 
                     NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if(hFile == INVALID_HANDLE_VALUE) 
    return 0;
  /* read magic number */
  if(!ReadFile(hFile, &magic, 4, &dwRead, NULL))
    {
      CloseHandle(hFile);
      return 0;
    }
  /* get the image's size */
  dwSize = GetFileSize(hFile, NULL);
  CloseHandle(hFile);
  if(!readableFormat(magic) && !readableFormat(byteSwapped(magic)))
    return 0;
  else
    return dwSize;
}

/****************************************************************************/
/*                      Usage of Squeak                                     */
/****************************************************************************/

/* print usage with different output levels */
int printUsage(int level)
{
  switch(level) {
    case 0: /* No command line given */
      abortMessage(TEXT("Usage: squeak [options] <imageFile>"));
      break;
    case 1: /* full usage */
      abortMessage(TEXT("%s"),
                   TEXT("Usage: squeak [vmOptions] imageFile [imageOptions]\n\n")
                   TEXT("vmOptions:\n")
                   TEXT("\t-service: ServiceName \t(install Squeak as NT service)\n")
                   TEXT("\t-headless \t\t(force Squeak to run headless)\n")
                   TEXT("\t-log: LogFile \t\t(use LogFile for VM messages)\n")
                   TEXT("\t-memory: megaByte \t(set memory to megaByte MB)"));
      break;
    case 2: /* No image found */
    default:
      abortMessage(
        TEXT("Could not open the Squeak image file '%s'\n\n")
        TEXT("There are several ways to open a Squeak image file. You can:\n")
        TEXT("  1. Double-click on the desired image file.\n")
        TEXT("  2. Drop the image file onto the Squeak application.\n")
        TEXT("Aborting...\n"), toUnicode(imageName));
  }
  return -1;
}

