/****************************************************************************
*   PROJECT: Squeak port for Win32 (NT / Win95)
*   FILE:    sqWin32Sound.c
*   CONTENT: Sound management
*
*   AUTHOR:  Andreas Raab (ar)
*   ADDRESS: University of Magdeburg, Germany
*   EMAIL:   raab@isg.cs.uni-magdeburg.de
*   RCSID:   $Id: sqWin32Sound.c,v 2.0.1.2 1998/07/23 19:44:39 raab Exp $
*
*   NOTES:
*
*****************************************************************************/
#include <windows.h>
#include "sq.h"

#ifndef NO_SOUND

#ifndef NO_RCSID
  static char RCSID[]="$Id: sqWin32Sound.c,v 2.0.1.2 1998/07/23 19:44:39 raab Exp $";
#endif

/* HACK! Define LPWAVEFORMAT as LPWAVEFORMATEX for WinVER >= 4 */
#if WINVER >= 0x0400
#define LPWAVEFORMAT LPWAVEFORMATEX
#endif

/* number of buffers used */
#define NUM_BUFFERS 4
/* initial buffer size */
#define BUFFER_SIZE 4096

/* Note: NUM_BUFFERS and BUFFER_SIZE should be carefully chosen.
   You better not touch it unless you know what you're doing, but
   ALWAYS use a multiple of 2 for NUM_BUFFERS. BUFFER_SIZE affects 
   only sound input. The output buffer size is determined by Squeak.
*/

HWAVEOUT hWaveOut = 0;
HWAVEIN hWaveIn = 0;
WAVEFORMATEX waveOutFormat;
WAVEFORMATEX waveInFormat;

static int outBufferSize = 0;

static WAVEHDR *(playBuffers[NUM_BUFFERS]);
static WAVEHDR *(recordBuffers[NUM_BUFFERS]);

static int currentRecordBuffer = 0; /* Index of the buffer currently recorded */
static int currentPlayBuffer = 0;   /* Index of the buffer currently played */
static int playingStarted = 0;

/* Initialize wave headers */
static void
initWaveHeaders(WAVEHDR **buffers, DWORD bufferSize)
{ int i;

  for(i=0;i<NUM_BUFFERS;i++)
    {
      (buffers[i]) = GlobalLock(GlobalAlloc(GMEM_ZEROINIT | GMEM_MOVEABLE,sizeof(WAVEHDR)));
      (buffers[i])->dwBufferLength = bufferSize;
      (buffers[i])->lpData = GlobalLock(GlobalAlloc(GMEM_MOVEABLE, bufferSize));
      (buffers[i])->dwFlags |= WHDR_DONE;
    }
}

/* Release wave headers */
static void
freeWaveHeaders(WAVEHDR **buffers)
{ int i;
  HANDLE h;

  for(i=0;i<NUM_BUFFERS;i++)
    {
      /* Free data buffer */
      h = GlobalHandle((buffers[i])->lpData);
      GlobalUnlock(h);
      GlobalFree(h);
      /* Free wave header */
      h = GlobalHandle(buffers[i]);
      GlobalUnlock(h);
      GlobalFree(h);
      buffers[i] = 0;
    }
}

/* find the next empty header which can be used for sound output */
static WAVEHDR *
findEmptyHeader(WAVEHDR **buffers)
{ int i;
  int bufIndex = currentPlayBuffer;

  for(i=0;i<NUM_BUFFERS;i++)
    {
      if(((buffers[bufIndex])->dwFlags & WHDR_DONE) != 0)
        return buffers[bufIndex];
      bufIndex = (bufIndex+1) % NUM_BUFFERS;
    }
  return NULL;
}

/* provide a buffer for sound input */
static int 
provideRecordBuffer(WAVEHDR *header)
{
  header->dwUser = 0; /* reset read index */
  if(waveInPrepareHeader(hWaveIn,header,sizeof(WAVEHDR)) != 0)
    return 0;
  if(waveInAddBuffer(hWaveIn,header,sizeof(WAVEHDR)) != 0)
    return 0;
  return 1;
}

static int
providePlayBuffer(WAVEHDR *header)
{
  /* give us a time stamp for the mix-in of samples */
  if(!playingStarted) 
    {
      (playBuffers[currentPlayBuffer])->dwUser = (DWORD) ioMicroMSecs();
      playingStarted = 1;
    }
  if(waveOutPrepareHeader(hWaveOut,header,sizeof(WAVEHDR)) != 0)
    return 0;
  if(waveOutWrite(hWaveOut,header,sizeof(WAVEHDR)) != 0)
    return 0;
  return 1;
}

/* The wave callback: Signal the associated semaphore that a buffer has completed.
   This means either that the buffer has been played and can be filled again (playing)
   or that the buffer is filled and waits for being read (recording) */
void CALLBACK
waveCallback(HWAVEOUT hWave, UINT uMsg, DWORD dwSemaphoreIndex, DWORD dwParam1, DWORD dwParam2)
{
  if(uMsg == WOM_DONE)
    { /* a buffer has been played */
      currentPlayBuffer = (currentPlayBuffer + 1) % NUM_BUFFERS;
      (playBuffers[currentPlayBuffer])->dwUser = (DWORD) ioMicroMSecs();
    }
  if(uMsg == WOM_DONE || uMsg == WIM_DATA)
    { /* a buffer has been completed */
      synchronizedSignalSemaphoreWithIndex((int)dwSemaphoreIndex);
    }
}

/*******************************************************************/
/*  Sound output functions                                         */
/*******************************************************************/

int snd_AvailableSpace(void)
{ int i;
  for(i=0;i<NUM_BUFFERS;i++)
    if(((playBuffers[i])->dwFlags & WHDR_DONE) != 0)
      return outBufferSize;
  return 0;
}

int snd_PlaySamplesFromAtLength(int frameCount, int arrayIndex, int startIndex)
{ WAVEHDR *header;
  int bytesWritten;
  char *src,*dst,*end;
  short *shortSrc;

  header = findEmptyHeader(playBuffers);
  if(!header) return 0; /* no buffer */

  bytesWritten = waveOutFormat.nBlockAlign  * frameCount;
  if(bytesWritten > outBufferSize)
    bytesWritten = outBufferSize;

  src = (char *) (arrayIndex + startIndex);
  dst = (char *) header->lpData;
  end = (char *) dst + bytesWritten;
  shortSrc = (short*) src;

  if (waveOutFormat.wBitsPerSample == 8) 
    {  /* 8-bit samples */
      if (waveOutFormat.nChannels == 2) /* stereo */
        while (dst < end)
          { *dst++ = (*shortSrc++ + 0x8000) >> 8;
            *dst++ = (*shortSrc++ + 0x8000) >> 8;
          }
      else
        while (dst < end)
          *dst++ = ( ((*shortSrc++ + 0x8000) >> 8) + ((*shortSrc++ + 0x8000) >> 8) ) / 2;
          
    } 
  else 
    if (waveOutFormat.nChannels == 2) /* stereo */
      while (dst < end)
        *((long*)dst)++ = *((long*)src)++;
    else /* if mono, skip every other frame of the source */
      while (dst < end) 
        {
          *((short*)dst)++ = (*shortSrc++ + *shortSrc++) / 2;
        }

  header->dwBufferLength = bytesWritten;
  if(!providePlayBuffer(header)) return 0;
  return bytesWritten / waveOutFormat.nBlockAlign;
}

void MixInSamples(int count, char *srcBufPtr, int srcStartIndex, char *dstBufPtr, int dstStartIndex) 
{ unsigned char *src, *dst, *end;
  short *shortSrc, *shortDst;
  int sample;

  src = (unsigned char *) srcBufPtr + (srcStartIndex * 4);
  dst = (unsigned char *) dstBufPtr + (dstStartIndex * waveOutFormat.nBlockAlign);
  end = (unsigned char *) dstBufPtr + ((dstStartIndex + count) * 4);
  shortSrc = (short*) src;
  shortDst = (short*) dst;

  if (waveOutFormat.wBitsPerSample == 8) 
    {  /* 8-bit samples */
      if (waveOutFormat.nChannels == 2) /* stereo */
        while (dst < end)
          { 
            sample = *dst + ((*shortSrc++ + 0x8000) >> 8);
            /* @@: working on unsigned - no underflow */
            if(sample > 255) sample = 255;
            *dst++ = sample;
          }
      else /* Mono */
        while (dst < end)
          {
            sample = *dst + ( ((*shortSrc++ + 0x8000) >> 8) + ((*shortSrc++ + 0x8000) >> 8) ) / 2;
            /* @@: working on unsigned - no underflow */
            if(sample > 255) sample = 255;
            *dst++ = sample;
          }
          
    } 
  else /* 16-bit samples */
    if (waveOutFormat.nChannels == 2) /* stereo */
      while (shortDst < (short*)end)
        {
          sample = (*shortDst + *shortSrc++) / 2;
          if (sample > 32767) sample = 32767;
          else if (sample < -32767) sample = -32767;
          *shortDst++ = sample;
        }
    else /* if mono, skip every other frame of the source */
      while (shortDst < (short*)end) 
        {
          sample = (*dst + *shortSrc++ + *shortSrc++) / 4;
          if (sample > 32767) sample = 32767;
          else if (sample < -32767) sample = -32767;
          *shortDst++ = sample;
        }
}


int snd_PlaySilence(void)
{ WAVEHDR *header;

  header = findEmptyHeader(playBuffers);
  if(!header) return 0; /* no buffer */
  ZeroMemory(header->lpData, header->dwBufferLength);
  if(!providePlayBuffer(header)) return 0;
  return header->dwBufferLength / waveOutFormat.nBlockAlign;
}

int snd_Start(int frameCount, int samplesPerSec, int stereo, int semaphoreIndex)
{ int bytesPerFrame;
  int bytesPerSample;
  int bufferBytes;
  int maxDevs = 0,devId = 0;
  MMRESULT result;

  static int initFlag = 0;

  if (hWaveOut) 
    {
      /* still open from last time; clean up before continuing */
      snd_Stop();
    }

  /* find a device supporting the requested output rate */
  maxDevs = waveOutGetNumDevs();
  /* try 16bit devices first */
  bytesPerSample = 2;
  for(devId=0;devId<maxDevs;devId++)
    { WAVEOUTCAPS caps;
      int flag;

      waveOutGetDevCaps(devId,&caps,sizeof(WAVEOUTCAPS));
      switch (samplesPerSec) {
        case 11025: flag = stereo ? WAVE_FORMAT_1S16 : WAVE_FORMAT_1M16; break;
        case 22050: flag = stereo ? WAVE_FORMAT_2S16 : WAVE_FORMAT_2M16; break;
        case 44100: flag = stereo ? WAVE_FORMAT_4S16 : WAVE_FORMAT_4M16; break;
        default: return 0;
      }
      if(caps.dwFormats & flag)
        break;
    }

  if(devId >= maxDevs)
    {/* try 8bit devices */
      bytesPerSample = 1;
      for(devId=0;devId<maxDevs;devId++)
        { WAVEOUTCAPS caps;
          int flag;

          waveOutGetDevCaps(devId,&caps,sizeof(WAVEOUTCAPS));
          switch (samplesPerSec) {
            case 11025: flag = stereo ? WAVE_FORMAT_1S08 : WAVE_FORMAT_1M08; break;
            case 22050: flag = stereo ? WAVE_FORMAT_2S08 : WAVE_FORMAT_2M08; break;
            case 44100: flag = stereo ? WAVE_FORMAT_4S08 : WAVE_FORMAT_4M08; break;
            default: return 0;
          }
          if(caps.dwFormats & flag)
            break;
        }
    }
  if(devId >= maxDevs)
    return 0;

  bytesPerFrame = stereo ? 2 * bytesPerSample : bytesPerSample;
  waveOutFormat.wFormatTag = WAVE_FORMAT_PCM;
  waveOutFormat.nChannels = stereo ? 2 : 1;
  waveOutFormat.nSamplesPerSec = samplesPerSec;
  waveOutFormat.nAvgBytesPerSec = samplesPerSec * bytesPerFrame;
  waveOutFormat.nBlockAlign = bytesPerFrame;
  waveOutFormat.wBitsPerSample = 8 * bytesPerSample;


  result = waveOutOpen(&hWaveOut,devId, (LPWAVEFORMAT)&waveOutFormat, 
                       (DWORD)waveCallback, semaphoreIndex, CALLBACK_FUNCTION);

  if(result != 0)
    {
#ifndef NO_WARNINGS
      TCHAR errorText[256];
      waveOutGetErrorText(result,errorText,255);
      warnPrintf(TEXT("%s\n"),errorText);
#endif
      return 0;
    }
  /* Initialize output buffers */
  bufferBytes = (bytesPerFrame * frameCount) / (NUM_BUFFERS / 2);
  bufferBytes = (bufferBytes / 16) * 16;
  if(bufferBytes != outBufferSize || !initFlag)
    {
#ifndef NDEBUG
      warnPrintf(TEXT("Initializing %d output buffers of size %d\n"), NUM_BUFFERS, bufferBytes);
      warnPrintf(TEXT("bytesPerSample: %d\nstereo: %d\nbytesPerFrame: %d\n"),bytesPerSample, stereo, bytesPerFrame);
#endif
      outBufferSize = bufferBytes;
      if(initFlag) freeWaveHeaders(playBuffers);
      initWaveHeaders(playBuffers, outBufferSize);
      initFlag = 1;
    }
  playingStarted = 0;
  currentPlayBuffer = 0;
  return 1;
}

int snd_Stop(void)
{
  if(!hWaveOut) return 0;
  waveOutReset(hWaveOut);
  waveOutClose(hWaveOut);
  hWaveOut = 0;
  return 1;
}

int snd_InsertSamplesFromLeadTime(int frameCount, int srcBufPtr, int samplesOfLeadTime) 
{ WAVEHDR *header;
  int currentBuffer;
  int samplesInserted;
  int startSample;
  int i, count;
  DWORD nowTime;

  if(!hWaveOut) return -1;
  samplesInserted = 0;
  currentBuffer = currentPlayBuffer;
  /* mix as many samples as can fit into the remainder of the currently playing buffer */
  header = playBuffers[currentBuffer];
  nowTime = (DWORD) ioMicroMSecs();
  startSample = ((waveOutFormat.nSamplesPerSec * (nowTime - header->dwUser)) / 1000) + samplesOfLeadTime;
  if((DWORD) startSample * waveOutFormat.nBlockAlign < header->dwBufferLength)
    {
      count = (header->dwBufferLength / waveOutFormat.nBlockAlign) - startSample;
      if(count > frameCount) count = frameCount;
      MixInSamples(count, (char*) srcBufPtr, 0, header->lpData, startSample);
      samplesInserted = count;
    }
  /* mix remaining samples into the inactive buffers */
  for(i=1;i<NUM_BUFFERS;i++)
    {
      currentBuffer = (currentBuffer+1) % NUM_BUFFERS;
      header = playBuffers[currentBuffer];
      count = (header->dwBufferLength / waveOutFormat.nBlockAlign);
      if((samplesInserted + count) > frameCount) count = frameCount - samplesInserted;
      MixInSamples(count, (char*) srcBufPtr, samplesInserted, header->lpData, 0);
      samplesInserted += count;
    }
  return samplesInserted;
}

/*******************************************************************/
/*  Sound input functions                                          */
/*******************************************************************/

int snd_SetRecordLevel(int level)
{
  /* Note: I don't see a simple way of accomplishing the change in recording level.
     One way (such as on my SB16) would be to modify the output level of the source
     signal. This works fine on on SoundBlaster cards but I don't know of other cards.
     Also we would have to know what we're recording (it's no good idea to change
     the output level of _all_ available drivers such as CD, MIDI, Wave, PC-Speaker,
     Line-In, or Microphone).
     Another way could be using the mixer functions, but my help files and books
     lack this topic completely. If somebody has an idea how to do it, let me know.
     AR
  */
  return 1;
}

int snd_StartRecording(int desiredSamplesPerSec, int stereo, int semaphoreIndex) 
{ int bytesPerSample;
  int bytesPerFrame, samplesPerSec;
  int maxDevs = 0,devId = 0, i;
  MMRESULT result;

  static int initFlag = 0;

  if(!initFlag)
    { /* first time -- initialize wave headers */
      initWaveHeaders(recordBuffers, BUFFER_SIZE);
      initFlag = 1;
    }

  /* round desiredSamplesPerSec to valid values */
  if (desiredSamplesPerSec <= 16538)
      samplesPerSec = 11025;
  else
    if (desiredSamplesPerSec <= 33075)
      samplesPerSec = 22050;
    else
      samplesPerSec = 44100;

  if (hWaveIn) 
    {
      /* still open from last time; clean up before continuing */
      snd_StopRecording();
    }

  /* find a device supporting the requested input rate */
  maxDevs = waveInGetNumDevs();
  /* try 16bit devices first */
  bytesPerSample = 2;
  for(devId=0;devId<maxDevs;devId++)
    { WAVEINCAPS caps;
      int flag;

      waveInGetDevCaps(devId,&caps,sizeof(WAVEINCAPS));
      switch (samplesPerSec) {
        case 11025: flag = stereo ? WAVE_FORMAT_1S16 : WAVE_FORMAT_1M16; break;
        case 22050: flag = stereo ? WAVE_FORMAT_2S16 : WAVE_FORMAT_2M16; break;
        case 44100: flag = stereo ? WAVE_FORMAT_4S16 : WAVE_FORMAT_4M16; break;
        default: return 0;
      }
      if(caps.dwFormats & flag)
        break;
    }

  if(devId >= maxDevs)
    {/* try 8bit devices */
      bytesPerSample = 1;
      for(devId=0;devId<maxDevs;devId++)
        { WAVEINCAPS caps;
          int flag;

          waveInGetDevCaps(devId,&caps,sizeof(WAVEINCAPS));
          switch (samplesPerSec) {
            case 11025: flag = stereo ? WAVE_FORMAT_1S08 : WAVE_FORMAT_1M08; break;
            case 22050: flag = stereo ? WAVE_FORMAT_2S08 : WAVE_FORMAT_2M08; break;
            case 44100: flag = stereo ? WAVE_FORMAT_4S08 : WAVE_FORMAT_4M08; break;
            default: return 0;
          }
          if(caps.dwFormats & flag)
            break;
        }
    }
  if(devId >= maxDevs)
    return 0;

  /* query a wave in device */
  bytesPerFrame = stereo ? 2 * bytesPerSample : bytesPerSample;
  waveInFormat.wFormatTag = WAVE_FORMAT_PCM;
  waveInFormat.nChannels = stereo ? 2 : 1;
  waveInFormat.nSamplesPerSec = samplesPerSec;
  waveInFormat.nAvgBytesPerSec = samplesPerSec * bytesPerFrame;
  waveInFormat.nBlockAlign = bytesPerFrame;
  waveInFormat.wBitsPerSample = 8 * bytesPerSample;

  result = waveInOpen(&hWaveIn,devId, (LPWAVEFORMAT)&waveInFormat, 
                      (DWORD)waveCallback, semaphoreIndex, CALLBACK_FUNCTION);
  if(result != 0)
    {
#ifndef NO_WARNINGS
      TCHAR errorText[256];
      waveInGetErrorText(result,errorText,255);
      warnPrintf(TEXT("%s\n"),errorText);
#endif
      return 0;
    }
  /* The device was successfully opened. Provide the input buffers. */
  for(i=0; i<NUM_BUFFERS; i++)
    if(!provideRecordBuffer(recordBuffers[i])) return 0; /* something went wrong */
  /* first buffer has index 0 */
  currentRecordBuffer = 0;
  /* And away we go ... */
  if(waveInStart(hWaveIn) != 0) return 0;

  return 1;
}

int snd_StopRecording(void)
{
  if(!hWaveIn) return 0;
  waveInReset(hWaveIn);
  waveInClose(hWaveIn);
  hWaveIn = 0;
  return 1;
}

double snd_GetRecordingSampleRate(void)
{
  if(!hWaveIn) return 0.0;
  return (double) waveInFormat.nSamplesPerSec;
}

int snd_RecordSamplesIntoAtLength(int buf, int startSliceIndex, int bufferSizeInBytes)
{
  /* if data is available, copy as many sample slices as possible into the
     given buffer starting at the given slice index. do not write past the
     end of the buffer, which is buf + bufferSizeInBytes. return the number
     of slices (not bytes) copied. a slice is one 16-bit sample in mono
     or two 16-bit samples in stereo. */
  int bytesPerSlice = (waveInFormat.nChannels * 2);
  int recordBytesPerSlice = (waveInFormat.wBitsPerSample / 8);
  char *nextBuf = (char *) buf + (startSliceIndex * bytesPerSlice);
  char *bufEnd = (char *) buf + bufferSizeInBytes;
  char *srcStart, *src, *srcEnd;
  int bytesCopied;
  WAVEHDR *recBuf;

  if (!hWaveIn) {
    success(false);
    return 0;
  }

  /* Copy all recorded samples into the buffer provided.
     We use an endless loop here which is exited if
       a) there is no room left in the provided buffer
       b) there is no data available from the wave input device
     The WAVEHDR's dwUser member is used to mark the read position
     in case a previous call has been exited because of a)
  */
  bytesCopied = 0;
  while(true)
    {
      /* select the buffer */
      recBuf = recordBuffers[currentRecordBuffer];
      /* We do _NOT_ copy partial buffer contents, so wait 
         until the buffer is marked */
      if((recBuf->dwFlags & WHDR_DONE) == 0) break;
      /* copy samples into the client's buffer */
      srcStart = recBuf->lpData + recBuf->dwUser;
      src = srcStart;
      srcEnd = recBuf->lpData + recBuf->dwBytesRecorded;
      if (waveInFormat.wBitsPerSample == 8) 
        {
          while ((src < srcEnd) && (nextBuf < bufEnd)) 
            {
              /* convert 8-bit sample to 16-bit sample */
              *nextBuf++ = 0;  /* low-order byte is zero */
              *nextBuf++ = (*src++) - 127;  /* convert from [0-255] to [-128-127] */
            }
        }
      else 
        {
          while ((src < srcEnd) && (nextBuf < bufEnd)) 
            *nextBuf++ = *src++;
        }
      bytesCopied += src - srcStart;
      /* are we at the end of the provided buffer? */
      if(src < srcEnd)
        { /* Yes, update read index */
          recBuf->dwUser = src - recBuf->lpData;
          /* and exit */
          break;
        }
      /* we have completed a buffer, send it back to the device for further use */
      if(!provideRecordBuffer(recBuf)) break; /* something went wrong */
      /* step on to the next buffer */
      currentRecordBuffer = (currentRecordBuffer + 1) % NUM_BUFFERS;
    }
  /* return the number of slices copied */
  return bytesCopied / recordBytesPerSlice;
}

#endif /* NO_SOUND */
