/****************************************************************************
*   PROJECT: Squeak port for Win32 (NT / Win95)
*   FILE:    sqWin32Net.c
*   CONTENT: Networking in Squeak
*
*   AUTHOR:  Andreas Raab (ar)
*   ADDRESS: University of Magdeburg, Germany
*   EMAIL:   raab@isg.cs.uni-magdeburg.de
*   RCSID:   $Id: sqWin32Net.c,v 2.0.1.9 1998/10/02 22:23:48 raab Exp $
*
*   NOTES:   1) For now, only TCP is supported.
*
*****************************************************************************/
#include <windows.h>
#include <winsock.h> /* needed under (_WIN32_WCE) */
#include "sq.h"

#ifndef NO_NETWORK

#ifndef NO_RCSID
  static char RCSID[]="$Id: sqWin32Net.c,v 2.0.1.9 1998/10/02 22:23:48 raab Exp $";
#endif

/*** Socket TYpe Constants ***/
#define TCPSocketType 0
#define UDPSocketType 1

/*** Resolver Status Constants ***/
#define RESOLVER_UNINITIALIZED 0
#define RESOLVER_SUCCESS  1
#define RESOLVER_BUSY   2
#define RESOLVER_ERROR   3

/*** TCP Socket Status Constants ***/
#define Unconnected    0
#define WaitingForConnection 1
#define Connected    2
#define OtherEndClosed   3
#define ThisEndClosed   4

#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define SIGNAL(index) synchronizedSignalSemaphoreWithIndex(index)

extern HWND stWindow;

#ifndef INVALID_SOCKET
#define INVALID_SOCKET (-1)
#endif
#ifndef SOCKET_ERROR
#define SOCKET_ERROR (-1)
#endif

static int resolverSemaphoreIndex = 0;
static int thisNetSession = 0;
static int zero = 0;
static int one = 1;

#define MAXHOSTNAMELEN 256
static char localHostName[MAXHOSTNAMELEN];
static u_long localHostAddress;

typedef struct privateSocketStruct {
  struct privateSocketStruct *next;
  SOCKET s;

  int sockState;
  int sockError;
  int semaphoreIndex;
  struct sockaddr_in peer;  /* socket address in connect() or send/rcv address for UDP */

  HANDLE mutex;             /* The mutex used for synchronized access to this socket */

  HANDLE hThread;           /* primary thread on this socket (used for connect(), accept() and recv() */
  HANDLE recvEvent;         /* event used for wake up of the sqRecv() thread */
  int    recvDataSize;      /* size of receive buffer */
  int    recvDataNum;       /* number of items in receive buffer */
  char   *recvData;         /* receive data buffer */


  HANDLE hThread2;          /* secondary thread on this socket (used only for send()) */
  HANDLE sendEvent;         /* event used for wake up of the sqSend() thread */
  int    sendDataSize;      /* size of send buffer */
  int    sendDataNum;       /* number of items in send buffer */
  char   *sendData;         /* send data buffer */
  int    closePending;      /* Is there any pending close operation?! */

} privateSocketStruct;

static privateSocketStruct *firstSocket = NULL;
/* Additional flags for sockState which will be received by async notification:
   SOCK_DATA_SENT      - all pending data has been sent
   SOCK_DATA_AVAILABLE - data is available for this connection
*/
#define SOCK_PUBLIC_MASK    0x0000FFFF
#define SOCK_DATA_SENT      0x00010000
#define SOCK_DATA_AVAILABLE 0x00020000

/********* Private accessors of a Squeak socket pointer *********/
#define PSP(s)         ((privateSocketStruct*) ((s)->privateSocketPtr))

#define SOCKET(s)      (PSP(s)->s)
#define SOCKETSTATE(s) (PSP(s)->sockState)
#define SOCKETERROR(s) (PSP(s)->sockError)
/* Thread specific functions */
#define MUTEX(s)         (PSP(s)->mutex)
#define THREAD(s)        (PSP(s)->hThread)
#define ADDRESS(s)       ((struct sockaddr_in*)(&PSP(s)->peer))

#define LOCKSOCKET(mutex, duration) \
  if(WaitForSingleObject(mutex, duration) == WAIT_FAILED)\
    {\
      printLastError(TEXT("Failed to lock socket"));\
      return 0;\
    }
#define UNLOCKSOCKET(mutex)\
  if(ReleaseMutex(mutex) == 0)\
    printLastError(TEXT("Failed to unlock socket"));

static void TerminatePrimaryThread(privateSocketStruct *pss);
static DWORD WINAPI sqGetHostByAddr(int netAddress);
static DWORD WINAPI sqGetHostByName(char *hostName);
static DWORD WINAPI sqConnect(privateSocketStruct *pss);
static DWORD WINAPI sqAccept(privateSocketStruct *pss);
static DWORD WINAPI sqRecv(privateSocketStruct *pss);
static DWORD WINAPI sqSend(privateSocketStruct *pss);



/*****************************************************************************
 *****************************************************************************
 *****************************************************************************/

static int sqAbortSocket(privateSocketStruct *pss)
{ struct linger l;

  LOCKSOCKET(pss->mutex, INFINITE);
  l.l_onoff = 1;
  l.l_linger = 0;
  setsockopt(pss->s, SOL_SOCKET, SO_LINGER, (char*)&l, sizeof(l));
  closesocket(pss->s);
  pss->s = 0;
  pss->sockState = Unconnected;
  UNLOCKSOCKET(pss->mutex);
}

/* TerminatePrimaryThread: Terminate the primary thread associated with the socket */
static void TerminatePrimaryThread(privateSocketStruct *pss)
{
  if(pss->hThread)
    {
      if(!TerminateThread(pss->hThread, 0))
        printLastError(TEXT("TerminateThread failed"));
      pss->hThread = NULL;
    }
}

/* sqRecv - thread for detecting incoming data */

static DWORD WINAPI sqRecv(privateSocketStruct *pss)
{ int result;
  int lastError;

  while(1) {
    /* give the main thread a clue that this thread is curently running */
    SetEvent(pss->recvEvent);
    /* read incoming data */
    result = recv(pss->s,pss->recvData, pss->recvDataSize, 0);
    if(result > 0)
      pss->recvDataNum = result;
    else
      pss->recvDataNum = 0;

    /* Set the flags */
    LOCKSOCKET(pss->mutex, INFINITE);
    if(result > 0)
      pss->sockState |= SOCK_DATA_AVAILABLE;
    else if(result == 0)
      pss->sockState = OtherEndClosed;
    else /* result < 0 */
      {
        lastError = WSAGetLastError();
        if(lastError == WSAECONNRESET) /* connection reset by peer */
          pss->sockState = OtherEndClosed;
        else
          pss->sockError |= lastError;
      }
    UNLOCKSOCKET(pss->mutex);

    /* signal Squeak */
    SIGNAL(pss->semaphoreIndex);
    /* and get to sleep until we get a signal */
    ResetEvent(pss->recvEvent);
    WaitForSingleObject(pss->recvEvent, INFINITE);
  }
}

/* sqSend - thread for sending data */

static DWORD WINAPI sqSend(privateSocketStruct *pss)
{ int result;
  int lastError;

  while(1) {
    /* Get to sleep if there is no data to be sent and if the
       connection hasn't been already closed */
    if(!pss->sendDataNum && !pss->closePending)
      {
        /* Wait until we have data */
        WaitForSingleObject(pss->sendEvent, INFINITE);
      }

    /* send data */
    if(pss->sendDataNum)
      result = send(pss->s, pss->sendData, pss->sendDataNum, 0);

    LOCKSOCKET(pss->mutex, INFINITE);
    /* Note: Squeak expects us to indicate pending data even if
             the connection is already closed. */
    if(result == SOCKET_ERROR)
      { /* something went wrong */
        lastError = WSAGetLastError();
        if(lastError == WSAECONNRESET) /* connection reset by peer */
          pss->sockState = OtherEndClosed;
        else
          pss->sockError |= lastError;
      }
    else
      pss->sendDataNum = 0; /* data has been sent */
    ResetEvent(pss->sendEvent);
    UNLOCKSOCKET(pss->mutex);
    /* check if this end has been closed so we should
       terminate the connection afterwards */
    if(pss->closePending)
      {
        LOCKSOCKET(pss->mutex,INFINITE);
        closesocket(pss->s);
        pss->s = 0;
        pss->closePending = 0;
        pss->sockState = Unconnected;
        UNLOCKSOCKET(pss->mutex);
        SIGNAL(pss->semaphoreIndex);
        /* And terminate the thread so that nobody is confused */
        TerminateThread(pss->hThread2, 0);
      }
    /* something changed */
    SIGNAL(pss->semaphoreIndex);
    /* and back to waiting ... */
  }
}


/* sqConnect: Perform a threaded connect() and go to sqRecv() */
static DWORD WINAPI sqConnect(privateSocketStruct *pss)
{ int err;

  /* If no error occurs, connect returns 0. 
     Otherwise, it returns SOCKET_ERROR, and a specific error 
     code may be retrieved by calling WSAGetLastError.*/
  err = connect( pss->s, (struct sockaddr*) &pss->peer, sizeof(struct sockaddr_in));
  LOCKSOCKET(pss->mutex, INFINITE);
  if(err)
    pss->sockError |= WSAGetLastError();
  else
    pss->sockState = Connected; /* connected and ready to send */
  UNLOCKSOCKET(pss->mutex);
  /* Something has happened. Inform any waiting process of the change */
  SIGNAL(pss->semaphoreIndex);
  /* and start reading if ok */
  if(!err) sqRecv(pss);
  /* else terminate the thread here */
  LOCKSOCKET(pss->mutex, INFINITE);
  pss->hThread = NULL;
  UNLOCKSOCKET(pss->mutex);
  TerminateThread(GetCurrentThread(), 0);
}

/* sqAccept: Perform a threaded accept() and go to sqRecv */
static DWORD WINAPI sqAccept(privateSocketStruct *pss)
{
  SOCKET newSocket;
  int err;

  /* If no error occurs, accept returns a value of type SOCKET which is 
  a descriptor for the accepted packet. Otherwise, a value of INVALID_SOCKET 
  is returned, and a specific error code may be retrieved by calling WSAGetLastError.*/
  newSocket = accept(pss->s,0,NULL);
  err = newSocket == INVALID_SOCKET;

  LOCKSOCKET(pss->mutex, INFINITE);
  if(err)
    pss->sockError |= WSAGetLastError();
  else
    {
      /* NOTE: This one is a hack.
         I would actually prefer to return a new socket when the connection is accepted.
         However, we have to re-use the listening socket, therefore we
           a) have accepted the connection
           b) close the listener
           c) store the accepted connection in the listeners place
       */
      closesocket(pss->s);
      pss->s = newSocket;
      pss->sockState = Connected; /* connected and ready to send */
    }
  UNLOCKSOCKET(pss->mutex);
  /* Something has happened. Inform any waiting process of the change */
  SIGNAL(pss->semaphoreIndex);
  /* and start reading if ok */
  if(!err) sqRecv(pss);
  /* else terminate the thread here */
  LOCKSOCKET(pss->mutex, INFINITE);
  pss->hThread = NULL;
  UNLOCKSOCKET(pss->mutex);
  TerminateThread(GetCurrentThread(), 0);
}

/* sqCreateSendThread: Create the secondary thread on a socket */
static void sqCreateSendThread(privateSocketStruct *pss)
{ DWORD id;

  /* Just to make sure there is no garbage */
  pss->sendDataNum = 0;
  /* create the send() thread which will immediately go to sleep */
  pss->hThread2 = 
        CreateThread(NULL,                    /* No security descriptor */
                     0,                       /* default stack size     */
                     (LPTHREAD_START_ROUTINE) &sqSend, /* what to do */
                     (LPVOID) pss,            /* parameter for thread   */
                     CREATE_SUSPENDED,        /* creation parameter -- create suspended */
                     &id);                    /* return value for thread id */
  if(!pss->hThread2)
    printLastError(TEXT("CreateThread() failed"));
  if(!SetThreadPriority(pss->hThread2, THREAD_PRIORITY_HIGHEST))
    printLastError(TEXT("SetThreadPriority() failed"));
  if(!ResumeThread(pss->hThread2))
    printLastError(TEXT("ResumeThread() failed"));
}

/* sqCreatePrimaryThread: Create the primary thread on a socket */
static void sqCreatePrimaryThread(privateSocketStruct *pss, 
                                  LPTHREAD_START_ROUTINE startFn)
{ DWORD id;

  /* Create the primary thread */
  pss->hThread =
    CreateThread(NULL,              /* No security descriptor */
                 0,                 /* default stack size     */
                 startFn,           /* what to do */
                 (LPVOID) pss,      /* parameter for thread   */
                 CREATE_SUSPENDED,  /* creation parameter -- create suspended so we can check the return value */
                 &id);              /* return value for thread id */
  if(!pss->hThread)
    printLastError(TEXT("CreateThread() failed"));
  if(!SetThreadPriority(pss->hThread, THREAD_PRIORITY_HIGHEST))
    printLastError(TEXT("SetThreadPriority() failed"));
  /* NOTE: We may get a connection immediately after resuming so set state here */
  pss->sockState = WaitingForConnection;
  pss->sockError = 0;
  if(!ResumeThread(pss->hThread))
    printLastError(TEXT("ResumeThread() failed"));
}

/*****************************************************************************
 *****************************************************************************
 *****************************************************************************/

/*****************************************************************************
  sqNetworkInit: Initialize network with the given DNS semaphore.
*****************************************************************************/
int sqNetworkInit(int resolverSemaIndex)
{ WSADATA wsaData;
  int err;


  if (thisNetSession != 0) return 0;  /* noop if network is already initialized */

  err = WSAStartup( MAKEWORD(1,1), &wsaData );
  if ( err != 0 ) 
    return -1;

  /* Confirm that the Windows Sockets DLL supports 1.1.*/
  /* Note that if the DLL supports versions greater */
  /* than 1.1 in addition to 1.1, it will still return */
  /* 1.1 in wVersion since that is the version we */
  /* requested.     */
  if ( LOBYTE( wsaData.wVersion ) != 1 || HIBYTE( wsaData.wVersion ) != 1 ) 
    {
      WSACleanup();
      return -1; 
    }

  gethostname(localHostName,MAXHOSTNAMELEN);
  /* success! */
  thisNetSession = GetTickCount();
  if (thisNetSession == 0) thisNetSession = 1;  /* don't use 0 */

  /* install resolver semaphore */
  resolverSemaphoreIndex = resolverSemaIndex;
  return 0;
}

/*****************************************************************************
  sqNetworkShutdown: Clean up networking.
*****************************************************************************/
void sqNetworkShutdown(void)
{
  if (thisNetSession == 0) return;  /* noop if network is already shut down */
  WSACleanup();
  thisNetSession = 0;
  /* TODO: Clean up pending sockets */
}

/*** Squeak Generic Socket Functions ***/

/*****************************************************************************
  SocketValid: Validate a given SocketPtr
*****************************************************************************/
static int SocketValid(SocketPtr s) {
  if ((s != NULL) &&
      (s->privateSocketPtr != NULL) &&
      (s->sessionID == thisNetSession)
     )
    return true;
  else 
    {
      success(false);
      return false;
    }
}

/*****************************************************************************
  sqSocketAbortConnection: Immediately terminate a pending connection
*****************************************************************************/
void sqSocketAbortConnection(SocketPtr s)
{
  if (!SocketValid(s)) return;

  /* Terminate the primary thread */
  TerminatePrimaryThread(PSP(s));
  /* abort the socket connection */
  sqAbortSocket(PSP(s));
}

/*****************************************************************************
  sqSocketCloseConnection: gracefully close an open socket
*****************************************************************************/
void sqSocketCloseConnection(SocketPtr s)
{ privateSocketStruct *pss = PSP(s);

  if (!SocketValid(s)) return;
  TerminatePrimaryThread(pss);
  LOCKSOCKET(MUTEX(s), INFINITE);
  /* Note: NT4 has a weird problem when closing the socket
           from a thread that's outside the send() loop.
           We have therefore to defer the close operation
           to the send thread regardless if there is pending
           data or not */
  pss->closePending = 1;
  /* Make sure the thread is not asleep */
  SetEvent(pss->sendEvent);
  UNLOCKSOCKET(MUTEX(s));
}

/*****************************************************************************
  sqSocketConnectionStatus: return public status flags of the socket
*****************************************************************************/
int sqSocketConnectionStatus(SocketPtr s)
{
  int status;

  if (!SocketValid(s)) return -1;
  LOCKSOCKET(MUTEX(s), INFINITE);
  status = SOCKETSTATE(s) & 0xFFFF;
  UNLOCKSOCKET(MUTEX(s));
  return status;
}

/*****************************************************************************
  sqSocketConnectToPort: connect() a given socket to a port
*****************************************************************************/
void sqSocketConnectToPort(SocketPtr s, int addr, int port)
{
  if (!SocketValid(s)) return;

  ZeroMemory(ADDRESS(s),sizeof(struct sockaddr_in));
  ADDRESS(s)->sin_family = AF_INET;
  ADDRESS(s)->sin_port = htons((short)port);
  ADDRESS(s)->sin_addr.s_addr = htonl(addr);
  /* kill any existing recv() thread */
  if(THREAD(s)) 
    {
#ifndef NO_WARNINGS
      warnPrintf(TEXT("Warning: Calling connect() on a socket with active primary thread\n"));
#endif
      TerminateThread(THREAD(s),0);
    }
  /* Create the primary thread starting with connect() */
  sqCreatePrimaryThread(PSP(s), (LPTHREAD_START_ROUTINE) &sqConnect);
}

/*****************************************************************************
  sqSocketListenOnPort: listen() on a given port
*****************************************************************************/
void sqSocketListenOnPort(SocketPtr s, int port)
{
  if (!SocketValid(s)) return;

  /* bind the socket */
  ZeroMemory(ADDRESS(s),sizeof(struct sockaddr_in));
  ADDRESS(s)->sin_family = AF_INET;
  ADDRESS(s)->sin_port = htons((short)port);
  ADDRESS(s)->sin_addr.s_addr = localHostAddress;;
  /* NOTE: 
     Neither bind() nor listen() can block, so there is no need to
     start a new thread here alredy. We can wait until accept() */
  bind( SOCKET(s), (struct sockaddr*) ADDRESS(s), sizeof(struct sockaddr_in));
  /* show our willingness to accept a single incoming connection */
  listen(SOCKET(s), 1);
  /* kill any existing recv() thread */
  if(THREAD(s)) 
    {
#ifndef NO_WARNINGS
      warnPrintf(TEXT("Warning: Calling accept() on a socket with active primary thread\n"));
#endif
      TerminateThread(THREAD(s),0);
    }
  /* Create the primary thread for this socket starting with accept() */
  sqCreatePrimaryThread(PSP(s), (LPTHREAD_START_ROUTINE) &sqAccept);
}

/*****************************************************************************
  sqSocketCreateNetTypeSocketTypeRecvBytesSendBytesSemaID:
  Create a socket for the given netType (which is always internet here)
  a given socketType (UDP or TCP) appropriate buffer size (being ignored ;-)
  and a semaphore to signal upon changes in the socket state.
*****************************************************************************/
void sqSocketCreateNetTypeSocketTypeRecvBytesSendBytesSemaID(
            SocketPtr s, int netType, int socketType,
            int recvBufSize, int sendBufSize, int semaIndex)
{
  SOCKET newSocket;
  privateSocketStruct *pss;

  s->sessionID = 0;
  /* perform internal initialization */
  if(socketType == 0)
    newSocket = socket(AF_INET,SOCK_STREAM, 0);
  else
    newSocket = socket(AF_INET, SOCK_DGRAM, 0);
  if(newSocket == INVALID_SOCKET)
   {
     success(false);
     return;
   }
  /* Allow the re-use of the current port */
  /* Looks like people had problem with it in the sqSocketListenOnPort() 
     function, so it's been moved here */
  setsockopt(newSocket, SOL_SOCKET, SO_REUSEADDR, (char*) &one, sizeof(one));
  pss = (privateSocketStruct*) calloc(1,sizeof(privateSocketStruct));
  pss->s = newSocket;
  pss->semaphoreIndex = semaIndex;
  pss->next = firstSocket;
  firstSocket = pss;
  /* fill the SQSocket */
  s->sessionID = thisNetSession;
  s->socketType = socketType;
  s->privateSocketPtr = pss;
  /* Make the socket blocking - everything will be run in a separate thread*/
  ioctlsocket(SOCKET(s),FIONBIO,&zero);

  /* create the receive data buffer */
  pss->recvDataSize = MAX(recvBufSize, 1024); /* at least 1k */
  pss->recvData = (char*) calloc(1, pss->recvDataSize);
  if(!pss->recvData) { success(false); return; }
  pss->recvDataNum = 0;

  /* create the send data buffer */
  pss->sendDataSize = MAX(sendBufSize, 1024); /* at least 1k */
  pss->sendData = (char*) calloc(1, pss->sendDataSize);
  if(!pss->sendData) { success(false); return; }
  pss->sendDataNum = 0;

  /* Create a new mutex object for synchronized access */
  MUTEX(s) = CreateMutex(NULL, 0,NULL);
  /* Create the sendEvent being used for waking up sqSend() */
  pss->sendEvent = CreateEvent(NULL, 1, 0, NULL);
  /* Create the recvEvent being used for waking up sqRecv() */
  pss->recvEvent = CreateEvent(NULL, 1, 0, NULL);
  /* And create the send() thread which will immediately go to sleep */
  sqCreateSendThread(pss);
}

/*****************************************************************************
  sqSocketDestroy: Release the resources associated with this socket. 
                   If a connection is open, it is aborted
*****************************************************************************/
void sqSocketDestroy(SocketPtr s)
{ privateSocketStruct *pss,*tmp;

  if (!SocketValid(s)) return;
  pss = s->privateSocketPtr;
  /* Note: We have to make sure that the send thread is finished */
  if(pss->closePending)
    { /* Nope. The send thread has still work to do.
         Ok, so we gotta suspend Squeak until the thread is finished */
         SetEvent(pss->sendEvent);
         WaitForSingleObject(pss->hThread2, 5000); /* five seconds at most */
#ifndef NO_WARNINGS
         warnPrintf("Send thread was not closed upon socket destroy");
#endif
    }
  /* close the socket if it is open */
  if(pss->s) sqSocketAbortConnection(s);
  /* Terminate any pending thread */
  TerminateThread(pss->hThread, 0);
  TerminateThread(pss->hThread2, 0);
  /* Release the mutex and events */
  CloseHandle(pss->recvEvent);
  CloseHandle(pss->sendEvent);
  CloseHandle(pss->mutex);
  /* clean up buffers */
  free(pss->sendData);
  free(pss->recvData);
  /* remove the private pointer from the list */
  if(pss == firstSocket) firstSocket = pss->next;
  else
    {
      tmp = firstSocket;
      while(tmp && tmp->next != pss) tmp = tmp->next;
      if(tmp) tmp->next = pss->next;
    }
  free(pss);
  s->privateSocketPtr = NULL;
}

/*****************************************************************************
  sqSocketError: Return any error on the socket.
*****************************************************************************/
int sqSocketError(SocketPtr s)
{
  int err;

  if(!SocketValid(s)) return -1;
  LOCKSOCKET(MUTEX(s), INFINITE);
  err = SOCKETERROR(s);
  UNLOCKSOCKET(MUTEX(s));
  return err;
}

/*****************************************************************************
  sqSocketLocalAddress: Return the address of the socket on this host.
*****************************************************************************/
int sqSocketLocalAddress(SocketPtr s)
{ struct sockaddr_in sin;
  int sinSize = sizeof(sin);
  int err;

  if (!SocketValid(s)) return -1;

  LOCKSOCKET(MUTEX(s), INFINITE);
  err = getsockname(SOCKET(s), (struct sockaddr *)&sin, &sinSize);
  UNLOCKSOCKET(MUTEX(s));

  if(err) return 0;
  if(sin.sin_family != AF_INET) return 0; /* can't handle other than internet addresses */
  return ntohl(sin.sin_addr.s_addr);
}

/*****************************************************************************
  sqSocketLocalPort: Return the port of the socket on this host
*****************************************************************************/
int sqSocketLocalPort(SocketPtr s)
{ struct sockaddr_in sin;
  int sinSize = sizeof(sin);
  int err;

  if (!SocketValid(s)) return -1;

  LOCKSOCKET(MUTEX(s), INFINITE);
  err = getsockname(SOCKET(s), (struct sockaddr *)&sin, &sinSize);
  UNLOCKSOCKET(MUTEX(s));

  if(err) return 0;
  if(sin.sin_family != AF_INET) return 0; /* can't handle other than internet addresses */
  return ntohs(sin.sin_port);
}

/*****************************************************************************
  sqSocketReceiveDataAvailable: Return non-zero if data available
*****************************************************************************/
int sqSocketReceiveDataAvailable(SocketPtr s)
{
  int haveData;

  if(!SocketValid(s)) return 0;

  LOCKSOCKET(MUTEX(s), INFINITE);
  haveData = (SOCKETSTATE(s) & SOCK_DATA_AVAILABLE) != 0;
  UNLOCKSOCKET(MUTEX(s));

  return haveData;
}

/*****************************************************************************
  sqSocketReceiveDataBufCount:
  Receive data into the given buffer. Do not exceed bufSize.
  Return the number of bytes actually read.
*****************************************************************************/
int sqSocketReceiveDataBufCount(SocketPtr s, int buf, int bufSize)
{ int minBuf;
  privateSocketStruct *pss = PSP(s);

  if (!SocketValid(s)) return -1;

  /* copy data from the buffer */

  if(SOCKETERROR(s)) return -1;

  /* Check if the recv thread is really asleep */
  if(WaitForSingleObject(pss->recvEvent, 0) == WAIT_OBJECT_0)
    {
      /* Argh! The thread is in the midst of a recv() */
      SetEvent(pss->recvEvent); /* in case we come here twice */
      return 0;
    }

  minBuf = MIN(bufSize, pss->recvDataNum);
  MoveMemory((void*)buf, pss->recvData, minBuf);
  if(minBuf < pss->recvDataNum) /* need to copy data */
    MoveMemory(&(pss->recvData[0]), &(pss->recvData[minBuf]), pss->recvDataNum - minBuf);
  pss->recvDataNum -= minBuf;

  if(!pss->recvDataNum)
    { /* no more data -- reset the flag and wake up sqRecv() */
      LOCKSOCKET(MUTEX(s), INFINITE);
      SOCKETSTATE(s) &= ~SOCK_DATA_AVAILABLE;
      UNLOCKSOCKET(MUTEX(s));
      SetEvent(pss->recvEvent);
    }
  return minBuf;
}

/*****************************************************************************
  sqSocketRemoteAddress: Return the address of the socket on the remote host
*****************************************************************************/
int sqSocketRemoteAddress(SocketPtr s)
{ struct sockaddr_in sin;
  int sinSize = sizeof(sin);
  int err;

  if (!SocketValid(s)) return -1;

  LOCKSOCKET(MUTEX(s), INFINITE);
  err = getpeername(SOCKET(s), (struct sockaddr *)&sin, &sinSize);
  UNLOCKSOCKET(MUTEX(s));

  if(err) return 0;
  if(sin.sin_family != AF_INET) return 0; /* can't handle other than internet addresses */
  return ntohl(sin.sin_addr.s_addr);
}

/*****************************************************************************
  sqSocketRemotePort: Return the port of the socket on the remote host
*****************************************************************************/
int sqSocketRemotePort(SocketPtr s)
{ struct sockaddr_in sin;
  int sinSize = sizeof(sin);
  int err;

  if (!SocketValid(s)) return -1;

  LOCKSOCKET(MUTEX(s), INFINITE);
  err = getpeername(SOCKET(s), (struct sockaddr *)&sin, &sinSize);
  UNLOCKSOCKET(MUTEX(s));

  if(err) return 0;
  if(sin.sin_family != AF_INET) return 0; /* can't handle other than internet addresses */
  return ntohs(sin.sin_port);
}

/*****************************************************************************
  sqSocketSendDataBufCount:
  Send bufSize bytes from the data pointed to by buf. 
  Return the number of bytes sent.
*****************************************************************************/
int sqSocketSendDataBufCount(SocketPtr s, int buf, int bufSize)
{ int minBuf;
  privateSocketStruct *pss = PSP(s);

  if (!SocketValid(s)) return -1;

  if(SOCKETERROR(s)) return -1;
  if(pss->sendDataNum) return 0; /* send in progress */

  /* Locking should actually not be necessary, but it's
     probably polite anyways ;-) */
  LOCKSOCKET(MUTEX(s), INFINITE);
  minBuf = MIN(bufSize, pss->sendDataSize);
  MoveMemory(pss->sendData, (void*) buf, minBuf);
  pss->sendDataNum = minBuf;
  UNLOCKSOCKET(MUTEX(s));
  /* Wake up the send thread */
  SetEvent(pss->sendEvent);
  return minBuf;
}

/*****************************************************************************
  sqSocketSendDone: Return non-zero if all data has been sent.
*****************************************************************************/
int sqSocketSendDone(SocketPtr s)
{ int doneData;
  privateSocketStruct *pss = PSP(s);

  if (!SocketValid(s)) return 1;
  LOCKSOCKET(MUTEX(s), INFINITE);
  doneData = pss->sendDataNum == 0;
  UNLOCKSOCKET(MUTEX(s));
  return doneData;
}



/*****************************************************************************
 *****                     Resolver Functions                            *****
 *****************************************************************************
 NOTE: Resolver functions don't need synchronization - there is only one
       resolver process running at a time.
 *****************************************************************************/

static char lastName[MAXHOSTNAMELEN+1];
static int lastAddr;

static int lastError;
static HANDLE asyncLookupHandle = 0;
static char hostentBuffer[MAXGETHOSTSTRUCT];

/*****************************************************************************
  Convenience functions. We may use them later.
*****************************************************************************/
static char *
nameOf(int netAddress)
{ u_long nAddr;
  struct hostent *he;

  lastError = 0;
  nAddr = htonl(netAddress);
  he = gethostbyaddr((char*)&nAddr,sizeof(nAddr),AF_INET);
  if(he) return he->h_name;
  lastError = h_errno;
  return "";
}

static int
addressOf(char *hostName)
{ struct hostent *he;

  lastError = 0;
  he = gethostbyname(hostName);
  if(he) return ntohl(*(long*)(he->h_addr_list[0]));
  lastError = h_errno;
  return 0;
}

/*****************************************************************************
  sqResolverAbort: Abort a pending DNS lookup
*****************************************************************************/
void sqResolverAbort(void)
{
  if(!asyncLookupHandle) return; /* lookup already finished */
  TerminateThread(asyncLookupHandle, 0);
  /* indicate finished operation */
  asyncLookupHandle = 0;
}

/*****************************************************************************
  sqResolverAddrLookupResult: Return the result of the last name lookup
*****************************************************************************/
void sqResolverAddrLookupResult(char *nameForAddress, int nameSize)
{
  MoveMemory(nameForAddress, lastName, nameSize);
}

/*****************************************************************************
  sqResolverAddrLookupResult: Return sizeof(result) of the last name lookup
*****************************************************************************/
int sqResolverAddrLookupResultSize(void)
{
  return strlen(lastName);
}

/*****************************************************************************
  sqResolverError: Return the last error of a DNS lookup
*****************************************************************************/
int sqResolverError(void)
{
  return lastError;
}

/*****************************************************************************
  sqResolverLocalAddress: Return the address of the local host
*****************************************************************************/
int sqResolverLocalAddress(void)
{
  return addressOf(localHostName);
}

/*****************************************************************************
  sqResolverNameLookupResult: Return the address of the last DNS lookup
*****************************************************************************/
int sqResolverNameLookupResult(void)
{
  return lastAddr;
}

/*****************************************************************************
  sqResolverStartAddrLookup: Look up the name to a given address
*****************************************************************************/
void sqResolverStartAddrLookup(int address)
{
  DWORD id;
  if(asyncLookupHandle) return; /* lookup in progress */
  asyncLookupHandle =
    CreateThread(NULL,                    /* No security descriptor */
                 0,                       /* default stack size     */
                 (LPTHREAD_START_ROUTINE) &sqGetHostByAddr, /* what to do */
                 (LPVOID) address,        /* parameter for thread   */
                 CREATE_SUSPENDED,        /* creation parameter -- create suspended so we can check the return value */
                 &id);                    /* return value for thread id */
  if(!asyncLookupHandle)
    printLastError(TEXT("CreateThread() failed"));
  /* lookups run with normal priority */
  if(!SetThreadPriority(asyncLookupHandle, THREAD_PRIORITY_NORMAL))
    printLastError(TEXT("SetThreadPriority() failed"));
  if(!ResumeThread(asyncLookupHandle))
    printLastError(TEXT("ResumeThread() failed"));
}

/*****************************************************************************
  sqResolverStartNameLookup: Look up the address to a given host name
*****************************************************************************/
void sqResolverStartNameLookup(char *hostName, int nameSize)
{ int len;
  DWORD id;

  if(asyncLookupHandle) return; /* lookup in progress */
  len = nameSize < MAXHOSTNAMELEN ? nameSize : MAXHOSTNAMELEN;
  MoveMemory(lastName,hostName, len);
  lastName[len] = 0;
  lastError = 0;
  asyncLookupHandle =
    CreateThread(NULL,                    /* No security descriptor */
                 0,                       /* default stack size     */
                 (LPTHREAD_START_ROUTINE) &sqGetHostByName, /* what to do */
                 (LPVOID) lastName,       /* parameter for thread   */
                 CREATE_SUSPENDED,        /* creation parameter -- create suspended so we can check the return value */
                 &id);                    /* return value for thread id */
  if(!asyncLookupHandle)
    printLastError(TEXT("CreateThread() failed"));
  /* lookups run with normal priority */
  if(!SetThreadPriority(asyncLookupHandle, THREAD_PRIORITY_NORMAL))
    printLastError(TEXT("SetThreadPriority() failed"));
  if(!ResumeThread(asyncLookupHandle))
    printLastError(TEXT("ResumeThread() failed"));
}

/*****************************************************************************
  sqResolverStatus: Return resolver status
*****************************************************************************/
int sqResolverStatus(void)
{
  if(!thisNetSession)
    return RESOLVER_UNINITIALIZED; /* not initialized */

  if(asyncLookupHandle)
    return RESOLVER_BUSY; /* lookup in progress */

  if(lastError)
    return RESOLVER_ERROR; /* resolver idle but last request failed */

  return RESOLVER_SUCCESS; /* ready and idle */
}



/*****************************************************************************
 sqGetHostByAddr: Perform a threaded gethostbyaddr()
*****************************************************************************/
DWORD WINAPI sqGetHostByAddr(int netAddress)
{ struct hostent *he;
  u_long nAddr;

  nAddr = htonl(netAddress);
  he = gethostbyaddr((char*)&nAddr, 4, PF_INET);
  if(he) 
    {
      strcpy(lastName,he->h_name);
      lastAddr = ntohl(*(long*)(he->h_addr_list[0]));
    }
  else
    lastError = WSAGetLastError();
  asyncLookupHandle = 0;
  synchronizedSignalSemaphoreWithIndex(resolverSemaphoreIndex);
  ExitThread(0);
}


/*****************************************************************************
 sqGetHostByName: Perform a threaded gethostbyname()
*****************************************************************************/
DWORD WINAPI sqGetHostByName(char *hostName)
{ struct hostent *he;

  he = gethostbyname(hostName);
  if(he) 
    {
      strcpy(lastName,he->h_name);
      lastAddr = ntohl(*(long*)(he->h_addr_list[0]));
    }
  else
    lastError = WSAGetLastError();
  asyncLookupHandle = 0;
  synchronizedSignalSemaphoreWithIndex(resolverSemaphoreIndex);
  ExitThread(0);
}

#endif /* NO_NETWORK */
