/****************************************************************************
*   PROJECT: Squeak port for Win32 (NT / Win95)
*   FILE:    sqWin32Directory.c
*   CONTENT: Directory management
*
*   AUTHOR:  Andreas Raab (ar)
*   ADDRESS: University of Magdeburg, Germany
*   EMAIL:   raab@isg.cs.uni-magdeburg.de
*   RCSID:   $Id: sqWin32Directory.c,v 2.0.1.5 1998/10/02 22:24:02 raab Exp $
*
*   NOTES:
*    1) Caching is not supported (as in the unix version) because there is
*       no noticeable delay.
*
*****************************************************************************/
#include <windows.h>
#include "sq.h"

#ifndef NO_RCSID
  static char RCSID[]="$Id: sqWin32Directory.c,v 2.0.1.5 1998/10/02 22:24:02 raab Exp $";
#endif

/***
	The interface to the directory primitive is path based.
	That is, the client supplies a Squeak string describing
	the path to the directory on every call. To avoid traversing
	this path on every call, a cache is maintained of the last
	path seen, along with the Mac volume and folder reference
	numbers corresponding to that path.
***/

/*** Constants ***/
#define ENTRY_FOUND     0
#define NO_MORE_ENTRIES 1
#define BAD_PATH        2

static TCHAR DELIMITER[] = TEXT("\\");
static TCHAR DOT[] = TEXT(".");

DWORD convertToSqueakTime(SYSTEMTIME st)
{ DWORD secs;
  DWORD dy;
  static DWORD nDaysPerMonth[14] = { 
    0,  0,  31,  59,  90, 120, 151,
      181, 212, 243, 273, 304, 334, 365 };
  /* Squeak epoch is Jan 1, 1901 */
  dy = st.wYear - 1901; /* compute delta year */
  secs = dy * 365 * 24 * 60 * 60       /* base seconds */
         + (dy >> 2) * 24 * 60 * 60;   /* seconds of leap years */
  /* check if month > 2 and current year is a leap year */
  if(st.wMonth > 2 && (dy & 0x0003) == 0)
    secs += 24 * 60 * 60; /* add one day */
  /* add the days from the beginning of the year */
  secs += (nDaysPerMonth[st.wMonth] + st.wDay - 1) * 24 * 60 * 60;
  /* add the hours, minutes, and seconds */
  secs += st.wSecond + 60*(st.wMinute + 60*st.wHour);
  return secs;
}

int dir_Create(char *pathString, int pathStringLength)
{
  /* Create a new directory with the given path. By default, this
     directory is created relative to the cwd. */
  return CreateDirectory(fromSqueak(pathString, pathStringLength),NULL);
}

int dir_Delimitor(void)
{
  return DELIMITER[0];
}

int dir_Lookup(char *pathString, int pathStringLength, int index,
/* outputs: */ char *name, int *nameLength, int *creationDate, int *modificationDate,
	       int *isDirectory, int *sizeIfFile)
{
  /* Lookup the index-th entry of the directory with the given path, starting
     at the root of the file system. Set the name, name length, creation date,
     creation time, directory flag, and file size (if the entry is a file).
     Return:	0 	if a entry is found at the given index
     		1	if the directory has fewer than index entries
		2	if the given path has bad syntax or does not reach a directory
  */
  TCHAR *win32Path;
  WIN32_FIND_DATA findData;
  HANDLE findHandle;
  FILETIME fileTime;
  SYSTEMTIME sysTime;
  int i,nameLen,mask;

  /* default return values */
  *name             = 0;
  *nameLength       = 0;
  *creationDate     = 0;
  *modificationDate = 0;
  *isDirectory      = false;
  *sizeIfFile       = 0;

#if defined(_WIN32_WCE)
	/* Like Unix, Windows CE does not have drive letters */
#else /* defined(_WIN32_WCE) */
  if(pathStringLength == 0)
    { /* we're at the top of the file system --- return possible drives */
      mask = GetLogicalDrives();
      for(i=0;i<26; i++)
        if(mask & (1 << i))
      if(--index == 0) /* found the drive ! */
        {
          name[0]           = 'A'+i;
          name[1]	          = ':';
          *nameLength       = 2;
          *creationDate     = 0;
          *modificationDate = 0;
          *isDirectory      = true;
          *sizeIfFile       = 0;
          return ENTRY_FOUND;
        }
      return NO_MORE_ENTRIES;
    }
#endif /* defined(_WIN32_WCE) */

  win32Path = fromSqueak(pathString, pathStringLength);
  if(win32Path[pathStringLength-1] != DELIMITER[0])
    lstrcat(win32Path,DELIMITER);
	lstrcat(win32Path,TEXT("*"));

	findHandle = FindFirstFile(win32Path,&findData);
	if(findHandle == INVALID_HANDLE_VALUE)
	{
	  /* Directory could be empty, so we must check for that */
		DWORD dwErr = GetLastError();
		return dwErr = ERROR_NO_MORE_FILES ? NO_MORE_ENTRIES : BAD_PATH;
	}
	while(1)
	  {
      nameLen = lstrlen(findData.cFileName);
      if (nameLen < 3 && findData.cFileName[0] == DOT[0])
        if (nameLen == 1 || findData.cFileName[1] == DOT[0])
          index = index + 1; /* hack us back to the last index */
      if(index <= 1) break;
		  if (!FindNextFile(findHandle,&findData))
		    {
			    FindClose(findHandle);
			    return NO_MORE_ENTRIES;
		    }
      index = index - 1;
	  }
	FindClose(findHandle);

	*nameLength = lstrlen(findData.cFileName);
	for (i=0;i<=*nameLength;i++)
		name[i] = (char)findData.cFileName[i];

  FileTimeToLocalFileTime(&findData.ftCreationTime, &fileTime);
  FileTimeToSystemTime(&fileTime, &sysTime);
  *creationDate = convertToSqueakTime(sysTime);
  FileTimeToLocalFileTime(&findData.ftLastWriteTime, &fileTime);
  FileTimeToSystemTime(&fileTime, &sysTime);
  *modificationDate = convertToSqueakTime(sysTime);

  if (findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
    *isDirectory= true;
  else
    *sizeIfFile= findData.nFileSizeLow; /* assuming that this is enough ;-) */

  return ENTRY_FOUND;
}

dir_SetMacFileTypeAndCreator(char *filename, int filenameSize,
			     char *fType, char *fCreator)
{
  /* dos files are untyped, and the creator is correct by default */
  return true;
}
