/****************************************************************************
*   PROJECT: Squeak port for Win32 (NT / Win95)
*   FILE:    sqWin32Alloc.c
*   CONTENT: Virtual Memory Management
*
*   AUTHOR:  Andreas Raab (ar)
*   ADDRESS: University of Magdeburg, Germany
*   EMAIL:   raab@isg.cs.uni-magdeburg.de
*   RCSID:   $Id: sqWin32Alloc.c,v 2.0.1.4 1998/10/02 23:17:44 raab Exp $
*
*   NOTES: How does all of this work?
*
*     What we're basically doing is allocating memory in a reserved state
*     so that no other application can get this portion of memory. No single bit
*     of the reserved memory is actually being used until it will be accessed
*     for the first time. This raises an exception which is being handled in
*     sqWin32Window.c
*
*     Once an exception occurs, we know that Squeak really needs this portion
*     of memory and map it into address space. That's all.
*
*     By using this simple scheme we only take the memory Squeak actually wants
*     to have, leaving as much space as possible for other applications.
*     Unfortunately, we can not yet release memory to the system since we don't
*     know if squeak needs a certain portion of memory any more.
*
*****************************************************************************/
#include <windows.h>
#include "sq.h"

#ifndef NO_VIRTUAL_MEMORY

#ifndef NO_RCSID
  static char RCSID[]="$Id: sqWin32Alloc.c,v 2.0.1.4 1998/10/02 23:17:44 raab Exp $";
#endif

static LPSTR  pageBase;     /* base address of allocated memory */
static DWORD  pageBits;     /* bit mask for the start of a memory page */
static DWORD  pageSize;     /* size of a memory page */
static DWORD  maxReserved;  /* maximum reserved virtual memory */

/* Just to see how much memory has actually being used */
static DWORD usedMemory;

/* this macro returns the page for a given pointer */
#define PAGE(ptr) ((void*)((unsigned long)ptr & (pageBits) ))

/************************************************************************/
/* sqMemError: raise an error message box and return 0                  */
/************************************************************************/
static int sqMemError(TCHAR *fctnCall,int lineNumber)
{ int lastErr;
  MEMORYSTATUS mStat;

  /* make sure the following doesn't raise the next exception */
  __try {
    lastErr = GetLastError();
    sqMessageBox(MB_TASKMODAL | MB_OK,
                 TEXT("Fatal Virtual Memory Error!"),
                 "Function call %s in line %d of sqWin32Alloc.c failed.\n GetLastError() returned %d\n%s",
                 fctnCall,lineNumber,lastErr,
#ifdef NO_RCSID
                 ""
#else /* NO_RCSID */
                 RCSID
#endif /* NO_RCSID */
                 );
    ZeroMemory(&mStat,sizeof(mStat));
    mStat.dwLength = sizeof(mStat);
    GlobalMemoryStatus(&mStat);
    sqMessageBox(MB_TASKMODAL | MB_OK,
                 TEXT("Memory Information"),
                 "Memory State:\n  Physical: \t%d \t/ \t%d\n  Virtual: \t%d \t/ \t%d",
                 mStat.dwAvailPhys,mStat.dwTotalPhys,
                 mStat.dwAvailVirtual,mStat.dwTotalVirtual);
 } __finally{}
  return 0;
}

/************************************************************************/
/* sqMemWarnUser: raise a message box and wait for continue or cancel   */
/************************************************************************/
#if 0 /* currently unused */
static int sqMemWarnUser(LPEXCEPTION_POINTERS exp, TCHAR *warnString)
{ void *pointer_to_cs = (void*) sqMemWarnUser;
  MEMORY_BASIC_INFORMATION mStat;

  /* find the address where we have been loaded */
  if(!VirtualQuery(pointer_to_cs,&mStat,sizeof(mStat)))
    return sqMemError(TEXT("sqMemWarnUser::VirtualQuery()"),__LINE__);

  return sqMessageBox(MB_ABORTRETRYIGNORE | MB_TASKMODAL | MB_SETFOREGROUND,
                      TEXT("Virtual Memory Error"),
                      "%s\nAccess from address CS:%x (AllocationBase=%x)\n Accessed address is DS:%x",
                      warnString, (DWORD) exp->ExceptionRecord->ExceptionAddress - (DWORD) mStat.AllocationBase,
                      mStat.AllocationBase, exp->ExceptionRecord->ExceptionInformation[1]) != ID_ABORT);
}
#endif /* 0 */

/************************************************************************/
/* sqMemIsReservedPage: Return 1 if the page is in reserved memory      */
/************************************************************************/
int sqMemIsReservedPage(void *page)
{ MEMORY_BASIC_INFORMATION mStat;

  /* check if the page is within range */
  if( (char*) page < pageBase || (char*)page >= pageBase+maxReserved)
    return 0;

  /* get memory information about page */
  if(!VirtualQuery(page,&mStat,sizeof(mStat)))
      sqMemError(TEXT("sqMemIsReservedPage::VirtualQuery()"),__LINE__);

  /* the page is acceptable if it is reserved and has no access yet */
  if(mStat.State == MEM_RESERVE && mStat.AllocationProtect == PAGE_NOACCESS)
    return 1;

  return 0;
}

/************************************************************************/
/* sqExceptionFilter: base filter for exception handling                */
/************************************************************************/
#ifndef NDEBUG
/* in debug mode, let the system crash so that we can see where it happened */
#define EXCEPTION_WRONG_ACCESS EXCEPTION_CONTINUE_SEARCH
#else
/* in release mode, execute the exception handler notifying the user what happened */
#define EXCEPTION_WRONG_ACCESS EXCEPTION_EXECUTE_HANDLER
#endif

LONG CALLBACK sqExceptionFilter(LPEXCEPTION_POINTERS exp)
{ void *Page;

  /* make the following as save as possible */
  /* if we have an exception inside this piece of code, */
  /* something is definitely wrong! */
  __try {

  /* if we're running in an access violation */
  if(exp->ExceptionRecord->ExceptionCode == EXCEPTION_ACCESS_VIOLATION)
    {
      /* get the page */
      Page = PAGE(exp->ExceptionRecord->ExceptionInformation[1]);

      /* can we catch the page ? */
      if(!sqMemIsReservedPage(Page))
        return EXCEPTION_WRONG_ACCESS; /* No */

      /* try to load the page in memory */
      if(VirtualAlloc(Page,pageSize,MEM_COMMIT,PAGE_READWRITE))
        {
          usedMemory += pageSize;
          return EXCEPTION_CONTINUE_EXECUTION;
        }
      /* if this does for some reason not work there is probably not enough
         swap left on your device */
      sqMemError(TEXT("sqExceptionFilter::VirtualAlloc()"),__LINE__);
    }
  return EXCEPTION_WRONG_ACCESS;
  
  /* if we get this far, we had an exception in the handler itself */
  } __except(EXCEPTION_EXECUTE_HANDLER) {
    /* if we get at this point we can't do anything anymore */
    sqMemError(TEXT("sqExceptionFilter::__except()"),__LINE__);
  }
  return EXCEPTION_WRONG_ACCESS;
}

/************************************************************************/
/* sqAllocateMemory: Initialize virtual memory                          */
/************************************************************************/
void *sqAllocateMemory(int minHeapSize, int desiredHeapSize)
{ SYSTEM_INFO sysInfo;
  DWORD initialCommit, commit;

  /* determine page boundaries */
  GetSystemInfo(&sysInfo);
  pageSize = sysInfo.dwPageSize;
  pageBits = ~(pageSize - 1);

  /* round the requested size up to the next page boundary */
  maxReserved = (desiredHeapSize + pageSize) & pageBits;

  /* round the initial commited size up to the next page boundary */
  initialCommit = (minHeapSize + pageSize) & pageBits;

  /* Here, we only reserve the maximum memory to be used
     It will later be committed during actual access */
  pageBase = VirtualAlloc(NULL,maxReserved,MEM_RESERVE, PAGE_NOACCESS);
  if(!pageBase) return 0;

  /* commit initial memory as requested */
  commit = maxReserved < initialCommit ? maxReserved : initialCommit;
  if(!VirtualAlloc(pageBase, commit, MEM_COMMIT, PAGE_READWRITE))
    sqMemError(TEXT("sqMalloc::VirtualAlloc()"),__LINE__);

  usedMemory += commit;
  return pageBase;
}

/************************************************************************/
/* sqMemReleaseMemory: Release virtual memory                            */
/************************************************************************/
void sqReleaseMemory(void)
{
  /* Win32 will do that for us */
}

#endif /* NO_VIRTUAL_MEMORY */
