/****************************************************************************
*   PROJECT: Squeak port for Win32 CE (NT / Win95)
*   FILE:    sqWinCEStdio.c
*   CONTENT: Routines to replace C runtime library stdio routines
*
*   AUTHOR:  Blair McGlashan
*   ADDRESS: Intuitive Systems Ltd, Stevenage, Herts SG1 3EE, UK
*   EMAIL:   blair@intuitive.co.uk
*   RCSID:   
*
*   NOTES:
*			To keep the size of these routines down, they do not implement
*			the full functionality of the associated C runtime library
*			routine (especially fopen()), Error checking, in particular, is
*			omitted.
*
*****************************************************************************/
#define __NOSQMALLOC
#include "sq.h"

// N.B. We must include windows.h, otherwise we may pick up the wrong functions
// under the emulator.
#include <windows.h>

//#include <errno.h>
#define ENOENT 2
#define EACCES 13


#if !defined(UNICODE) && !defined(_UNICODE)
	#error "Requires unicode"
#endif


int vtrace(const TCHAR* szFormat, va_list args)
{
	TCHAR buf[512];
	int ret;

	ret = wvsprintf(buf, szFormat, args);
	OutputDebugString(buf);
	// MessageBox(0,toUnicode(buf),TEXT("SqueakCE (vtrace)"),MB_OK); // Ale
	return ret;
}

int printf(const char* fmt, ...)
{
	TCHAR* szFormat;
	va_list args;
	int ret;

	szFormat = toUnicode(fmt);
	va_start(args, fmt);
	ret = vtrace(szFormat, args);
	va_end(args);
	free(szFormat);
	return ret;
}


// Simulation of CRT fseek(). Not not buffered
int fseek(FILE *s, long offset, int origin)
{
	DWORD pos = SetFilePointer(s, offset, NULL, origin);
#ifdef _DEBUG
	if (pos == (DWORD)-1)
		printf("fseek(%x,%d,%d) failed with %d\n", s, offset, origin, GetLastError());
#endif
	return pos;
}

long ftell(FILE* s)
{
	DWORD pos = SetFilePointer(s,0,NULL,FILE_CURRENT);
#ifdef _DEBUG
	if (pos == (DWORD)-1)
		printf("ftell(%x) failed with %d\n", s, GetLastError());
#endif
	return pos;
}


size_t fread(void *buffer, size_t size, size_t count, FILE *stream)
{
	DWORD numberOfBytesRead;
	ReadFile(stream, buffer, size*count, &numberOfBytesRead, NULL);
	return numberOfBytesRead/size;
}

size_t fwrite(const void *buffer, size_t size, size_t count, FILE *stream)
{
	DWORD numberOfBytesWritten;
	WriteFile(stream, buffer, size*count, &numberOfBytesWritten, NULL);
	return numberOfBytesWritten/size;
}

// N.B. This does very little error checking (in order to save space), and does
// not correctly support the 'a' mode (i.e. it is possible to overwrite existing data)
// This is OK in the Squeak at present, because it does not rely on the autoseek facility
// (in fact it only ever opens a file in 'a+' mode if it can't open it in 'r+' mode, which
// should only happen if the file does not exist, and therefore 'a+' ought really to be 'w+'
// for that usage).
FILE* _wfopen(const wchar_t *cFilename, const wchar_t *cMode)
{
	DWORD dwDesiredAccess = 0;
	DWORD dwCreationDistribution = 0;
	unsigned i;
	HANDLE fh;

	for (i=0;i<lstrlen(cMode);i++)
	{
		switch(cMode[i])
		{
		case 'r':
			dwDesiredAccess |= GENERIC_READ;
			dwCreationDistribution = OPEN_EXISTING;
			break;
		case 'w':
			dwDesiredAccess |= GENERIC_WRITE;
			dwCreationDistribution = TRUNCATE_EXISTING;
			break;
		case 'a':
			dwDesiredAccess = GENERIC_READ|GENERIC_WRITE;
			dwCreationDistribution = OPEN_ALWAYS;
			break;
		case '+':
			dwDesiredAccess |= GENERIC_WRITE;
			break;

		case 'b':
		case 't':
		case 'c':
		case 'n':
			break;

		default:
			return NULL;
		}
	}

	fh = CreateFile(
		cFilename,
		dwDesiredAccess,
		FILE_SHARE_READ|FILE_SHARE_WRITE,
		NULL,
		dwCreationDistribution,
		FILE_ATTRIBUTE_NORMAL,
		NULL);
	return fh == INVALID_HANDLE_VALUE ? 0 : fh;
}

// ANSI version
FILE* fopen(const char *filename, const char *mode)
{
	WCHAR* cFilename;
	WCHAR* cMode;
	FILE* f;

	cFilename = toUnicode(filename);
	cMode = toUnicode(mode);
	f = _wfopen(cFilename, cMode);
	free(cFilename);
	free(cMode);
	return f;
}

int fclose(FILE *f)
{
	return CloseHandle(f) ? 0 : EOF;
}

int remove(const char *path)
{
	WCHAR* cFilename;
	int ret;
	cFilename = toUnicode(path);
	ret = DeleteFile(cFilename) ? 0 : -1;
	free(cFilename);
	return ret;
}

int rename(const char *oldname, const char* newname)
{
	TCHAR* cOldName;
	TCHAR* cNewName;
	BOOL bSuccess;

	cOldName = toUnicode(oldname);
	cNewName = toUnicode(newname);
	bSuccess = MoveFile(cOldName, cNewName);
	free(cOldName);
	free(cNewName);

	if (!bSuccess)
	{
		DWORD err = GetLastError();
		errno = err == ERROR_FILE_NOT_FOUND || 
						err == ERROR_PATH_NOT_FOUND ?
							ENOENT : EACCES;
		return -1;
	}
	else
		return 0;
}

int mkdir(const char *path)
{
	TCHAR tpath[MAX_PATH+1];
	wsprintf(tpath, TEXT("%hs"), path);
	return CreateDirectory(tpath, (LPSECURITY_ATTRIBUTES)NULL) ? 0 : -1;
}

int fprintf(FILE* f, const char* fmt, ...)
{
	TCHAR buf[256];
	va_list args;
	int ret;

	TCHAR* szFormat = toUnicode(fmt);
	va_start(args, fmt);
	ret = wvsprintf(buf, szFormat, args);
	va_end(args);
	free(szFormat);
	{
		DWORD dwWritten;
		WriteFile((HANDLE)f, &buf, ret*sizeof(TCHAR), &dwWritten, NULL);
	}
	return ret;
}

int putchar(int c)
{
	TCHAR buf[2];
	buf[0] = c;
	buf[1] = 0;
	OutputDebugString(buf);
}
