/****************************************************************************
*   PROJECT: Squeak port for Win32 (WinCE)
*   FILE:    sqWin32CE.c
*   CONTENT: Special support stuff only for WinCE on SHx
*
*   AUTHOR:  Alejandro F. Reimondo (afr)
*   ADDRESS: SUGAR - Argentina
*   EMAIL:   aleReimondo@sugarWeb.com
*   RCSID:   $Id: sqWin32CE.c,v 1.1 1998/07/23 20:48:54 afr Exp $
*
*   NOTES:
*    1) Support code for VM under WindowsCE 2.00 (_WIN32_WCE)
*****************************************************************************/
#include <windows.h>
#include <commdlg.h>
#include "sq.h"
#include "sqWin32Args.h"

/*** Variables -- Imported from Virtual Machine ***/
extern int fullScreenFlag;

/* Import from the Virtual Machine */
int byteSwapped(int);

/* Import from sqWin32Alloc.c */
LONG CALLBACK sqExceptionFilter(LPEXCEPTION_POINTERS exp);

/*** Variables -- command line */
char *initialCmdLine;
int  numOptionsVM = 0;
char *(vmOptions[MAX_OPTIONS]);
int  numOptionsImage = 0;
char *(imageOptions[MAX_OPTIONS]);

/****************************************************************************/
/*                           Jitter Stuff                                   */
/****************************************************************************/

/* Jitter is not supported directly by this VM */


/****************************************************************************/
/*                      Console Window functions                            */
/****************************************************************************/

/* Note: Console output is not supported by this VM */


/****************************************************************************/
/*                          main                                            */
/****************************************************************************/

int PASCAL WinMain (HINSTANCE hInst,
                    HINSTANCE hPrevInstance,
                    LPSTR  lpCmdLine,
                    int    nCmdShow)
{ int virtualMemory;
  sqImageFile imageFile;
  int imageSize;

  /* fetch us the name of the executable */
  GetModuleFileName(hInst, vmName, MAX_PATH);

  /* get us the instance handle */
  hInstance = hInst;

  // No command line, so pop up an open file dialog
  if(*lpCmdLine == 0)
	{
		OPENFILENAME ofn;
		memset(&ofn, 0, sizeof(ofn));
		ofn.lStructSize = sizeof(ofn);
		ofn.lpstrFilter = TEXT("Image Files (*.image)\0*.image\0All Files (*.*)\0*.*\0");
		ofn.lpstrFile = vmPath;
		ofn.nMaxFile = MAX_PATH;
		ofn.lpstrTitle = TEXT("Open Squeak image ...");
		ofn.Flags = OFN_EXPLORER | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
		ofn.lpstrDefExt = TEXT("image");
		if (!GetOpenFileName(&ofn))
			return -1;
		strcpy(imageName, fromUnicode(vmPath));
	}
  else
  {
	/* parse command line args - this should be done a little  more sophisticated*/
	strcpy(imageName, fromUnicode(lpCmdLine));
  }

  // Hard code the path name (this is not a great hardship in a driveless 4Mb filestore)
  lstrcpy(vmPath, TEXT("\\"));

  //SetupFilesAndPath();

  // Set the wait cursor because reading the image can take a 
  // long time on CE.
  SetCursor((HCURSOR)IDC_WAIT);

  /* initialisation */
  SetupKeymap();
  SetupWindows();
  SetupPixmaps();
  SetupPrinter();
  SetupTimer();
#ifndef NO_MIDI
  SetupMIDI();
#endif
  sqFileInit();
#ifndef NO_JOYSTICK
  joystickInit();
#endif

  /* check the interpreter's size assumptions for basic data types */
  if (sizeof(int) != 4)    error("This C compiler's integers are not 32 bits.");
  if (sizeof(double) != 8) error("This C compiler's floats are not 64 bits.");
  if (sizeof(time_t) != 4) error("This C compiler's time_t's are not 32 bits.");

  imageSize = SqueakImageLength(toUnicode(imageName));
  if(imageSize == 0)
  {
	  printUsage(2);
	  exit(1);
  }
  /* allocate the synchronization mutex before anything is going to happen */
  vmSemaphoreMutex = CreateMutex(NULL, 0, NULL);

#ifdef NO_VIRTUAL_MEMORY
  if(!dwMemorySize) dwMemorySize = 4;

#if defined(_WIN32_WCE)
  virtualMemory = initializeCEMemorySpace(imageSize);
#else //  defined(_WIN32_WCE)
  virtualMemory = (int)imageSize + max(imageSize, dwMemorySize * 0x00100000);
#endif //  defined(_WIN32_WCE)

#else // NO_VIRTUAL_MEMORY

#if defined(_WIN32_WCE)
  if(!dwMemorySize) dwMemorySize = 4;
  virtualMemory = initializeCEMemorySpace(imageSize);
#else //  defined(_WIN32_WCE)
  if(!dwMemorySize) dwMemorySize = MAX_VIRTUAL_MEMORY;
  virtualMemory = max((int)dwMemorySize * 0x00100000, (int) 2*imageSize);
#endif // defined(_WIN32_WCE)

  /* The exception handler. 
     Everything happening in here will be caught by the handler at the end */
  __try { 

#endif // NO_VIRTUAL_MEMORY

  /* open and read the image file */
  imageFile = sqImageFileOpen(imageName,"rb");
  readImageFromFileHeapSize(imageFile, virtualMemory);
  sqImageFileClose(imageFile);

  ioSetFullScreen(fullScreenFlag);

  /* Note: There is good reason for putting the prefs here. */
  //SetupPreferences();
  fDeferredUpdate = 0; /* No deferred update under CE */
  //fReduceCPUUsage = 1; /* Should we reduce CPU usage? */

  /* Restore normal cursor */
  SetCursor(NULL);

  /* display the main window */
  //SetWindowSize();   unused under CE
  if(!fHeadlessImage) 
    ShowWindow(stWindow,nCmdShow);

  /* run Squeak */
  interpret();

#ifndef NO_VIRTUAL_MEMORY
  } __except(sqExceptionFilter(GetExceptionInformation())) {
    /* if we get this far our app has finally crashed */
    abortMessage(TEXT("Squeak has crashed!"));
  }
#endif
}

/****************************************************************************/
/*              Display and printing                                        */
/****************************************************************************/

int reverse_image_bytesCE(DWORD* dest, DWORD *const src,int depth, int width, RECT *const rect)
{ 
	int scanLine, pixPerWord,pixPerM1;
	int startX, stopX, startY, stopY;
	int i,j,xLen;
	
	pixPerWord= 32/depth;
	pixPerM1= pixPerWord-1;
	scanLine= ((width+pixPerM1) & ~pixPerM1) / pixPerWord;
	startX = rect->left / pixPerWord;
	stopX = rect->right / pixPerWord + 1;
	xLen = stopX - startX;
	
	startY = rect->top * scanLine;
	stopY = rect->bottom * scanLine;
	for(j = startY; j < stopY; j += scanLine)
	{
		DWORD* srcPixPtr = src + j + startX;
		DWORD* destPixPtr = dest + j + startX;
		i = xLen;
		while(i--)
		{
			DWORD bigEndian = ~*srcPixPtr++;
			*destPixPtr++ = ((bigEndian >> 24) & 0xFF) +
				((bigEndian >> 8) & 0xFF00) +
				((bigEndian << 8) & 0xFF0000) +
				((bigEndian << 24) & 0xFF000000);
		}
	}
	return 1;
}


int
reverse_image_words(int *ptr,int d, int w,int h,int x0,int y0,int x1,int y1)
{ int scanLine, pixPerWord;
  int startX, stopX, startY, stopY;
  int *linePtr,*pixPtr;
  int j,xLen;
  const int log2Depth=4;    /* log2Depth = log2(d) = log2(16) */
  const int pixPerWordShift = 5-log2Depth;

  if (d!=16)
	return 0;  /* The person calling us is confused.  We only deal with 16 bit words */

  pixPerWord= 1 << pixPerWordShift;  /* was = 32/depth */
  scanLine= (w+(pixPerWord-1)) >> pixPerWordShift;  /* words per scan line */
  startX = (x0 >> pixPerWordShift);
  stopX  = (x1 >> pixPerWordShift) + 1;
  xLen = stopX - startX;
  startY = y0 * scanLine;
  stopY  = y1 * scanLine;
  if(stopX <= startX || stopY <= startY) return 1;
  for(j = startY; j < stopY; j += scanLine)
    {
      linePtr = ptr + j;
      pixPtr = linePtr + startX;
#ifdef _M_IX86
      /* Speed-up x86 code: Deal with a word (two 16 bit pixels) per iteration.  -mdr */
      /* When the data is in the cache, this does once word in 4 cycles */
      _asm {
        mov edx, pixPtr
        mov ecx, xLen
      next2pixels:
        mov eax,[edx]	;u1   32 bit loads/stores are fastest in 32 bit mode
        rol eax,16	;u1
        mov [edx],eax	;u1
        add edx,4	;v0
        dec ecx		;u1    the dec/jump pair is faster than loop
        jnz next2pixels	;v0
      }
#else
#define SWAPWORDS(l) (((((l) & 0xFFFF) << 16) | ((l) >> 16)))
      { int i;
        for (i=xLen; i--; pixPtr++)
          *pixPtr = SWAPWORDS(*pixPtr);
      }
#undef SWAPWORDS
#endif
    }
  return 1;
}

HDC createBitmapDC(HDC dc, int depth, int width, int height, void** pBitsOut)
{
	/* Cached DIBSection */
	static HBITMAP hbm = NULL;
	static int lastDepth = 0;
	static int lastWidth = 0;
	static int lastHeight = 0;
	static void* pBits;
	int i;
	static HDC memDC;

	switch(depth) 
	{
		case 1: i = 0; break;
		case 2: i = 1; break;
		default:
		{	error("Error invalid depth");
			return 0;
		}
	}

	if (depth != lastDepth || width != lastWidth || height != lastHeight)
	{
		lastDepth = depth;
		lastHeight = height;
		lastWidth = width;
		bmi1->bmiHeader.biWidth = width;
		bmi1->bmiHeader.biHeight = -height;
		if (hbm)
		{
			DeleteObject(hbm);
		}

		/* Don't really need to create such a large DIB */
		hbm = CreateDIBSection(dc, bmi1, DIB_RGB_COLORS, &pBits, NULL, 0);
		if (!hbm)
		{	error("Can't create bitmap.");
			return 0;
		}
	}

	*pBitsOut = pBits;

	memDC = CreateCompatibleDC(NULL);
	SelectObject(memDC, hbm);
	return memDC;
}

void releaseBitmapDC(HDC memDC)
{
	DeleteDC(memDC);
}

/****************************************************************************/
/*                      System Attributes                                   */
/****************************************************************************/
char * GetVMOption(int id) { return "";}
char * GetImageOption(int id) { return "";}


/****************************************************************************/
/*                           main                                           */
/****************************************************************************/

/* Uh - that's again a main function?! How many do we have? And why?

   The answer is that windows function can run either as a GUI
   application or as a console application. GUI apps are started
   via WinMain(), console apps via main(). Since some people might
   want to run Squeak as console app (imagine a small Squeak calculator)
   this main() function is given as a wrapper for WinMain.
*/

#ifdef _CONSOLE
int main(int argc, char **argv)
{ char *cmdLine;
  cmdLine = GetCommandLine();
  cmdLine += strlen(argv[0]);
  return WinMain(GetModuleHandle(NULL), NULL, cmdLine, SW_SHOW);
}
#endif


/****************************************************************************/
/*                           Other                                          */
/****************************************************************************/

/////////////////////////////////////////////////////////////////////////////////////////////
//  NOTE: Assumes that the library is already loaded and mapped into our address space.
//        We are really only interested in the DLL's handle.
HMODULE GetModuleHandle( LPCTSTR lpModuleName )
{
	HINSTANCE hInst;

	hInst = LoadLibrary( lpModuleName );

	// Decrement reference count.
	if (hInst)
		FreeLibrary( hInst );

	return (HMODULE)(hInst);
}