/****************************************************************************
*   PROJECT: Squeak port for Win32 (NT / Win95)
*   FILE:    sqWin32Intel.c
*   CONTENT: Special support stuff only for Win95/WinNT on x86
*
*   AUTHOR:  Andreas Raab (ar)
*   ADDRESS: University of Magdeburg, Germany
*   EMAIL:   raab@isg.cs.uni-magdeburg.de
*   RCSID:   $Id: sqWin32Intel.c,v 1.1 1998/07/22 20:48:54 raab Exp $
*
*   NOTES:
*    1) When using this module the virtual machine MUST NOT be compiled
*       with Unicode support.
*****************************************************************************/
#include <windows.h>
#include <stdio.h>
#include <fcntl.h> /* _O_BINARY */
#include "sq.h"
#include "sqWin32Args.h"

/*** Variables -- Imported from Virtual Machine ***/
extern int fullScreenFlag;

/*** Crash debug -- Imported from Virtual Machine ***/
extern const int primitiveIndex;
int getCurrentBytecode(void);
int printCallStack(void);


/* Import from sqWin32Alloc.c */
LONG CALLBACK sqExceptionFilter(LPEXCEPTION_POINTERS exp);


/*** Variables -- command line */
char *initialCmdLine;
int  numOptionsVM = 0;
char *(vmOptions[MAX_OPTIONS]);
int  numOptionsImage = 0;
char *(imageOptions[MAX_OPTIONS]);

/* console buffer */
TCHAR consoleBuffer[4096];

/* stderr and stdout names */
char stderrName[MAX_PATH+1];
char stdoutName[MAX_PATH+1];

TCHAR *logName = TEXT("");             /* full path and name to log file */

/* Service stuff */
TCHAR  serviceName[MAX_PATH+1];   /* The name of the NT service */
TCHAR *installServiceName = NULL; /* the name under which the service is to install */
BOOL  fBroadcastService95 = 0;   /* Do we need a broadcast when a user has logged on? */
UINT  WM_BROADCAST_SERVICE = 0;  /* The broadcast message we send */
TCHAR *msgBroadcastService = TEXT("SQUEAK_SERVICE_BROADCAST_MESSAGE"); /* The name of the broadcast message */


void SetSystemTrayIcon(BOOL on);


#if defined(JITTER) && defined(_MSC_VER) && defined(_M_IX86)
/* Works only when using the dynamic translator with MS VC++ on x86 */

/****************************************************************************/
/*                           Jitter Stuff                                   */
/****************************************************************************/
/* Note 1: We use naked methods whenever we need to determine the callers   */
/*         address. Naked methods contain no prologue and epilogue code.    */
/* Note 2: Using __fastcall convention allows to pass arguments in          */
/*         in registers (ecx and edx).                                      */
/****************************************************************************/

/* This function is called by the beginOp macro from the dynamic translator.
   It needs to determine the address of the actual opcode which will follow
   at some point after the callers IP. Since we have emitted a nop-sequence
   (0x90909090) we scan for this sequence and return the address past this
   sequence.

   Note: This method does _not_ need to be efficient, it is only called
         during startup from initialization code of the dynamic translator.
*/
__declspec(naked) int __fastcall set_label(int *address)
{
  __asm {
    /* fetch the return value into eax */
    mov eax,[esp]
    dec eax
    /* look up the byte sequence stored by beginOp */
  nextIP:
    inc eax
    cmp DWORD PTR [eax], 0x90909090
    jne nextIP
    /* Ok, we have found an entry. Now we have to make sure that the first
       bytes of the 0x90909090 sequence is not the end of any previous
       instruction sequence. We do this by looking forward until we find
       something different from a nop (this indicates the actual start
       of the opCode implementation
    */
  nextNOOP:
    inc eax
    cmp DWORD PTR [eax], 0x90909090
    je nextNOOP
    /* We are now three bytes before the actual start of the opCode implementation
       (the sequence pointed to by eax is 0x90,0x90,0x90,0x??). We have to calculate
       the address and we must return a Squeak SmallInteger (i.e. LSB set). To avoid
       expensive calculation during nextOp and endOp we allow a preceeding nop.
    */
    add eax,2
    or eax, 1
    /* store it in address */
    mov [ecx], eax
    /* and get out of here */
    ret
  }
}

/* goto_label needs to pass control to the instruction in ip.
   This method is only for testing, we actually use 
   goto_label_and_patch_sender in nextOp/endOp
*/
__declspec(naked) int __fastcall goto_label(int ip)
{
  __asm pop eax
  __asm jmp ecx

}

/* Patch the address pointed to by patchIP with patchValue.
   We have to keep care of any access protection here.
*/

void __fastcall patch_address_with(int patchIP, int patchValue)
{ MEMORY_BASIC_INFORMATION mbi;
  int oldProtect;

#ifndef NDEBUG
  warnPrintf("Patching IP at %x\n",patchIP);
#endif
  VirtualQuery((LPVOID)patchIP, &mbi, sizeof(mbi));
  VirtualProtect(mbi.BaseAddress, mbi.RegionSize, PAGE_READWRITE, &oldProtect);
  (*(short*) patchIP) = (short)patchValue;
  VirtualProtect(mbi.BaseAddress, mbi.RegionSize, oldProtect, &oldProtect);
}

/* The following function passes control to the instruction in ip.
   It also patches the "call goto_label_and_patch_sender" instruction
   to be an immediate "jmp ecx" which is equivalent (see goto_label
   and the note on __fastcall above)
*/
__declspec(naked) int __fastcall goto_label_and_patch_sender(int ip)
{
  __asm {
    pop eax     /* get the return address */
    sub eax, 5  /* subtract the size of the call goto_label_and_patch_sender instruction */
    push eax
    push ecx

    mov ecx, eax     /* first argument in ecx */
    mov edx, 0xe1ff  /* second argument (jmp ecx) in edx */
    call patch_address_with

    pop ecx
    pop eax
    /* pass control */
    jmp ecx
  }
}

#endif /* defined(JITTER) && defined(MSC_VER) && defined(_M_IX86) */


/****************************************************************************/
/*                      Console Window functions                            */
/****************************************************************************/
int OutputLogMessage(char *string)
{ FILE *fp;

  if(!*logName) return 1;
  fp = fopen(logName, "at");
  if(!fp) return 1;
  fprintf(fp, "%s", string);
  fflush(fp);
  fclose(fp);
  return 1;
}

int OutputConsoleString(char *string)
{ int pos;

  pos = SendMessage(consoleWindow,WM_GETTEXTLENGTH, 0,0);
  SendMessage(consoleWindow, EM_SETSEL, pos, pos);
  while(*string)
    {
      SendMessage( consoleWindow, WM_CHAR, *string, 1);
      string++;
    }
  /* something has been written in the console */
  if(fDynamicConsole)
    {
      ShowWindow(consoleWindow, SW_SHOW);
      fShowConsole = TRUE;
      CheckMenuItem(vmPrefsMenu, 0x0030, MF_BYCOMMAND | MF_CHECKED);
    }
}

int printf(const char *fmt, ...)
{ va_list al;

  va_start(al, fmt);
  wvsprintf(consoleBuffer, fmt, al);
  OutputLogMessage(consoleBuffer);
  if(IsWindow(stWindow)) /* not running as service? */
    OutputConsoleString(consoleBuffer);
  vfprintf(stdout, fmt, al);
  va_end(al);
}

int fprintf(FILE *fp, const char *fmt, ...)
{ va_list al;

  va_start(al, fmt);
  if(fp == stdout || fp == stderr)
    {
      wvsprintf(consoleBuffer, fmt, al);
      OutputLogMessage(consoleBuffer);
      if(IsWindow(stWindow)) /* not running as service? */
        OutputConsoleString(consoleBuffer);
    }
  vfprintf(fp, fmt, al);
  va_end(al);
}


int putchar(int c)
{
  return printf("%c",c);
}

/****************************************************************************/
/*                   Message Processing                                     */
/****************************************************************************/

static messageHook nextMessageHook = NULL;

int ServiceMessageHook(void * hwnd, unsigned int message, unsigned int wParam, long lParam)
{
  if(fRunService && fWindows95 && message == WM_BROADCAST_SERVICE && hwnd == stWindow)
    {
      /* broadcast notification - install the running Win95 service in the system tray */
      SetSystemTrayIcon(1);
      return 1;
    }
   if(message == WM_USERCHANGED)
      {
        SetSystemTrayIcon(1);
        return 1;
      }
  if(nextMessageHook)
    return(*nextMessageHook)(hwnd, message,wParam, lParam);
  else
    return 0;
}


/****************************************************************************/
/*                     Window Setup                                         */
/****************************************************************************/
/* SetSystemTrayIcon(): Set the icon in the system tray */
void SetSystemTrayIcon(BOOL on)
{ BOOL (WINAPI *ShellNotifyIcon)(DWORD,PNOTIFYICONDATA);
  static HMODULE hShell = NULL;
  NOTIFYICONDATA nid;

  /* NOTE: There is explicitly no unload of the shell32.dll in here.
           Win95 has _serious_ problems with the module counter and
           may just unload the shell32.dll even if it is referenced
           by other processes */
  if(!hShell) hShell = LoadLibrary(TEXT("shell32.dll"));
  if(!hShell) return; /* should not happen */
  /* On WinNT 3.* the following will just return NULL */
  (FARPROC)ShellNotifyIcon = GetProcAddress(hShell, "Shell_NotifyIconA");
  if(!ShellNotifyIcon) return;  /* ok, we don't have it */
  nid.cbSize = sizeof(nid);
  nid.hWnd   = stWindow;
  nid.uID    = (UINT)hInstance;
  nid.uFlags = NIF_MESSAGE | NIF_TIP | NIF_ICON;
  nid.uCallbackMessage = WM_USER+42;
  nid.hIcon  = LoadIcon(hInstance, MAKEINTRESOURCE(1));
  strcpy(nid.szTip, "Squeak!");
  if(on)
    (*ShellNotifyIcon)(NIM_ADD, &nid);
  else
    (*ShellNotifyIcon)(NIM_DELETE, &nid);
}

void SetupService95()
{
#ifndef NO_SERVICE
#ifndef RSP_SIMPLE_SERVICE
#define RSP_SIMPLE_SERVICE 1
#endif

  BOOL (WINAPI *RegisterServiceProcess)(DWORD,DWORD);
  HMODULE hKernel32 = NULL;
 
  /* Inform Windows95 that we're running as a service process */
  if(!fRunService || !fWindows95) return;
  hKernel32 = LoadLibrary(TEXT("kernel32.dll"));
  if(!hKernel32)
    {
      printLastError(TEXT("Unable to load kernel32.dll"));
      return;
    }
  (FARPROC) RegisterServiceProcess = GetProcAddress(hKernel32, "RegisterServiceProcess");
  if(!RegisterServiceProcess)
    {
      printLastError(TEXT("Unable to find RegisterServiceProcess"));
      return;
    }
  if( !(*RegisterServiceProcess)(GetCurrentProcessId(), RSP_SIMPLE_SERVICE ) )
    printLastError(TEXT("RegisterServiceProcess failed"));
#endif /* NO_SERVICE */
}

/****************************************************************************/
/*              Display and printing                                        */
/****************************************************************************/
#ifdef NO_STD_BYTE_REVERSAL
int reverse_image_bytes(int *ptr,int d, int w,int h,int x0,int y0,int x1,int y1)
{ int scanLine, pixPerWord;
  int startX, stopX, startY, stopY;
  int *linePtr, *pixPtr;
  int log2Depth,pixPerWordShift;
  int j,xLen;
  /* note: this function should only be called for a d<=8 */
  switch (d) {
  case 8: log2Depth=3; break;
  case 4: log2Depth=2; break;
  case 2: log2Depth=1; break;
  case 1: log2Depth=0; break;
  default: return 0;  /* don't do anything distructive if the person calling us is confused */
  }
  pixPerWordShift = 5-log2Depth;
  pixPerWord= 1 << pixPerWordShift;  /* was = 32/depth */
  scanLine= (w+pixPerWord-1) >> pixPerWordShift;  /* words per scan line */
  startX = (x0 >> pixPerWordShift);
  stopX  = (x1 + pixPerWord -1) >> pixPerWordShift;
  xLen = stopX - startX;
  startY = y0 * scanLine;
  stopY  = y1 * scanLine;
  if(stopX <= startX || stopY <= startY) return 1;
  for(j = startY; j < stopY; j += scanLine)
    {
      linePtr = ptr + j;
      pixPtr = linePtr + startX;
  /* Contributed by Michael Rutenberg:
  Speed-up x86 code: Deal with a word (4 or more pixels) per iteration.
  The speed-up code runs best on a Pentium and later and requires a 486 for the
  one bswap instruction.  It tries to issue two instructions per cycle, one into
  the U pipe and one into the V (free!) pipe as annotated.  I looked at unrolling
  the loops a bit, to do two or more words per iteration, but it just makes things
  more complicated and is only minimally faster.  The big win (on a Pentium Pro)
  would be prefetching the required data into the cache well before we need it. On
  the Pentium it is hard to do this without stalling and so loosing the benefit. -mdr */
      __asm {
        mov edx, pixPtr
        mov ecx, xLen
      nextGeneral:
        mov eax,[edx]	;u1
        bswap eax	;u2??
        mov [edx],eax	;u1
        add edx,4	;v0
        dec ecx		;u1
        jnz nextGeneral	;v0
      }
    }
  return 1;
}

int
reverse_image_words(int *ptr,int d, int w,int h,int x0,int y0,int x1,int y1)
{ int scanLine, pixPerWord;
  int startX, stopX, startY, stopY;
  int *linePtr,*pixPtr;
  int j,xLen;
  const int log2Depth=4;    /* log2Depth = log2(d) = log2(16) */
  const int pixPerWordShift = 5-log2Depth;

  if (d!=16)
	return 0;  /* The person calling us is confused.  We only deal with 16 bit words */

  pixPerWord= 1 << pixPerWordShift;  /* was = 32/depth */
  scanLine= (w+(pixPerWord-1)) >> pixPerWordShift;  /* words per scan line */
  startX = (x0 >> pixPerWordShift);
  stopX  = (x1 >> pixPerWordShift) + 1;
  xLen = stopX - startX;
  startY = y0 * scanLine;
  stopY  = y1 * scanLine;
  if(stopX <= startX || stopY <= startY) return 1;
  for(j = startY; j < stopY; j += scanLine)
    {
      linePtr = ptr + j;
      pixPtr = linePtr + startX;
      /* Speed-up x86 code: Deal with a word (two 16 bit pixels) per iteration.  -mdr */
      /* When the data is in the cache, this does once word in 4 cycles */
      _asm {
        mov edx, pixPtr
        mov ecx, xLen
      next2pixels:
        mov eax,[edx]	;u1   32 bit loads/stores are fastest in 32 bit mode
        rol eax,16	;u1
        mov [edx],eax	;u1
        add edx,4	;v0
        dec ecx		;u1    the dec/jump pair is faster than loop
        jnz next2pixels	;v0
      }
    }
  return 1;
}
#endif /* NO_STD_BYTE_REVERSAL */

/****************************************************************************/
/*                      System Attributes                                   */
/****************************************************************************/

char *GetVMOption(int id)
{
  if(id < numOptionsVM)
    return vmOptions[id];
  else
    return "";
}

char *GetImageOption(int id)
{
  if(id < numOptionsImage)
    return imageOptions[id];
  else
    return "";
}


/****************************************************************************/
/*                      Error handling                                      */
/****************************************************************************/
void SetupStderr()
{ TCHAR tmpName[MAX_PATH+1];

  *stderrName = *stdoutName = 0;
  /* re-open stdout && stderr */
  GetTempPath(MAX_PATH,tmpName);
  if(GetStdHandle(STD_ERROR_HANDLE) == INVALID_HANDLE_VALUE)
    {
      GetTempFileName(tmpName,TEXT("sq"),0,stderrName);
      freopen(stderrName,"w+t",stderr);
    }
  else *stderrName = 0;

  if(GetStdHandle(STD_OUTPUT_HANDLE) == INVALID_HANDLE_VALUE)
    {
      GetTempFileName(tmpName,TEXT("sq"),0,stdoutName);
      freopen(stdoutName,"w+t",stdout);
    }
  else *stdoutName = 0;
}

/****************************************************************************/
/*                      Release                                             */
/****************************************************************************/
#ifndef NO_VIRTUAL_MEMORY
void printCrashDebugInformation()
{ TCHAR crashInfo[1024];
  FILE *f;
  int byteCode;

  /* Retrieve current byte code.
     If this fails the IP is probably wrong */
  __try {
#ifndef JITTER
    byteCode = getCurrentBytecode();
#else
    byteCode = -1;
#endif
  } __except(EXCEPTION_EXECUTE_HANDLER) {
    byteCode = -1;
  }

  __try {
  wsprintf(crashInfo,TEXT("Sorry but the VM has crashed.\n\n")
                     TEXT("Current byte code: %d\n")
                     TEXT("Primitive index: %d\n\n")
                     TEXT("This information has also been stored in the file\n")
                     TEXT("%s%s\n")
                     TEXT("with a complete stack dump"),
                     byteCode,
                     primitiveIndex,
                     vmPath,
                     TEXT("crash.dmp"));
  SetCurrentDirectory(vmPath);
  /* print the above information */
  f = fopen("crash.dmp","a");
  if(f)
    {  time_t crashTime = time(NULL);
       fprintf(f,"---------------------------------------------------------------------\n"
                 "%s\n"
                 "Current byte code: %d\n"
                 "Primitive index: %d\n"
                 "Stack dump follows:\n\n",
                 ctime(&crashTime),
                 byteCode,
                 primitiveIndex);
    }
  fclose(f);

  /* print the caller's stack twice (to stdout and "crash.dmp")*/
  printCallStack();
  freopen("crash.dmp","at",stdout);
  printCallStack();
  if(*stdoutName)
    freopen(stdoutName,"a+t",stdout);

  if(!fHeadlessImage)
    MessageBox(0,crashInfo,TEXT("Squeak fatal error"),
                 MB_OK | MB_APPLMODAL | MB_ICONSTOP);

  } __except(EXCEPTION_EXECUTE_HANDLER) {
    /* that's to bad ... */
    if(!fHeadlessImage)
      MessageBox(0,TEXT("Squeak has crashed. Sorry."),TEXT("Squeak fatal error:"),
                 MB_OK | MB_APPLMODAL | MB_ICONSTOP);
    else
      abortMessage(TEXT("Squeak has crashed. Sorry."));
  }
}
#endif /* NO_VIRTUAL_MEMORY */

void printErrors()
{ TCHAR *errorMsg;
  fpos_t stdoutSize,stderrSize;

  if(*stdoutName)
    {
      fgetpos(stdout,&stdoutSize);
      fseek(stdout,0,SEEK_SET);
    }
  else stdoutSize = 0;

  if(*stderrName)
    {
      fgetpos(stderr,&stderrSize);
      fseek(stderr,0,SEEK_SET);
    }
  else stderrSize = 0;

  if(stdoutSize <= 0 && stderrSize <= 0) return;
  errorMsg = (char*) calloc(stdoutSize+stderrSize+2,1);
  fread(errorMsg,stdoutSize,1,stdout);
  errorMsg[stdoutSize] = '\n';
  fread(&errorMsg[stdoutSize+1],stderrSize,1,stderr);
  if(!fHeadlessImage)
    MessageBox(0,errorMsg,TEXT("Squeak Error:"),MB_OK);
  free(errorMsg);
}

void Cleanup(void)
{ /* not all of these are essential, but they're polite... */

  ReleaseTimer();
  /* tricky ... we have no systray icon when running
     headfull or when running as service on NT */
  if(fHeadlessImage && (!fRunService || fWindows95))
    SetSystemTrayIcon(0);
  if(palette) DeleteObject(palette);
  PROFILE_SHOW(ticksForReversal);
  PROFILE_SHOW(ticksForBlitting);
  printErrors();
  if(*stderrName)
    {
      fclose(stderr);
      remove(stderrName);
    }
  if(*stdoutName)
    {
      fclose(stdout);
      remove(stdoutName);
    }
#ifndef NO_VIRTUAL_MEMORY
  sqReleaseMemory();
#endif
}


/****************************************************************************/
/*                        sqMain                                            */
/****************************************************************************/
static vmArg args[] = {
  { ARG_STRING, &installServiceName, "-service:" }, /* the name of a service */
  { ARG_FLAG, &fHeadlessImage, "-headless" },       /* do we run headless? */
  { ARG_STRING, &logName, "-log:" },                /* VM log file */
  { ARG_UINT, &dwMemorySize, "-memory:" },          /* megabyte of memory to use */

  /* NOTE: the following two flags are "undocumented" for service support on 95 */
  
  { ARG_FLAG, &fRunService, "-service95" },           /* do we start as service? */
  { ARG_FLAG, &fBroadcastService95, "-broadcast95" }, /* should we notify services of a user logon? */
  { ARG_NONE, NULL, NULL }
};

/* sqMain: This is common entry point regardless of whether we're running
           as a normal app or as a service. Note that a number of things
           may have been set up before coming here. In particular,
             * fRunService - to determine whether we're running as NT service
           However, the command line must always contain all parameters necessary.
           In other words, even though the logName may have been set before,
           the command line has to include the -log: switch.
*/
int sqMain(char *lpCmdLine, int nCmdShow)
{ int virtualMemory;
  sqImageFile imageFile;
  int imageSize;

  /* parse command line args */
  if(!parseArguments(strdup(GetCommandLine()), args))
    return printUsage(1);

  /* a quick check if we have any argument at all */
  if(*imageName == 0 && *lpCmdLine == 0) return printUsage(2);

#ifdef NO_SERVICE
  fRunService = 0;
#endif

  /* look for a few things easy to handle */
  if(fWindows95 && fBroadcastService95)
    {
      PostMessage(HWND_BROADCAST, WM_BROADCAST_SERVICE, 0, 0);
      return 0;
    }

  /* set time zone accordingly */
  _tzset();

  /* Give us some log information when running as service */
  if(fRunService)
    { time_t svcStart;

      svcStart = time(NULL);
      OutputLogMessage("\n\n");
      OutputLogMessage(ctime(&svcStart));
      if(fWindows95) /* don't have a service name */
        OutputLogMessage("The service");
      else
        OutputLogMessage(serviceName);
      OutputLogMessage(" started with the following command line\n");
      OutputLogMessage(initialCmdLine);
      OutputLogMessage("\n");
    }

  SetupFilesAndPath();

  /* release resources on exit */
  atexit(Cleanup);

#ifndef NO_SERVICE
  /* if service installing is requested, do so */
  if(installServiceName && *installServiceName) 
    {
      strcpy(serviceName, installServiceName);
      sqServiceInstall();
      /* When installing was successful we won't come
         to this point. Otherwise ... */
      exit(-1); /* this will show any printfs during install */
    }
#endif

  /* initialisation */
  SetupKeymap();
  SetupWindows();
  SetupPixmaps();
  SetupPrinter();
  SetupService95();
  SetupTimer();
  SetupMIDI();
  sqFileInit();
  joystickInit();

  /* check the interpreter's size assumptions for basic data types */
  if (sizeof(int) != 4)    error("This C compiler's integers are not 32 bits.");
  if (sizeof(double) != 8) error("This C compiler's floats are not 64 bits.");
  if (sizeof(time_t) != 4) error("This C compiler's time_t's are not 32 bits.");


  imageSize = SqueakImageLength(toUnicode(imageName));
  if(imageSize == 0) printUsage(2);

  /* allocate the synchronization mutex before anything is going to happen */
  vmSemaphoreMutex = CreateMutex(NULL, 0, NULL);

#ifdef NO_VIRTUAL_MEMORY
  if(!dwMemorySize) dwMemorySize = 4;
  virtualMemory = (int)imageSize + max(imageSize, dwMemorySize * 0x00100000);
#else

  if(!dwMemorySize) dwMemorySize = MAX_VIRTUAL_MEMORY;
  virtualMemory = max((int)dwMemorySize * 0x00100000, (int) 2*imageSize);
  /* The exception handler. 
     Everything happening in here will be caught by the handler at the end */
  __try { 

#endif

  /* read the image file */
  imageFile = sqImageFileOpen(imageName,"rb");
  readImageFromFileHeapSize(imageFile, virtualMemory);
  sqImageFileClose(imageFile);

  /* display the main window */
  SetWindowSize();
  if(!fHeadlessImage) 
    ShowWindow(stWindow,nCmdShow);
  /* if headless running is requested, try to to create an icon
     in the Win95/NT system tray */
  else if(!fRunService || fWindows95)
    SetSystemTrayIcon(1);

  ioSetFullScreen(fullScreenFlag);

  SetupPreferences();
  /* run Squeak */
  interpret();

#ifndef NO_VIRTUAL_MEMORY
  } __except(sqExceptionFilter(GetExceptionInformation())) {
    /* if we get this far our app has finally crashed */
    printCrashDebugInformation();
  }
#endif
}

/****************************************************************************/
/*                        WinMain                                           */
/****************************************************************************/
int PASCAL WinMain (HINSTANCE hInst,
                    HINSTANCE hPrevInstance,
                    LPSTR  lpCmdLine,
                    int    nCmdShow)
{
  
  /* a few things which need to be done first */

  /* check if we're running NT or 95 */
  fWindows95 = (GetVersion() & 0x80000000) != 0;

  /* fetch us a copy of the command line */
  initialCmdLine = strdup(lpCmdLine);

  /* fetch us the name of the executable */
  GetModuleFileName(hInst, vmName, MAX_PATH);

  /* open all streams in binary mode */
  _fmode  = _O_BINARY;

  /* get us the instance handle */
  hInstance = hInst;


#ifndef NO_SERVICE
  /* Find out if we're running from a service.
     That's a bit tricky since there is no difference between
     usual startup and service startup. We do two assumptions here:
      1) If we're NOT running on NT we can't be service
      2) If there is a command line we can't be service
     Still, there is a chance that a user just double clicks on
     the Squeak executable in NT. Therefore we _try_ to connect
     to the service control manager in the sqServiceMain function.
     If this fails, we try the usual startup. It might take a bit
     longer than the normal startup but this only happens if there
     is no image name given - and that's not our fault. Anyways,
     if somebody out there knows how to find out when we're starting
     as a service - LET ME KNOW!
  */
  if(!fWindows95)                      /* 1) running NT */
    if(!*lpCmdLine)                    /* 2) No command line */
      if(sqServiceMain())              /* try starting the service */
         return 0;                     /* service was run - exit */

#endif

  /* Special startup stuff for windows 95 */
  if(fWindows95)
    {
      /* The message we use for notifying services of user logon */
      WM_BROADCAST_SERVICE = RegisterWindowMessage(msgBroadcastService);
    }

  /* start the non-service version */
  sqMain(lpCmdLine, nCmdShow);
  return 0;
}
