TextDocs.NewDoc     .R   CWindowsLeft *   WindowsRight S  WindowsTop !   WindowsButtom   Color    Flat  Locked  Controls  Org s\   BIER           3     Oberon10.Scn.Fnt     Syntax10.Scn.Fnt  b                     Syntax12.Scn.Fnt             
   )    5       7            8       \  (* ETH Oberon, Copyright 2001 ETH Zuerich Institut fuer Computersysteme, ETH Zentrum, CH-8092 Zuerich.
Refer to the "General ETH Oberon System Source License" contract available at: http://www.oberon.ethz.ch/ *)

MODULE Viewers; (** portable, except where noted / source: Win32.Viewers.Mod *)	(* JG 4.8.93, ejz   *)

(**
The module viewers provide the data type for implementing the tiled viewers of the Oberon system. Each track of the Oberon system consists of a number of viewers.
*)

	IMPORT SYSTEM, Kernel32, Objects, User32, Displays, Display, Threads;

	CONST
	(** the different types (kind) of viewers: *)
		IsDisplay* = 0; IsTrack* = 1; IsFiller* = 2; IsViewer* = 3; IsDocument* = 4; IsControl* = 5; (** non-portable *)

	TYPE
		Viewer* = POINTER TO ViewerDesc;

		Window* = POINTER TO RECORD (Displays.Display)	(** non-portable *)
			viewer*: Viewer;
			track*: Displays.InputEvent
		END;

		ViewerDesc* = RECORD (Display.FrameDesc)
			state*: INTEGER;	(** state > 1: displayed, state = 1: filler, state = 0: closed, state < 0: suspended.*)
			kind*: INTEGER;	(** IsTrack, IsFiller, IsViewer, IsDocument, IsControl *) (** non-portable *)
			win*: Displays.Display	(** non-portable *)
		END;

		Track = POINTER TO TrackDesc;
		TrackDesc = RECORD (ViewerDesc)
			under: Display.Frame
		END;

	VAR
		curW*: INTEGER;	(** Current width of the logical display. *)
		minH*: INTEGER;	(** Minimum viewer height. *)
		FillerTrack, left, right: Track;
		FillerViewer, recall*: Viewer;	(** non-portable *)	(* for closed viewers *)
		bcF: Display.Frame; bcLev: LONGINT;
		moduleCS*: Threads.Mutex;	(** non-portable *)
		marked*: Viewer;	(** the marked viewer *) (** non-portable *)
		tracksWin*: Displays.Display;	(** non-portable *)
		sorted*: BOOLEAN;	(** FALSE: the list of viewers must be sorted (in z-order) by the next broacast. *) (** non-portable *)

	PROCEDURE ToWindow(V: Viewer; VAR M: Objects.ObjMsg);
	BEGIN
		V.win.handle(V.win, M)	
	END ToWindow;

	PROCEDURE SetUpdate(V: Viewer; update: BOOLEAN);
		VAR A: Objects.AttrMsg;
	BEGIN
		IF update THEN V.state := 2 END;
		A.name := "Update"; A.class := Objects.Bool; A.b := update;
		A.id := Objects.set; A.res := -1; ToWindow(V, A)
	END SetUpdate;

	(** Open a new viewer V with top at Y in track X. *)
	PROCEDURE Open*(V: Viewer; X, Y: INTEGER);
		VAR T, u, v: Display.Frame; M: Display.ControlMsg; N: Display.ModifyMsg; cur: Displays.Display;
	BEGIN
		ASSERT(tracksWin # NIL);
		V.kind := IsViewer; V.win := tracksWin;
		cur := Display.cur; Display.SetCurrent(V.win);
		IF (V.state = 0) & (X > left.X) & (X < right.X) THEN
			IF Y > Display.Height THEN Y := Display.Height END;
			T := left.next;
			WHILE (X >= T.X + T.W) & (T.next # right) DO T := T.next END;
			u := T.dsc; v := u.next;
			WHILE Y > v.Y + v.H DO u := v; v := u.next END;
			IF Y < v.Y + minH THEN Y := v.Y + minH END;
			IF (v.next.Y # 0) & (Y > v.Y + v.H - minH) THEN
				WITH v: Viewer DO
					V.X := T.X; V.W := T.W; V.Y := v.Y; V.H := v.H;
					M.F := NIL; M.id := Display.suspend;
					v.handle(v, M); v.state := 0; recall := v;
					V.next := v.next; u.next := V
				END
			ELSE
				V.X := T.X; V.W := T.W; V.Y := v.Y; V.H := Y - v.Y;
				N.F := v; N.id := Display.reduce;
				N.Y := Y; N.H := v.Y + v.H - Y;
				v.handle(v, N); v.Y := N.Y; v.H := N.H;
				V.next := v; u.next := V
			END;
			SetUpdate(V, TRUE)
		END;
		Display.SetCurrent(cur)
	END Open;

	(** Expand or shrink a viewer vertically to new top Y. *)
	PROCEDURE Change*(V: Viewer; Y: INTEGER);
		VAR v: Display.Frame; M: Display.ModifyMsg;
	BEGIN
		ASSERT(V.kind = IsViewer);
		IF V.state > 1 THEN
			IF Y > Display.Height THEN Y := Display.Height END;
			v := V.next;
			IF (v.next.Y # 0) & (Y > v.Y + v.H - minH) THEN
				Y := v.Y + v.H - minH
			END;
			IF Y > v.Y THEN
				M.F := v; M.id := Display.reduce;
				M.Y := Y; M.H := v.Y + v.H - Y;
				v.handle(v, M); v.Y := M.Y; v.H := M.H;
				V.H := Y - V.Y
			ELSIF Y >= V.Y + minH THEN
				M.F := v; M.id := Display.extend;
				M.Y := Y; M.H := v.Y + v.H - Y;
				v.handle(v, M); v.Y := M.Y; v.H := M.H;
				V.H := Y - V.Y
			END
		END
	END Change;

	PROCEDURE RestoreTrack(S: Display.Frame);
		VAR T, t, v: Display.Frame; M: Display.ControlMsg;
	BEGIN
		WITH S: Track DO
			t := S.next;
			WHILE t.next.X # S.X DO t := t.next END;
			T := S.under;
			WHILE T.next # NIL DO T := T.next END;
			t.next := S.under; T.next := S.next;
			M.F := NIL; M.id := Display.restore;
			REPEAT t := t.next;
				v := t.dsc;
				REPEAT
					v := v.next; v.handle(v, M);
					WITH v: Viewer DO v.state := - v.state END
				UNTIL v = t.dsc
			UNTIL t = T
		END
	END RestoreTrack;

	PROCEDURE [WINAPI] EnumSortWindow(hwnd: User32.HWND; lParam: User32.LParam): Kernel32.BOOL;
		VAR p, v, dsc, last: Display.Frame; i: LONGINT;
	BEGIN (* i: 0 left, 1 right, 2 tracks *)
		i := 0;
		WHILE i < 3 DO
			IF i < 2 THEN
				IF i = 0 THEN dsc := left.dsc ELSE dsc := right.dsc END;
				p := dsc; v := dsc.next;
				WHILE v # dsc DO
					IF v(Viewer).win.hWndParent = hwnd THEN
						p.next := v.next;
						SYSTEM.GET(lParam, last);
						last.next := v;
						SYSTEM.PUT(lParam, v)
					ELSE
						p := v
					END;
					v := v.next
				END
			ELSIF (tracksWin # NIL) & (tracksWin.hWndParent = hwnd) THEN
				SYSTEM.GET(lParam, recall)
			END;
			INC(i)
		END;
		RETURN Kernel32.True
	END EnumSortWindow;

	PROCEDURE ZSort();
		VAR F, bak: Viewer; last, v: Display.Frame;
	BEGIN
		sorted := TRUE; bak := recall;
		NEW(F); F.next := NIL; last := F; recall := F;
		User32.EnumWindows(EnumSortWindow, SYSTEM.ADR(last));
v := left.dsc.next;
WHILE v # left.dsc DO
	last.next := v; last := v;
	left.dsc.next := v.next;
	v := left.dsc.next
END;
v := right.dsc.next;
WHILE v # right.dsc DO
	last.next := v; last := v;
	right.dsc.next := v.next;
	v := right.dsc.next
END;
		IF recall # F THEN (* left: F.next .. recall, tracks, right: recall.next .. last *)
			IF last # recall THEN
				right.dsc.next := recall.next; last.next := right.dsc
			END;
			left.dsc.next := F.next; recall.next := left.dsc
		ELSIF last # F THEN (* left: empty, tracks, right: F.next .. last *)
			right.dsc.next := F.next; last.next := right.dsc
		END;
		recall := bak
	END ZSort;

	(** Local broadcast within viewer V. *)
	PROCEDURE Send*(V: Viewer; VAR M: Display.FrameMsg);	(** non-portable *)
		VAR t, v: Display.Frame; cur: Displays.Display;
	BEGIN
		IF V = NIL THEN RETURN END;
		Threads.Lock(moduleCS);
		IF ~sorted THEN ZSort() END;
		cur := Display.cur;
		M.res := MIN(INTEGER); Objects.Stamp(M);
		IF V.kind IN {IsViewer, IsDocument, IsControl} THEN
			M.dlink := NIL; M.x := 0; M.y := 0;
			IF bcLev = 0 THEN bcF := V END;	(* remember outermost frame *)
			INC(bcLev); Display.SetCurrent(V.win);
			V.handle(V, M); DEC(bcLev)
		ELSE
			t := left.next;
			WHILE (t # right) & (M.res < 0) DO v := t.dsc;
				REPEAT v := v.next; M.dlink := NIL; M.x := 0; M.y := 0;
					IF bcLev = 0 THEN bcF := v END;	(* remember outermost frame *)
					INC(bcLev); Display.SetCurrent(v(Viewer).win);
					v.handle(v, M); DEC(bcLev)
				UNTIL (v = t.dsc) OR (M.res >= 0);
				t := t.next
			END
		END;
		IF bcLev = 0 THEN bcF := NIL END;
		Display.SetCurrent(cur);
		Threads.Unlock(moduleCS)
	END Send;

	(** Remove viewer V from the display. *)
	PROCEDURE Close*(V: Viewer);
		VAR T, U: Display.Frame; M: Display.ControlMsg; N: Display.ModifyMsg; safe: BOOLEAN;
	BEGIN
		IF V = NIL THEN	(* called from System.Trap *)
			IF (bcF = NIL) OR ~(bcF IS Viewer) THEN	(* can not do anything *)
				bcF := NIL; bcLev := 0; RETURN
			END;
			V := bcF(Viewer);	(* close the outermost viewer *)
			IF ~(V.kind IN {IsViewer, IsDocument}) THEN RETURN END;
			safe := FALSE; bcF := NIL; bcLev := 0
		ELSE
			safe := TRUE	(* normal case *)
		END;
		IF marked = V THEN marked := NIL END;
		IF V.state > 1 THEN
			ASSERT(V.kind IN {IsViewer, IsDocument, IsControl});
			IF V.kind = IsViewer THEN
				U := V.next; T := left;
				REPEAT T := T.next UNTIL V.X < T.X + T.W;
				IF (T(Track).under = NIL) OR (U.next # V) THEN
					IF safe THEN
						M.F := NIL; M.id := Display.suspend;
						V.handle(V, M)
					END;
					V.state := 0; recall := V;
					N.F := U; N.id := Display.extend;
					N.Y := V.Y; N.H := V.H + U.H;
					U.handle(U, N); U.Y := N.Y; U.H := N.H;
					WHILE U.next # V DO U := U.next END;
					U.next := V.next
				ELSE (*close track*)
					M.F := NIL; M.id := Display.suspend;
					IF safe THEN V.handle(V, M) END;
					V.state := 0; recall := V;
					U.handle(U, M); U(Viewer).state := 0;
					RestoreTrack(T)
				END
			ELSE
				SetUpdate(V, FALSE);
				IF safe THEN M.F := NIL; M.id := Display.suspend; Send(V, M) END;
				T := left.dsc; U := T.next;
				WHILE (U # V) & (U # left.dsc) DO
					T := U; U := U.next
				END;
				IF U # V THEN
					T := right.dsc; U := T.next;
					WHILE (U # V) & (U # right.dsc) DO
						T := U; U := U.next
					END
				END;
				ASSERT(U = V);
				T.next := U.next;
				V.state := 0; recall := V;
				M.F := NIL; M.id := Display.remove;
				ToWindow(V, M)
			END;
			Objects.Stamp(M); recall.stamp := M.stamp
		ELSIF (V.kind = IsDisplay) & (V.win = tracksWin) THEN
			M.F := NIL; M.id := Display.remove;
			ToWindow(V, M); tracksWin := NIL;
			left.next := right; curW := 0
		END
	END Close;

	(** Recall most recently closed viewer. *)
	PROCEDURE Recall*(VAR V: Viewer);
	BEGIN
		V := recall
	END Recall;

	(** Return viewer located at display coordinates X, Y. *)
	PROCEDURE This*(X, Y: INTEGER): Viewer;
		VAR T, V: Display.Frame;
	BEGIN
		IF (X > left.X) & (X < right.X) & (Y < Display.Height) THEN
			T := left;
			REPEAT T := T.next UNTIL X < T.X + T.W;
			IF T = right THEN RETURN NIL END;
			V := T.dsc;
			REPEAT V := V.next UNTIL Y < V.Y + V.H;
			RETURN V(Viewer)
		ELSE
			RETURN NIL
		END
	END This;

	(** Return next upper neighbour of V in a track. *)
	PROCEDURE Next*(V: Viewer): Viewer;
	BEGIN
		RETURN V.next(Viewer)
	END Next;

	(** In the track at X locate the following viewers: filler fil, bottom-most viewer, an 
	alternative viewer alt of height >= H, and the viewer with the maximum height. *)
	PROCEDURE Locate*(X, H: INTEGER; VAR fil, bot, alt, max: Display.Frame);
		VAR T, V: Display.Frame;
	BEGIN
		IF (X > left.X) & (X < right.X) THEN
			T := left;
			REPEAT T := T.next UNTIL X < T.X + T.W;
			fil := T.dsc; bot := fil.next;
			IF bot.next # fil THEN
				alt := bot.next; V := alt.next;
				WHILE (V # fil) & (alt.H < H) DO
					IF V.H > alt.H THEN alt := V END; V := V.next
				END
			ELSE alt := bot
			END;
			max := T.dsc; V := max.next;
			WHILE V # fil DO
				IF V.H > max.H THEN max := V END; V := V.next
			END
		END
	END Locate;

	(** Append to the current logical display and init track of width W and height H, and install filler. *)
	PROCEDURE InitTrack*(W, H: INTEGER; Filler: Viewer);
		VAR S: Display.Frame; T: Track;
	BEGIN
		IF Filler.state = 0 THEN
			Filler.X := curW; Filler.W := W; Filler.Y := 0; Filler.H := H;
			Filler.state := 1; Filler.next := Filler;
			Filler.kind := IsFiller; Filler.win := tracksWin;
			NEW(T); T.kind := IsTrack; T.win := tracksWin;
			T.X := curW; T.W := W; T.Y := 0; T.H := H;
			T.dsc := Filler; T.under := NIL;
			FillerViewer.X := curW + W; FillerViewer.W := MAX(INTEGER) - FillerViewer.X;
			FillerTrack.X := FillerViewer.X; FillerTrack.W := FillerViewer.W;
			S := left;
			WHILE S.next # right DO S := S.next END;
			S.next := T; T.next := right;
			curW := curW + W
		END
	END InitTrack;

	(** Open new track overlaying span of [X, X +W[. *)
	PROCEDURE OpenTrack*(X, W: INTEGER; Filler: Viewer);
		VAR newT: Track; S, T, t, v: Display.Frame; M: Display.ControlMsg;
	BEGIN
		IF (X > left.X) & (X < right.X) & (Filler.state = 0) THEN
			S := left; T := S.next;
			WHILE X >= T.X + T.W DO S := T; T := S.next END;
			WHILE X + W > T.X + T.W DO T := T.next END;
			M.F := NIL; M.id := Display.suspend;
			t := S;
			REPEAT t := t.next; v := t.dsc;
				REPEAT v := v.next;
					WITH v: Viewer DO v.state := -v.state; v.handle(v, M) END
				UNTIL v = t.dsc
			UNTIL t = T;
			Filler.X := S.next.X; Filler.W := T.X + T.W - S.next.X; Filler.Y := 0; Filler.H := Display.Height;
			Filler.state := 1; Filler.next := Filler;
			Filler.kind := IsFiller; Filler.win := tracksWin;
			NEW(newT); newT.kind := IsTrack; newT.win := tracksWin;
			newT.X := Filler.X; newT.W := Filler.W; newT.Y := 0; newT.H := Display.Height;
			newT.dsc := Filler; newT.under := S.next; S.next := newT;
			newT.next := T.next; T.next := NIL
		END
	END OpenTrack;

	(** Close track at X and restore overlaid tracks. *)
	PROCEDURE CloseTrack*(X: INTEGER);
		VAR T, V: Display.Frame; M: Display.ControlMsg;
	BEGIN
		IF (X > left.X) & (X < right.X) THEN
			T := left;
			REPEAT T := T.next UNTIL X < T.X + T.W;
			IF T(Track).under # NIL THEN
				M.F := NIL; M.id := Display.suspend; V := T.dsc;
				REPEAT V := V.next; V.handle(V, M); V(Viewer).state := 0 UNTIL V = T.dsc;
				RestoreTrack(T)
			END
		END
	END CloseTrack;

	PROCEDURE *Broadcast(VAR M: Display.FrameMsg);
		VAR F, V: Display.Frame; cur: Displays.Display;
	BEGIN
		Threads.Lock(moduleCS);
		IF ~sorted THEN ZSort() END;
		cur := Display.cur;
		M.res := MIN(INTEGER); Objects.Stamp(M);
		F := FillerTrack.next;
		WHILE (F # FillerTrack) & (M.res < 0) DO V := F.dsc;
			REPEAT V := V.next; M.dlink := NIL; M.x := 0; M.y := 0;
				IF bcLev = 0 THEN bcF := V END;	(* remember outermost frame *)
				INC(bcLev); Display.SetCurrent(V(Viewer).win);
				V.handle(V, M); DEC(bcLev)
			UNTIL (V = F.dsc) OR (M.res >= 0);
			F := F.next
		END;
		IF bcLev = 0 THEN bcF := NIL END;
		Display.SetCurrent(cur);
		Threads.Unlock(moduleCS)
	END Broadcast;

	PROCEDURE ThisPoint*(point: User32.Point): Viewer;	(** non-portable *)
		VAR dsc, V: Display.Frame; i: LONGINT; rect: User32.Rect; X, Y: INTEGER;
	BEGIN
		Threads.Lock(moduleCS);
		IF ~sorted THEN ZSort() END;
		i := 0;
		WHILE i < 3 DO
			IF (i = 1) & (tracksWin # NIL) THEN
				User32.GetWindowRect(tracksWin.hWnd, rect);
				IF (point.x >= rect.left) & (point.x <= rect.right) & (point.y >= rect.top) & (point.y <= rect.bottom) THEN
					(* X := SHORT(point.x-rect.left); Y := SHORT(Display.Height-(point.y-rect.top)); *)
					User32.ScreenToClient(tracksWin.hWnd, point);
					X := SHORT(point.x); Y := SHORT(Display.Height-point.y);
					Threads.Unlock(moduleCS); RETURN This(X, Y)
				END
			ELSIF i # 1 THEN
				IF i = 0 THEN dsc := left.dsc ELSE dsc := right.dsc END;
				V := dsc.next;
				WHILE V # dsc DO
					WITH V: Viewer DO
						IF V.win # NIL THEN
							User32.GetWindowRect(V.win.hWnd, rect);
							IF (point.x >= rect.left) & (point.x <= rect.right) & (point.y >= rect.top) & (point.y <= rect.bottom) THEN
								Threads.Unlock(moduleCS); RETURN V
							END
						END
					END;
					V := V.next
				END
			END;
			INC(i)
		END;
		Threads.Unlock(moduleCS);
		RETURN NIL
	END ThisPoint;

	PROCEDURE *handle(obj: Objects.Object; VAR M: Objects.ObjMsg);
	BEGIN
	END handle;

	PROCEDURE DocumentModify(V: Viewer; menu, main: Display.Frame; mode: INTEGER);
		VAR M: Display.ModifyMsg;
	BEGIN
		V.X := 0; V.Y := 0; V.W := SHORT(V.win.width); V.H := SHORT(V.win.height);

		M.F := menu; M.dlink := V; M.id := Display.extend; M.mode := mode;
		M.x := 0; M.y := 0; M.res := MIN(INTEGER); Objects.Stamp(M);
		M.X := 0; M.dX := -menu.X;
		M.Y := V.H-menu.H; M.dY := M.Y-menu.Y;
		M.W := V.W; M.dW := M.W-menu.W;
		M.H := menu.H; M.dH := 0;
		menu.handle(menu, M);

		M.F := main; M.dlink := V; M.id := Display.extend; M.mode := mode;
		M.x := 0; M.y := 0; M.res := MIN(INTEGER); Objects.Stamp(M);
		M.X := 0; M.dX := -main.X;
		M.Y := 0; M.dY := -main.Y;
		M.W := V.W; M.dW := M.W-main.W;
		M.H := V.H-menu.H; M.dH := M.H-main.H;
		main.handle(main, M)
	END DocumentModify;

	PROCEDURE ^ DocumentHandler*(V: Objects.Object; VAR M: Objects.ObjMsg);

	PROCEDURE DocumentModifyFrame(V: Viewer; F: Display.Frame; VAR M: Display.ModifyMsg);
		VAR handle: Objects.Handler;
	BEGIN
		handle := V.handle; V.handle := DocumentHandler; (* resize window -> disable Locked *)
		ToWindow(V, M); V.handle := handle;
		F.handle(F, M)
	END DocumentModifyFrame;

	(** default handler for document viewers (menu := V.dsc; main := menu.next) *)
	PROCEDURE DocumentHandler*(V: Objects.Object; VAR M: Objects.ObjMsg);	(** non-portable *)
		VAR menu, main: Display.Frame;
	BEGIN
		WITH V: Viewer DO
			IF M IS Display.FrameMsg THEN
				WITH M: Display.FrameMsg DO
					menu := V.dsc; main := menu.next; M.dlink := V; V.dlink := NIL;
					IF M IS Display.DisplayMsg THEN
						WITH M: Display.DisplayMsg DO
							IF M.F = V THEN
								IF M.device = Display.screen THEN
									M.id := Display.full;
									M.F := menu; menu.handle(menu, M);
									M.F := main; main.handle(main, M)
								END
							ELSE
								menu.handle(menu, M); main.handle(main, M)
							END
						END
					ELSIF M IS Display.ModifyMsg THEN
						WITH M: Display.ModifyMsg DO
							IF M.F = V THEN
								DocumentModify(V, menu, main, M.mode)
							ELSIF M.F = menu THEN
								DocumentModifyFrame(V, menu, M);
								DocumentModify(V, menu, main, M.mode)
							ELSIF M.F = main THEN
								DocumentModifyFrame(V, main, M);
								DocumentModify(V, menu, main, M.mode)
							ELSE
								menu.handle(menu, M); main.handle(main, M)
							END
						END
					ELSIF M IS Display.LocateMsg THEN
						WITH M: Display.LocateMsg DO
							IF (M.loc = NIL) & (M.X <= V.W) & (M.Y <= V.H) THEN
								M.x := 0; M.y := 0;
								IF M.Y <= main.H THEN
									main.handle(main, M)
								ELSE
									menu.handle(menu, M)
								END
							END
						END
					ELSE
						menu.handle(menu, M); main.handle(main, M)
					END
				END
			ELSE
				ToWindow(V, M)
			END
		END
	END DocumentHandler;

	PROCEDURE ControlModify(V: Viewer; F: Display.Frame; mode: INTEGER);
		VAR M: Display.ModifyMsg;
	BEGIN
		V.X := 0; V.Y := 0; V.W := SHORT(V.win.width); V.H := SHORT(V.win.height);

		M.F := F; M.dlink := V; M.id := Display.extend; M.mode := mode;
		M.x := 0; M.y := 0; M.res := MIN(INTEGER); Objects.Stamp(M);
		M.X := 0; M.dX := -F.X;
		M.Y := 0; M.dY := -F.Y;
		M.W := V.W; M.dW := M.W-F.W;
		M.H := V.H; M.dH := M.H-F.H;
		F.handle(F, M)
	END ControlModify;

	(** default handler for control viewers (F := V.dsc) *)
	PROCEDURE ControlHandler*(V: Objects.Object; VAR M: Objects.ObjMsg);	(** non-portable *)
		VAR F: Display.Frame;
	BEGIN
		WITH V: Viewer DO
			IF M IS Display.FrameMsg THEN
				WITH M: Display.FrameMsg DO
					F := V.dsc; M.dlink := V; V.dlink := NIL;
					IF M IS Display.DisplayMsg THEN
						WITH M: Display.DisplayMsg DO
							IF M.F = V THEN
								IF M.device = Display.screen THEN
									M.id := Display.full;
									M.F := F; F.handle(F, M)
								END
							ELSE
								F.handle(F, M)
							END
						END
					ELSIF M IS Display.ModifyMsg THEN
						WITH M: Display.ModifyMsg DO
							IF M.F = V THEN
								ControlModify(V, F, M.id)
							ELSIF M.F = F THEN
								F.handle(F, M);
								ControlModify(V, F, M.id)
							ELSE
								F.handle(F, M)
							END
						END
					ELSIF M IS Display.LocateMsg THEN
						WITH M: Display.LocateMsg DO
							IF (M.loc = NIL) & (M.X <= V.W) & (M.Y <= V.H) THEN
								M.x := 0; M.y := 0;
								F.handle(F, M)
							END
						END
					ELSE
						F.handle(F, M)
					END
				END
			ELSE
				ToWindow(V, M)
			END
		END
	END ControlHandler;

	(** update a viewer (complete redraw), if modify = TRUE resize contents to viewer size *)
	PROCEDURE Update*(V: Viewer; modify: BOOLEAN);	(** non-portable *)
		VAR
			cur: Displays.Display;
			menu, main: Display.Frame;
			C: Display.ControlMsg;
			D: Display.DisplayMsg;
	BEGIN
		IF V.win # NIL THEN
			Threads.Lock(moduleCS);
			cur := Display.cur; Display.SetCurrent(V.win);
			Display.ResetClip();
			IF (V.kind IN {IsDocument, IsControl}) & (V.state > 1) THEN
				menu := V.dsc; main := menu.next;
				C.F := NIL; C.id := Display.suspend; Send(V, C);
				IF modify THEN
					IF V.kind = IsDocument THEN
						DocumentModify(V, menu, main, Display.state)
					ELSIF V.kind = IsControl THEN
						ControlModify(V, menu, Display.state)
					END
				END;
				C.F := NIL; C.id := Display.restore; Send(V, C);
				D.F := V; D.device := Display.screen; D.id := Display.full; Send(V, D)
			ELSE
				IF V.kind = IsDisplay THEN SetUpdate(V, TRUE); V.state := 0 END;
				C.F := NIL; C.id := Display.suspend; Send(V, C);
				C.F := NIL; C.id := Display.restore; Send(V, C)
			END;
			Display.SetCurrent(cur);
			Threads.Unlock(moduleCS)
		END
	END Update;

	(** install a new viewer in the display-space, Open installs viewers automatically *)
	PROCEDURE Install*(V: Viewer);	(** non-portable *)
		VAR v: Display.Frame;
	BEGIN
		ASSERT(V.kind IN {IsDocument, IsControl});
		Threads.Lock(moduleCS);
		v := left.dsc.next;
		WHILE v.next # left.dsc DO
			v := v.next
		END;
		v.next := V; V.next := left.dsc;
		SetUpdate(V, TRUE);
		ZSort();
		Update(V, TRUE);
		Threads.Unlock(moduleCS)
	END Install;

	PROCEDURE IsInstalled*(V: Viewer): BOOLEAN;
		VAR F, v: Display.Frame;
	BEGIN
		Threads.Lock(moduleCS);
		F := FillerTrack.next;
		WHILE F # FillerTrack DO
			v := F.dsc;
			REPEAT
				v := v.next;
				IF v = V THEN
					Threads.Unlock(moduleCS);
					RETURN TRUE
				END
			UNTIL v = F.dsc;
			F := F.next
		END;
		Threads.Unlock(moduleCS);
		RETURN FALSE
	END IsInstalled;

	PROCEDURE InitWindow*(win: Window);
	BEGIN
		Displays.Init(win); win.viewer := NIL; win.handle := handle;
		NEW(win.track); win.track.id := Displays.track;
		win.track.keys := {}; win.track.X := 0; win.track.Y := 0
	END InitWindow;

	PROCEDURE InitFiller(V: Viewer);
	BEGIN
		V.handle := handle; V.next := V; V.dsc := NIL; V.win := NIL;
		V .X := 0; V.Y := 0; V.W := MAX(INTEGER); V.H := 0
	END InitFiller;

	PROCEDURE NewTrack(): Track;
		VAR track: Track; filler: Viewer;
	BEGIN
		NEW(track); InitFiller(track); track.kind := IsTrack;
		NEW(filler); InitFiller(filler); filler.kind := IsFiller;
		track.dsc := filler;
		RETURN track
	END NewTrack;

	PROCEDURE InitTracks();
	BEGIN
		FillerTrack := NewTrack(); FillerViewer := FillerTrack.dsc(Viewer);
		left := NewTrack(); left.X := MIN(INTEGER) DIV 2; left.W := 0;
		right := NewTrack(); right.X := MAX(INTEGER) DIV 2; right.W := 0;
		FillerTrack.next := left; left.next := right; right.next := FillerTrack
	END InitTracks;

BEGIN
	NEW(moduleCS); Threads.Init(moduleCS);
	recall := NIL; curW := 0; minH := 1; marked := NIL;
	bcF := NIL; bcLev := 0; sorted := FALSE;
	InitTracks(); tracksWin := NIL;
	Display.Broadcast := Broadcast
END Viewers.

(** Remarks:

1. Each track consists of a filler and a set of viewers linked together in a ring (with the next field) with the filler as sentinel. The filler is the top-most viewer in a track and covers the remainding part of the track the viewers do not cover. The set of tracks form the root objects of the display space.

2. Tracks can overlay each other. Closing a track exposes the track (and viewers) lying below it. Overlayed tracks and viewers do not receive message broadcasts in the display space. Before being overlayed, the contents of a track receive a Display.ControlMsg with id set to suspend.

3. The logical display increases from X coordinate 0 onwards through multiple physical displays. Opening a new display involves adding tracks beyond curW (typically a system and user track). Oberon uses a single coordinate system to address all the different displays. Note that many Oberon systems restrict the size of the display to the size of the host window.

4. Changing the top coordinate of a viewer with Change results in a Display.ModifyMsg with id set to reduce or extend (in size) being sent to the viewer contents (located in V.dsc).

5. The ratio of user and system track width is 5:3.

6. Programmers seldom need to use the Viewers module. Higher level modukes like Documents provide a simpler display abstraction.

*)
BIER_  8_   |^  ^  ^    <       g 
     C  Syntax10.Scn.Fnt 19.10.2002  12:31:52  "         d      d
     C  "         d      d
     C  TimeStamps.New TextGadgets.NewStyleProc  