TextDocs.NewDoc     |   CWindowsLeft    WindowsRight v  WindowsTop &   WindowsButtom   Color    Flat  Locked  Controls  Org L   BIER           3     Oberon10.Scn.Fnt     Syntax10.Scn.Fnt  Y         }            Syntax12.Scn.Fnt  |    
    =       f    7                     	                   e  (* ETH Oberon, Copyright 2001 ETH Zuerich Institut fuer Computersysteme, ETH Zentrum, CH-8092 Zuerich.
Refer to the "General ETH Oberon System Source License" contract available at: http://www.oberon.ethz.ch/ *)

MODULE Pictures; (** portable, except where noted / source: Win32.Pictures.Mod *)	(* ejz   *)
	IMPORT S := SYSTEM, Kernel32, Kernel, Files, Modules, Objects, User32, GDI32, Displays,
		Display, Texts, Oberon, Out;

(** Module Pictures implement an abstract data type (and object type) for manipulating colored bitmaps of various color depths. *)

	CONST
		TryAll = TRUE;	(* try all converters if first one fails *)
		redraw* = 4; resize* = 5;	(** UpdateMsg id. *)
		PictFileId*= - 4093;	(** First two bytes of a .Pict file (0F0H, 3H). *)
		ALIGN = 4;

	TYPE
		Picture* = POINTER TO PictureDesc;

		UpdateMsg* = RECORD (Display.FrameMsg)
			id*, u*, v*, w*, h*: INTEGER;
			pict*: Picture
		END;

		PictureDesc* = RECORD (Objects.ObjDesc)
			bmi*: GDI32.BitmapInfo;	(** non-portable *)
			colors: ARRAY 256 OF GDI32.ColorRef;
			wth: LONGINT; bits*: Kernel32.ADDRESS;	(** non-portable *)
			width*, height*, depth*: INTEGER	(** Width, height in pixels, and depth in bits per pixel (8). *)
		END;

	VAR
		colorD*: INTEGER;	(** Default bitmap color depth. *)
		handler: Objects.Handler;
		(* installable converters *)		
		cRes: LONGINT; cName: ARRAY 128 OF CHAR; cP: Picture;
		first: BOOLEAN;
	PROCEDURE InBoundaries(x, y, w, h: LONGINT): BOOLEAN;
	CODE {SYSTEM.i386}
		MOV EAX, x[EBP]
		MOV EBX, y[EBP]
		CMP EAX, 0
		JL false
		CMP EBX, 0
		JL false
		CMP w[EBP], EAX
		JLE false
		CMP h[EBP], EBX
		JLE false
		MOV EAX, 1
		LEAVE
		RET 16
	false:
		XOR EAX, EAX
		LEAVE 
		RET 16
	END InBoundaries;

	(** Get the color index of the bitmap pixel at x, y. *)
	PROCEDURE Get*(P: Picture; x, y: INTEGER): INTEGER;
		VAR ch: CHAR;
	BEGIN
		IF InBoundaries(x, y, P.width, P.height) THEN
			S.GET(P.bits+y*P.wth+x, ch);
			RETURN ORD(ch)
		ELSE
			RETURN Display.BG
		END
	END Get;

	(** Put a pixel of color col at x, y using mode. *)
	PROCEDURE Dot*(P: Picture; col: Display.Color; x, y, mode: INTEGER);
		VAR adr: Kernel32.ADDRESS; ch: CHAR;
	BEGIN
		IF InBoundaries(x, y, P.width, P.height) THEN
			adr := P.bits + y * P.wth + x;
			IF mode = Display.invert THEN
				S.GET(adr, ch); S.PUT(adr, S.VAL(CHAR, S.VAL(SET, ch) / S.VAL(SET, col)))
			ELSE
				S.PUT(adr, CHR(col))
			END
		END
	END Dot;

	(** Starting at position x, y, determine the longest run of the same colored pixels (col) on the same scanline.
		Afterwards x indicates the first pixel of a different color thatn col. *)
	PROCEDURE GetRun*(P: Picture; VAR col: INTEGER; VAR x: INTEGER; y: INTEGER);
	BEGIN
		IF InBoundaries(x, y, P.width, P.height) THEN
			col := Get(P, x, y); INC(x); 
			WHILE (Get(P, x, y) = col) & (x < P.width) DO INC(x) END
		END
	END GetRun;

	(** Copy a the block sx, sy, w, h from picture sP to position dx, dy in destination picture dP. Source and
		destination picture may be the same. *)
	PROCEDURE CopyBlock*(sP, dP: Picture; sx, sy, w, h, dx, dy, mode: INTEGER);
		VAR a0, b0, a, b, c: LONGINT; col4, t: SET; ch, ch0: CHAR;
	BEGIN
		IF InBoundaries(sx, sy, sP.width, sP.height) & InBoundaries(dx, dy, dP.width, dP.height) THEN
			IF sx + w > sP.width THEN w := sP.width - sx END;
			IF dx + w > dP.width THEN w := dP.width - dx END;
			IF sy + h > sP.height THEN h := sP.height - sy END;
			IF dy + h > dP.height THEN h := dP.height - dy END;
			a0 := sP.bits + sy * sP.wth + sx; a := dP.bits + dy * dP.wth + dx;
			IF (sP = dP) & ((sy < dy) OR (sy = dy) & (sx < dx)) THEN	(* eos 6.12.95 *)
				INC(a0, h * sP.wth); INC(a, h * dP.wth);
				IF mode = Display.invert THEN
					a0 := a0 + w - 1; a := a + w -1;
					WHILE h > 0 DO
						DEC(h); DEC(a0, sP.wth); DEC(a, dP.wth);
						c := w; b0 := a0; b := a;
						WHILE c > 4 DO S.GET(b0, col4); S.GET(b, t); S.PUT(b, col4 / t); DEC(c, 4); DEC(b0, 4); DEC(b, 4) END;
						WHILE c > 0 DO
							S.GET(b0, ch); S.GET(b, ch0); S.PUT(b, S.VAL(CHAR, S.VAL(SET, ch) / S.VAL(SET, ch0)));
							DEC(c); DEC(b0); DEC(b)
						END
					END
				ELSE
					WHILE h > 0 DO
						DEC(h); DEC(a0, sP.wth); DEC(a, dP.wth);
						S.MOVE(a0, a, w)
					END
				END
			ELSE
				IF mode = Display.invert THEN
					WHILE h > 0 DO
						c := w; b0 := a0; b := a; 
						WHILE c > 4 DO S.GET(b0, col4); S.GET(b, t); S.PUT(b, col4 / t); DEC(c, 4); INC(b0, 4); INC(b, 4) END;
						WHILE c > 0 DO S.GET(b0, ch); S.GET(b, ch0); S.PUT(b, S.VAL(CHAR, S.VAL(SET, ch) / S.VAL(SET, ch0))); DEC(c); INC(b0); INC(b) END;
						DEC(h); INC(a0, sP.wth); INC(a, dP.wth)
					END
				ELSE
					WHILE h > 0 DO
						S.MOVE(a0, a, w); 
						DEC(h); INC(a0, sP.wth); INC(a, dP.wth)
					END
				END
			END
		END
	END CopyBlock;

	PROCEDURE CopyBitmapPattern(P: Picture; col: Display.Color; hBm: GDI32.HBitmap; x, y, w, h: LONGINT; mode: INTEGER);
		VAR buf: ARRAY 1024 OF CHAR; i, j, c, dx, n, wth: LONGINT; adr: Kernel32.ADDRESS; ch: CHAR;
	BEGIN
		wth := 2*((w+15) DIV 16);
		GDI32.GetBitmapBits(hBm, wth*h, S.ADR(buf[0]));
		j := h-1;
		WHILE j >= 0 DO
			adr := S.ADR(buf[0])+(h-1-j)*wth;
			i := 0;
			WHILE i < w DO
				S.GET(adr, ch); c := ORD(ch);
				dx := i+8-1; n := 0;
				WHILE n < 8 DO
					IF ((c MOD 2) = 0) & (dx < w) THEN
						Dot(P, col, SHORT(x+dx), SHORT(y+j), mode) (* could optimze here, note: clipping done in Dot *)
					END;
					c := c DIV 2; DEC(dx);
					INC(n)
				END;
				INC(adr); INC(i, 8)
			END;
			DEC(j)
		END
	END CopyBitmapPattern;

	(** Copy the pattern pat in color col to position x, y using mode. *)
	PROCEDURE CopyPattern*(P: Picture; col: Display.Color; pat: Display.Pattern; x, y, mode: INTEGER);
		VAR
			p: Displays.Pattern; font: Displays.Font; str: ARRAY 4 OF CHAR;
			hBm: GDI32.HBitmap; fromHDC, toHDC: User32.HDC; w, h: INTEGER;
	BEGIN
		Display.GetDim(pat, w, h);
		IF InBoundaries(x, y, P.width, P.height) THEN
			toHDC := GDI32.CreateCompatibleDC(Displays.desktop.hDC);
			hBm := GDI32.CreateBitmap(w, h, 1, 1, NIL);
			GDI32.SelectObject(toHDC, hBm);
			IF pat < Display.FirstPattern THEN
				font := Displays.fontTable[pat DIV 256];
				str[0] := CHR(pat MOD 256); str[1] := 0X;
				GDI32.SelectObject(toHDC, font.hFont);
				GDI32.TextOut(toHDC, -font.metrics[ORD(str[0])].x, h+font.metrics[ORD(str[0])].y-font.maxY, str, 1)
			ELSE
				p := S.VAL(Displays.Pattern, pat);
				fromHDC := GDI32.CreateCompatibleDC(Displays.desktop.hDC);
				GDI32.SelectObject(fromHDC, p.hBm);
				GDI32.BitBlt(toHDC, 0, 0, w, h, fromHDC, p.x, p.y, GDI32.SrcCopy);
				GDI32.DeleteDC(fromHDC)
			END;
			CopyBitmapPattern(P, col, hBm, x, y, w, h, mode);
			GDI32.DeleteDC(toHDC); GDI32.DeleteObject(hBm)
		END
	END CopyPattern;

	(** Block fill area x, y, w, h in color col using mode. *)
	PROCEDURE ReplConst*(P: Picture; col: Display.Color; x, y, w, h, mode: INTEGER);
		VAR a, b, c: LONGINT; col4, t: SET; ch, ch0: CHAR;
	BEGIN 
		IF InBoundaries(x, y, P.width, P.height) THEN
			IF x + w > P.width THEN w := P.width - x END;
			IF y + h > P.height THEN h := P.height - y END;
			col4 := S.VAL(SET, col * 1010101H);
			ch := CHR(col);
			a := P.bits + y * P.wth + x;
			IF mode = Display.invert THEN
				WHILE h > 0 DO
					c := w; b := a;
					WHILE c > 4 DO S.GET(b, t); S.PUT(b, col4 / t); DEC(c, 4); INC(b, 4) END;
					WHILE c > 0 DO S.GET(b, ch0); S.PUT(b, S.VAL(CHAR, S.VAL(SET, ch) / S.VAL(SET, ch0))); DEC(c); INC(b) END;
					DEC(h); INC(a, P.wth)
				END
			ELSE
				WHILE h > 0 DO
					c := w; b := a;
					WHILE c > 4 DO S.PUT(b, col4); DEC(c, 4); INC(b, 4) END;
					WHILE c > 0 DO S.PUT(b, ch); DEC(c); INC(b) END;
					DEC(h); INC(a, P.wth)
				END
			END
		END
	END ReplConst;

	(** Pattern fill pattern pat in the area x, y, w, h in color col using mode. *)
	PROCEDURE ReplPattern*(P: Picture; col: Display.Color; pat: Display.Pattern; x, y, w, h, mode: INTEGER);
		VAR dx, dy, pw, ph: INTEGER;
	BEGIN
		Display.GetDim(pat, pw, ph);
		dy := 0;
		WHILE dy < h DO
			dx := 0;
			WHILE dx < w DO
				CopyPattern(P, col, pat, x+dx, y+dy, mode);
				INC(dx, pw)
			END;
			INC(dy, ph)
		END
	END ReplPattern;

	(** Copy area sx, sy, sw, sh of source picture sP to area dx, dy, dw, dh of destination picture dP. Appropriate
		scaling is done. *)
	PROCEDURE Copy*(sP, dP: Picture; sx, sy, sw, sh, dx, dy, dw, dh, mode: INTEGER);
		VAR
			hx, hy, DW2, SW2, dX, dY, SXo, DXo, ynew, xnew, wNew, wOld: LONGINT;
			adrOld, adrNew, adrOld0, adrNew0: Kernel32.ADDRESS;
			lineIndex: POINTER TO ARRAY OF INTEGER;
	BEGIN
		IF (sw > 0) & (sh > 0) & (dw > 0) & (dh > 0) THEN
			IF (sP.depth # 8) OR (dP.depth # 8) THEN
				dY := dy + dh; dX := dx + dw;
				SXo := sx; DXo := dx;
				DW2 := 2 * dw; SW2 := 2 * sw; 
				hy := 2*sh - dh; 
				WHILE dy < dY DO
					hx := 2* sw - dw; dx := SHORT(DXo); sx := SHORT(SXo);
					WHILE dx < dX DO
						Dot(dP, Get(sP, sx, sy), dx, dy, mode);
						WHILE hx > 0 DO INC(sx); DEC(hx, DW2) END;
						INC(dx); INC(hx, SW2)
					END;
					WHILE hy > 0 DO INC(sy); hy := hy - 2 * dh END;
					INC(dy); hy := hy + 2* sh
				END
			ELSE
				adrOld0 := sP.bits; adrNew0 := dP.bits; 
				wOld := sP.wth; wNew := dP.wth;
				IF sw = dw THEN
					FOR ynew := 0 TO dh-1 DO
						S.MOVE(adrOld0+sx+wOld*(sy+ynew*sh DIV dh), adrNew0+dx+wNew*(dy+ynew), sw)
					END
				ELSE
					NEW(lineIndex, dw); 
					FOR xnew := 0 TO dw-1 DO lineIndex[xnew] := SHORT(xnew*sw DIV dw) END;
					FOR ynew := 0 TO dh-1 DO
						adrOld := adrOld0+sx+wOld*(sy+ynew*sh DIV dh);
						adrNew := adrNew0+dx+wNew*(dy+ynew);
						FOR xnew := 0 TO dw-1 DO 
							S.MOVE(adrOld+lineIndex[xnew], adrNew+xnew, 1)
						END
					END
				END
			END
		END
	END Copy;

	(** Define the color palette for color index col. *)
	PROCEDURE SetColor*(P: Picture; col: Display.Color; red, green, blue: INTEGER);
	BEGIN
		P.colors[col] := GDI32.RGB(blue, green, red)
	END SetColor;

	(** Retrieve the color palette entry for color index col. *)
	PROCEDURE GetColor*(P: Picture; col: Display.Color; VAR red, green, blue: INTEGER);
		VAR c: GDI32.ColorRef;
	BEGIN
		c := P.colors[col];
		blue := SHORT(GDI32.Red(c));
		green := SHORT(GDI32.Green(c));
		red := SHORT(GDI32.Blue(c))
	END GetColor;

	(** Indicate that a change has been made to the area x, y, w, h of P. This results in an UpdateMsg with id = redraw
		to be broadcasted into the display space. *)
	PROCEDURE Update*(P: Picture; x, y, w, h: INTEGER);
		VAR msg: UpdateMsg;
	BEGIN
		msg.u := x; msg.v := y; msg.w := w; msg.h := h; 
		msg.id := redraw; msg.pict := P; msg.res := -1; msg.F := NIL;
		Display.Broadcast(msg)
	END Update;

	(** Copy the area x, y, w, h of picture P to position dx, dy on the display. *)
	PROCEDURE DisplayBlock*(P: Picture; x, y, w, h, dx, dy, mode: INTEGER);
	BEGIN
		IF mode = Display.invert THEN
HALT(99) (* use GDI32.StretchDIBits with invert ROP *)
		ELSE
			GDI32.SetDIBitsToDevice(Display.cur.hDC, dx, Display.cur.height-dy-h, w, h, x, y, 0, P.height, P.bits, P.bmi, GDI32.DIBRGBColors)
		END
	END DisplayBlock;

	PROCEDURE DisplayStretchedBlock*(P: Picture; x, y, w, h, dx, dy, dw, dh, mode: INTEGER);	(** non-portable *)
	BEGIN
		IF mode = Display.invert THEN
HALT(99) (* use GDI32.StretchDIBits with invert ROP *)
		ELSE
			GDI32.StretchDIBits(Display.cur.hDC, dx, Display.cur.height-dy-dh, dw, dh, x, y, w, h, P.bits, P.bmi, GDI32.DIBRGBColors, GDI32.SrcCopy)
		END
	END DisplayStretchedBlock;

	PROCEDURE *Finalize(P: PTR);
	BEGIN
		WITH P : Picture DO
			IF P.bits # Kernel32.NULL THEN
				Kernel32.VirtualFree(P.bits, P.bmi.bmiHeader.biSizeImage, {Kernel32.MEMDecommit});
				Kernel32.VirtualFree(P.bits, 0, {Kernel32.MEMRelease});
				P.bits := Kernel32.NULL; P.bmi.bmiHeader.biSizeImage := 0
			END
		END	
	END Finalize;

	PROCEDURE InitPicture(P: Picture; width, height: LONGINT);
		VAR i: LONGINT; r, g, b: INTEGER;
	BEGIN
		P.handle := handler;
		P.wth := ((width * colorD + 7) DIV 8 + ALIGN - 1) DIV ALIGN * ALIGN;
		P.bmi.bmiHeader.biSize := SIZE(GDI32.BitmapInfoHeader);
		P.bmi.bmiHeader.biWidth := width; P.bmi.bmiHeader.biHeight := height;
		P.bmi.bmiHeader.biPlanes := 1; P.bmi.bmiHeader.biBitCount := colorD;
		P.bmi.bmiHeader.biCompression := GDI32.BIRGB;
		P.bmi.bmiHeader.biSizeImage := P.wth*height;
		P.bmi.bmiHeader.biXPelsPerMeter := 0; P.bmi.bmiHeader.biYPelsPerMeter := 0;
		P.bmi.bmiHeader.biClrUsed := 256; P.bmi.bmiHeader.biClrImportant := 256;
		FOR i := 0 TO 255 DO
			Display.GetColor(i, r, g, b); P.colors[i] := GDI32.RGB(b, g, r)
		END;
		P.bits := Kernel32.VirtualAlloc(0, P.bmi.bmiHeader.biSizeImage, {Kernel32.MEMCommit}, {Kernel32.PageReadWrite});
		IF P.bits # Kernel32.NULL THEN
			Kernel.RegisterObject(P, Finalize, FALSE);
			P.width := SHORT(width); P.height := SHORT(height); P.depth := colorD
		ELSE
			P.width := 0; P.height := 0; P.depth := 0
		END
	END InitPicture;

	(** Create a picture of size width x height with depth bits per pixel. The picture palette is initialized to a default state. If 
	not enough memory is available to allocate the bitmap data, the width, height, and depth of the picture is set to zero. *) 
	PROCEDURE Create*(P: Picture; width, height, depth: INTEGER);
	BEGIN
		InitPicture(P, width, height);
		ReplConst(P, Display.BG, 0, 0, P.width, P.height, Display.replace)	
	END Create;

	(** Like Dot, for a line of pixels. *)
	PROCEDURE PutLine*(P: Picture; VAR data: ARRAY OF INTEGER; x, y, w: INTEGER);
		VAR adr: Kernel32.ADDRESS;
	BEGIN
		adr := P.bits+y*P.wth+x; x := 0;
		WHILE x < w DO
			S.PUT(adr, CHR(data[x])); INC(adr); INC(x)
		END
	END PutLine;

	(** Like Get, for a line of pixels. *)
	PROCEDURE GetLine*(P: Picture; VAR data: ARRAY OF INTEGER; x, y, w: INTEGER);
		VAR adr: Kernel32.ADDRESS; ch: CHAR;
	BEGIN
		adr := P.bits+y*P.wth+x; x := 0;
		WHILE x < w DO
			S.GET(adr, ch); data[x] := ORD(ch); INC(adr); INC(x)
		END
	END GetLine;

	PROCEDURE StoreCT(VAR R: Files.Rider; P: Picture);
		VAR i, n: LONGINT; c: GDI32.ColorRef;
	BEGIN
		i := 0; n := ASH(1, P.depth);
		WHILE i < n DO 
			c := P.colors[i];
			Files.Write(R, CHR(GDI32.Blue(c)));
			Files.Write(R, CHR(GDI32.Green(c)));
			Files.Write(R, CHR(GDI32.Red(c)));
			INC(i)
		END
	END StoreCT;

	PROCEDURE StoreData(VAR R: Files.Rider; P: Picture);
		VAR
			adr: Kernel32.ADDRESS; j, x, x0, y: LONGINT;
			h: SHORTINT; ch: CHAR; buf: ARRAY 129 OF SHORTINT;
	BEGIN
		y := P.height;
		WHILE y > 0 DO
			DEC(y);
			adr := P.bits+y*P.wth;
			buf[0] := 0; S.GET(adr, buf[1]); INC(adr);
			x0 := 0; x := 1; j := 1;
			WHILE x < P.width DO
				S.GET(adr, ch); INC(adr); h := SHORT(ORD(ch));
				IF ((x-x0) < 127) & ((buf[0] = 0) OR ((h = buf[j]) &(j = 1)) OR ((h # buf[j]) & (j > 1))) THEN (* same run *)
					IF h # buf[j] THEN INC(buf[0]); INC(j); buf[j] := h ELSE DEC(buf[0]) END
				ELSE (* new run *)
					IF (buf[j] = h) & ((x-x0) # 127) THEN
						DEC(buf[0]); Files.WriteBytes(R, buf, j); buf[0] := -1
					ELSE
						Files.WriteBytes(R, buf, j + 1); buf[0] := 0
					END;
					j := 1; buf[1] := h; x0 := x
				END;
				INC(x)
			END;
			Files.WriteBytes(R, buf, j + 1)
		END
	END StoreData;

	(** Stores picture run length encoded to file F (including tag). *)
	PROCEDURE Store*(P: Picture; F: Files.File; pos: LONGINT; VAR len: LONGINT);
		VAR R: Files.Rider;
	BEGIN
		Files.Set(R, F, pos); 	
		Files.WriteInt(R, PictFileId); 
		Files.WriteInt(R, P.width); Files.WriteInt(R, P.height); Files.WriteInt(R, P.depth);
		StoreCT(R, P); StoreData(R, P);
		len := Files.Pos(R) - pos
	END Store;

	PROCEDURE LoadCT(VAR R: Files.Rider; P: Picture; depth: LONGINT);
		VAR n, i: LONGINT; r, g, b: CHAR;
	BEGIN
		ASSERT(depth IN {1, 4, 8});
		n := ASH(1, depth);
		IF n = 2 THEN
			Files.Read(R, r); Files.Read(R, g); Files.Read(R, b);
			Files.Read(R, r); Files.Read(R, g); Files.Read(R, b)
		ELSE
			i := 0;
			WHILE i < n DO 
				Files.Read(R, r); Files.Read(R, g); Files.Read(R, b);
				P.colors[i] := GDI32.RGB(ORD(b), ORD(g), ORD(r));
				INC(i)
			END
		END
	END LoadCT;

	PROCEDURE LoadData1(VAR R: Files.Rider; P: Picture);
		VAR adr: Kernel32.ADDRESS; c, i, k, x, y: LONGINT; ch: CHAR;
	BEGIN
		y := P.height;
		WHILE y > 0 DO
			DEC(y);
			adr := P.bits+y*P.wth; x := 0;
			WHILE x < P.width DO
				Files.Read(R, ch); k := ORD(ch);
				IF k < 128 THEN
					REPEAT
						Files.Read(R, ch); c := ORD(ch);
						i := 0;
						WHILE (i < 8) & (x < P.width) DO
							IF ODD(c) THEN
								S.PUT(adr, CHR(Display.FG))
							ELSE
								S.PUT(adr, CHR(Display.BG))
							END;
							c := c DIV 2; INC(adr);
							INC(x); INC(i)
						END;
						DEC(k)
					UNTIL k < 0
				ELSE
					k := 257 - k; Files.Read(R, ch);
					REPEAT
						i := 0; c := ORD(ch);
						WHILE (i < 8) & (x < P.width) DO
							IF ODD(c) THEN
								S.PUT(adr, CHR(Display.FG))
							ELSE
								S.PUT(adr, CHR(Display.BG))
							END;
							c := c DIV 2; INC(adr);
							INC(x); INC(i)
						END;
						DEC(k) 
					UNTIL k < 1
				END
			END
		END
	END LoadData1;

	PROCEDURE LoadData4(VAR R: Files.Rider; P: Picture);
		VAR adr: Kernel32.ADDRESS; k, x, y: LONGINT; ch: CHAR;
	BEGIN
		y := P.height;
		WHILE y > 0 DO
			DEC(y);
			adr := P.bits+y*P.wth; x := 0;
			WHILE x < P.width DO
				Files.Read(R, ch); k := ORD(ch);
				IF k < 128 THEN 
					REPEAT 
						Files.Read(R, ch);
						S.PUT(adr, CHR(ORD(ch) MOD 16)); INC(adr);
						S.PUT(adr, CHR(ORD(ch) DIV 16)); INC(adr);
						DEC(k); INC(x, 2)
					UNTIL k < 0
				ELSE
					k := 257 - k; Files.Read(R, ch);
					REPEAT
						S.PUT(adr, CHR(ORD(ch) MOD 16)); INC(adr);
						S.PUT(adr, CHR(ORD(ch) DIV 16)); INC(adr);
						DEC(k); INC(x, 2)
					UNTIL k < 1
				END
			END
		END
	END LoadData4;

	PROCEDURE LoadData8(VAR R: Files.Rider; P: Picture);
		VAR adr: Kernel32.ADDRESS; k, x, y: LONGINT; ch: CHAR;
	BEGIN
		y := P.height;
		WHILE y > 0 DO
			DEC(y);
			adr := P.bits+y*P.wth; x := 0;
			WHILE x < P.width DO
				Files.Read(R, ch); k := ORD(ch);
				IF k < 128 THEN
					REPEAT
						Files.Read(R, ch);
						S.PUT(adr, ch); INC(adr);
						DEC(k); INC(x)
					UNTIL k < 0
				ELSE
					k := 257 - k; Files.Read(R, ch);
					REPEAT
						S.PUT(adr, ch); INC(adr);
						DEC(k); INC(x) 
					UNTIL k < 1
				END
			END
		END
	END LoadData8;

	(** Load a run length encoded picture from position pos from file F. Pos should be AFTER the two byte
		picture identification of a picture file. *)
	PROCEDURE Load*(P: Picture; F: Files.File; pos: LONGINT; VAR len: LONGINT);
		VAR R: Files.Rider; depth, id: INTEGER;
	BEGIN
		Files.Set(R, F, pos - 2); Files.ReadInt(R, id); 
		IF id = PictFileId THEN
			Files.ReadInt(R, P.width); Files.ReadInt(R, P.height); Files.ReadInt(R, depth);
			InitPicture(P, P.width, P.height); LoadCT(R, P, depth);
			CASE depth OF
				1: LoadData1(R, P)
				|4: LoadData4(R, P)
				|8: LoadData8(R, P)
			END
		ELSE HALT(99)	
		END;
		len := Files.Pos(R) - pos
	END Load;

	PROCEDURE SplitName (VAR name, MName, PName: ARRAY OF CHAR);
		VAR i, j: LONGINT;
	BEGIN i := 0;
		WHILE name[i] # "." DO MName[i] := name[i]; INC(i) END;
		MName[i] := 0X; INC(i); j := 0;
		WHILE name[i] # 0X DO PName[j] := name[i]; INC(i); INC(j) END;
		PName[j] := 0X
	END SplitName;

	PROCEDURE TryConverter(newproc: ARRAY OF CHAR);
		VAR
			mname, pname: ARRAY 32 OF CHAR;
			mod: Modules.Module; cmd: Modules.Command;
			M: Objects.FileMsg; F: Files.File;
	BEGIN
		SplitName(newproc, mname, pname);
		mod := Modules.ThisMod(mname);
		IF Modules.res = 0 THEN
			cmd := Modules.ThisCommand(mod, pname);
			IF Modules.res = 0 THEN Objects.NewObj := cP; cmd;
				F := Files.Old(cName);
				M.id := Objects.load; Files.Set(M.R, F, 0); cP.handle(cP, M);
				IF M.len = 0 THEN cRes := 1 ELSE cRes := 0 END
			ELSE cRes := 1
			END
		ELSE cRes := 1
		END
	END TryConverter;

	PROCEDURE GetExt(VAR name, ext: ARRAY OF CHAR);
		VAR i, j: LONGINT;
	BEGIN
		i := 0; j := 0; WHILE name[i] # 0X DO IF name[i] = "." THEN j := i+1 END; INC(i) END;
		i := 0; WHILE name[j] # 0X DO ext[i] := name[j]; INC(i); INC(j) END; ext[i] := 0X
	END GetExt;

	PROCEDURE Lower(VAR str: ARRAY OF CHAR);
		VAR i: LONGINT; ch: CHAR;
	BEGIN
		i := 0; ch := str[0];
		WHILE ch # 0X DO
			IF (ch >= "A") & (ch <= "Z") THEN str[i] := CHR(ORD(ch) - ORD("A") + ORD("a")) END;
			INC(i); ch := str[i]
		END
	END Lower;

	PROCEDURE ConverterOpen;
		VAR ext: ARRAY 32 OF CHAR; newproc: ARRAY 64 OF CHAR; s: Texts.Scanner;
	BEGIN
		cRes := 0; GetExt(cName, ext); Lower(ext);
		Oberon.OpenScanner(s, "PictureConverters");  newproc := "";
		IF first & (s.class = Texts.Inval) THEN
			first := FALSE;  Out.String("Oberon.Text - PictureConverters not found");  Out.Ln
		END;
		WHILE s.class IN {Texts.Name, Texts.String} DO
			IF s.s = ext THEN
				Texts.Scan(s);  Texts.Scan(s);
				IF s.class IN {Texts.Name, Texts.String} THEN
					COPY(s.s, newproc);  s.class := Texts.Inval
				END
			ELSE
				Texts.Scan(s);  Texts.Scan(s);  Texts.Scan(s)
			END
		END;
		IF newproc # "" THEN TryConverter(newproc) ELSE cRes := 1 END;
		IF TryAll & (cRes # 0) THEN
			Oberon.OpenScanner(s, "PictureConverters");
			WHILE (cRes # 0) & (s.class = Texts.Name) DO
				Texts.Scan(s);  Texts.Scan(s);
				IF (s.class = Texts.Name) & (s.s # newproc) THEN TryConverter(s.s) END;
				Texts.Scan(s)
			END
		END
	END ConverterOpen;

	(** Open the picture file with name from disk. Any graphic file format listed in the PictureConverters section
		of the Oberon.Text can be loaded (e.g. GIF, JPEG, XBM, BMP). *)
	PROCEDURE Open*(P: Picture; name: ARRAY OF CHAR; color: BOOLEAN);
		VAR F: Files.File; R: Files.Rider; dname: ARRAY 128 OF CHAR; len: LONGINT; id: INTEGER;
	BEGIN
		F := Files.Old(name);
		IF F # NIL THEN
			Files.Set(R, F, 0); Files.ReadInt(R, id);
			IF id = 07F7H THEN
				Files.ReadString(R, dname); Files.ReadInt(R, id); Files.ReadInt(R, id); Files.ReadInt(R, id); Files.ReadInt(R, id);
				Files.ReadInt(R, id)
			END;
			IF id = PictFileId THEN
				Load(P, F, Files.Pos(R), len); P.handle := handler
			ELSE
				cP := P; COPY(name, cName); ConverterOpen();
				IF cRes # 0 THEN P.width := 0; P.height := 0; P.depth := 0 END;
				cP := NIL; P.handle := handler
			END
		ELSE
			P.width := 0; P.height := 0; P.depth := 0
		END
	END Open;

	(** Returns the address of the bitmap data of a picture. *)
	PROCEDURE Address*(P: Picture): LONGINT;
	BEGIN
		RETURN P.bits
	END Address;

	(** Default picture object handler. *)
	PROCEDURE Handle*(P: Objects.Object; VAR M: Objects.ObjMsg);
		VAR len: LONGINT; P2: Picture; ref: INTEGER;
	BEGIN		
		WITH P: Picture DO
			IF M IS Objects.AttrMsg THEN
				WITH M: Objects.AttrMsg DO 
					IF (M.id = Objects.get) & (M.name = "Gen") THEN 
						M.class := Objects.String; M.s := "Pictures.NewPicture"; M.res := 0 
					END
				END
			ELSIF M IS Objects.FileMsg THEN
				WITH M: Objects.FileMsg DO
					IF M.id = Objects.store THEN
						Store(P, Files.Base(M.R), Files.Pos(M.R), len); 
						Files.Set(M.R, Files.Base(M.R), Files.Pos(M.R) + len)
					ELSIF M.id = Objects.load THEN
						Load(P, Files.Base(M.R), Files.Pos(M.R)+ 2, len); 
						Files.Set(M.R, Files.Base(M.R), Files.Pos(M.R) +2 + len)
					END
				END
			ELSIF M IS Objects.CopyMsg THEN
				WITH M: Objects.CopyMsg DO
					IF M.stamp = P.stamp THEN
						M.obj := P.dlink
					ELSE
						NEW(P2); P.stamp := M.stamp; P.dlink := P2;
						InitPicture(P2, P.width, P.height); P2.colors := P.colors;
						CopyBlock(P, P2, 0, 0, P.width, P.height, 0, 0, Display.replace);
						M.obj := P2
					END
				END
			ELSIF M IS Objects.BindMsg THEN
				WITH M: Objects.BindMsg DO
					IF (P.lib = NIL) OR ((P.lib.name = "") & (P.lib # M.lib)) THEN
						M.lib.GenRef(M.lib, ref); M.lib.PutObj(M.lib, ref, P)
					END
				END
			END
		END
	END Handle;

	(** Generator for a picture object. *)
	PROCEDURE NewPicture*;
		VAR P: Picture;
	BEGIN
		NEW(P); Create(P, 8, 8, colorD); Objects.NewObj := P
	END NewPicture;

BEGIN
	colorD := 8; handler := Handle; cP := NIL; first := TRUE
END Pictures.

(** Remarks

1. The origin (0, 0) is the bottom left corner of a bitmap. A picture has a bitmap and a color palette. The color palette specifies the reg, green, blue components (in the range 0 - 255) of the color indexes in the bitmap data. Only palettized bitmaps are supported in depths of 1, 4, and 8 bits per pixel. On most Oberon platforms the color palette of a picture is ignored when displaying the picture. Instead the picture is displayed with the current screen palette. To see the true colors of a picture, the picture palette has to be copied into the screen palette.

2. The drawing modes are the same as that of module Display (replace, paint, invert). After drawing into a picture, a module should indicate using procedure Update what area of the picture should be displayed fresh.

3. On many Oberon platforms the address of the bitmap data is set to zero. This indicates that it is impossible to access the bitmap directly.

4. Oberon for Windows only supports bit depths of 8 bits per pixel. The general bitmap allocation scheme is as follows. Address points to the bottom most scan-line of a picture. Each scan-line is a sequence of bytes (color indexes), with each scan line padded with unused bytes to make the length a multiple of 4. Color index entries have values depending on the display card used (and is typically a different value than the color indexes used to write into a picture). Win32.ColTrans and Win32.RevColTrans contains the forward and backward mappings of Oberon color indexes and Windows color indexes. As expected, usage of this knowledge results in unportable programs.

5. Implementation restrictions on Oberon for Windows
CopyPattern only works for Oberon fonts (no truetype fonts are supported currently).

6. When opening a GIF picture with Pictures.Open, the GIF background color is burned in to color 14 (light grey).

7. Implementation restriction
Picture objects cannot be named or have attributes (see module Attributes). Picture objects only understand the messages Objects.FileMsg, Objects.CopyMsg and Objects.BindMsg.

*)BIER?g  ig   f  f  g    <       g 
     C  Syntax10.Scn.Fnt 14.05.2004  08:02:04  "         d      d
     C  "         d      d
     C  TimeStamps.New TextGadgets.NewStyleProc  