TextDocs.NewDoc     O|   CWindowsLeft   WindowsRight f  WindowsTop #   WindowsButtom   Color    Flat  Locked  Controls  Org ۡ   BIER           3  ,  Oberon10.Scn.Fnt     Syntax10.Scn.Fnt  Q                             Syntax12.Scn.Fnt  |    5
   S    m                     g              
   &       S    *        A                   Q         (* ETH Oberon, Copyright 2001 ETH Zuerich Institut fuer Computersysteme, ETH Zentrum, CH-8092 Zuerich.
Refer to the "General ETH Oberon System Source License" contract available at: http://www.oberon.ethz.ch/ *)

MODULE Oberon; (** portable, except where noted / source: Win32.Oberon.Mod *)	(* JG 22.7.94, ejz   *)
	IMPORT SYSTEM, Kernel32, Kernel, Registry, Modules, Objects, User32, Displays, Display,
		Fonts, Input, Dates, Strings, Texts, Threads, Viewers, FileDir, Windows;

(**
Oberon system manager for dispatch of keyboard and mouse input, scheduling of tasks, cursor handling and command execution.
*)

	CONST
		(** Message ids: *)
		defocus* = 0; neutralize* = 1; mark* = 2; (** ControlMsg*)
		consume* = 0; track* = 1; (** InputMsg*)
		get* = 0; set* = 1; reset* = 2; (** CaretMsg id, SelectMsg id*)

		GCInterval = 100000;

		FillerColor = 12;

	TYPE
		Painter* = PROCEDURE (x, y: INTEGER);
		Marker* = RECORD
			Fade*, Draw*: Painter;	(** Remove and draw marker. *)
			hCursor*: User32.HCursor	(** non-portable *)
		END;

		Cursor* = RECORD
			marker*: Marker;	(** Cursor marker. *)
			on*: BOOLEAN;	(** Is cursor shown? *)
			X*, Y*: INTEGER	(** Absolute cursor position. *)
		END;

		ParList* = POINTER TO ParRec;
		ParRec* = RECORD	(** Area for passing command parameters. *)
			vwr*: Viewers.Viewer;	(** Viewer in which command is executed. *)
			frame*: Display.Frame;	(** Frame of vwr from where command is executed. *)
			obj*: Objects.Object;	(** Object in vwr executing command. *)
			text*: Texts.Text;	(** Text parameter to be passed to command. *)
			pos*: LONGINT	(** Starting position in text of parameter. *)
		END;

		ControlMsg* = RECORD (Display.FrameMsg)
			id*: INTEGER;	(** defocus, neutralize, mark *)
			X*, Y*: INTEGER	(** Absolute mark position. *)
		END;

		InputMsg* = RECORD (Display.FrameMsg)
			id*: INTEGER;	(** consume, track *)
			keys*: SET;	(** Mouse buttons. *)
			X*, Y*: INTEGER;	(** Mouse position. *)
			ch*: CHAR;	(** Character typed. *)
			fnt*: Fonts.Font;	(** Font of typed character. *)
			col*, voff*: SHORTINT	(** Color and vertical offset of typed character. *)
		END;

		CaretMsg* = RECORD (Display.FrameMsg)	(** Text caret handling. *)
			id*: INTEGER;	(** get, set, reset *)
			car*: Display.Frame;	(** Destination frame, returned frame. *)
			text*: Texts.Text;	(** Text represented by car. *)
			pos*: LONGINT	(** Caret position. *)
		END;

		SelectMsg* = RECORD (Display.FrameMsg)	(** Text selection handling. *)
			id*: INTEGER;	(** get, set, reset *)
			time*: LONGINT;	(** Time of the selection. *)
			sel*: Display.Frame;	(** Destination frame, returned frame. *)
			text*: Texts.Text;	(** Text represented by sel. *)
			beg*, end*: LONGINT	(** Text stretch of the selection. *)
		END;

		ConsumeMsg* = RECORD (Display.FrameMsg)	(** Drag and drop control of text. *)
			text*: Texts.Text;	(** Text to be inserted. *)
			beg*, end*: LONGINT	(** Text stretch to be inserted. *)
		END;

		RecallMsg* = RECORD (Display.FrameMsg)
		END;

		Task* = POINTER TO TaskDesc;
		Handler* = PROCEDURE (me: Task);
		TaskDesc* = RECORD
			next*: Task;	(** for internal use. *)
			time*: LONGINT;	(** Earliest time to schedule task. *)
			safe*: BOOLEAN;	(** Don't remove from task queue when a trap occurs. *)
			handle*: Handler	(** Task handler. *)
		END;

		UpdateObjProc* = PROCEDURE (obj: Objects.Object);	(** non-portable *)

	VAR
		Arrow*, Star*: Marker;	(** Normal Oberon arrow, and the star marker. *)
		Mouse*, Pointer*: Cursor;	(** Normal Oberon mouse, and the star pointer. *)
		Log*: Texts.Text;	(** The Oberon log. *)
		Par*: ParList;	(** Actual parameters of executed command. *)
		CurFnt*: Fonts.Font;	(** Current input font when typing. *)
		CurCol*, CurOff*: SHORTINT;	(** Current color and offset when typing. *)
		OptionChar*: CHAR;	(** Option character "/" or "\" *)
		OpenText*: PROCEDURE (title: ARRAY OF CHAR; T: Texts.Text; W, H: INTEGER);
		CurTask, NextTask*: Task;	(** non-portable, for internal use. *)
		New*: BOOLEAN;	(** enable new style mouse handling suitable for two-button mice *)
		SystemW: INTEGER;
		GCTask: Task; (* garbage collection task *)
		nextTaskTime: LONGINT;
		W: Texts.Writer;
		pointerPos: User32.Point;
		conftext: Texts.Text; confStamp: LONGINT;
		curPath: FileDir.FileName;
		(** upcalls called by the Oberon.Loop
				updateObj: upcall to Gadgets.Update
				execute: upcall to Gadgets.Execute
				install: upcall to WinPlugIns.Install
				print: upcall to WinFrames.Print *)
		updateObj*: UpdateObjProc;	(** non-portable *)
		execute*: PROCEDURE (executorObj: Display.Frame; cmd: ARRAY OF CHAR);	(** non-portable *)
		install*: PROCEDURE (win: Viewers.Window);	(** non-portable *)
		print*: PROCEDURE (V: Viewers.Viewer; print: Displays.PrintEvent);	(** non-portable *)
		event: Displays.Event;

(* clocks *)

	(** Get time (t) and date (d).
		day = d MOD 32, month = d DIV 32 MOD 16, year = 1900+d DIV 512,
		hour = t DIV 4096 MOD 32, minute = t DIV 64 MOD 64, second = t MOD 64 *)
	PROCEDURE GetClock*(VAR t, d: LONGINT);
		VAR st: Kernel32.SystemTime;
	BEGIN
		t := 0; d := 0;
		Kernel32.GetLocalTime(st);
		d := st.wYear - 1900; d := ASH(d, 4);
		d := d + st.wMonth; d := ASH(d, 5);
		d := d + st.wDay;
		t := st.wHour; t := ASH(t, 6);
		t := t + st.wMinute; t := ASH(t, 6);
		t := t + st.wSecond
	END GetClock;

	(** Set time (t) and date (d). *)
	PROCEDURE SetClock*(t, d: LONGINT);
		VAR st: Kernel32.SystemTime;
	BEGIN
		st.wDay := SHORT(d MOD 20H); d := ASH(d, -5);
		st.wMonth := SHORT(d MOD 10H); d := ASH(d, -4);
		st.wYear := SHORT(d MOD 80H) + 1900;
		st.wMilliseconds := 0;
		st.wSecond := SHORT(t MOD 40H); t := ASH(t, -6);
		st.wMinute := SHORT(t MOD 40H); t := ASH(t, -6);
		st.wHour := SHORT(t MOD 20H);
		Kernel32.SetLocalTime(st)
	END SetClock;

	(** Return the number of timer ticks since Oberon startup. (See module Input for frequency) *)
	PROCEDURE Time*(): LONGINT;
	BEGIN
		RETURN Kernel32.GetTickCount()
	END Time;

(* cursor handling *)

	PROCEDURE *FlipArrow(X, Y: INTEGER);
	(* Windows cursor is used instead *)
	END FlipArrow;

	PROCEDURE *FadeStar(X, Y: INTEGER);
		VAR cur: Displays.Display;
	BEGIN
		IF Viewers.marked # NIL THEN
			cur := Display.cur; Display.SetCurrent(Viewers.marked.win);
			Display.CopyPattern(Display.FG, Display.star, X - 7, Y - 7, Display.invert);
			Display.SetCurrent(cur); Viewers.marked := NIL
		END
	END FadeStar;

	PROCEDURE *DrawStar(X, Y: INTEGER);
	BEGIN
		Display.CopyPattern(Display.BG, Display.star, X - 7, Y - 7, Display.invert);
		IF (Display.cur # NIL) & (Display.cur IS Viewers.Window) THEN
			Viewers.marked := Display.cur(Viewers.Window).viewer;
			Pointer.X := X; Pointer.Y := Y;
			IF ~(Viewers.marked.kind IN {Viewers.IsDocument, Viewers.IsControl}) THEN
				Viewers.marked := Viewers.This(X, Y)
			END
		ELSE
			Viewers.marked := NIL
		END
	END DrawStar;

	(** Initialize a cursor, setting it to off, and at position 0, 0. *)
	PROCEDURE OpenCursor*(VAR c: Cursor);
	BEGIN
		c.on := FALSE; c.X := 0; c.Y := 0
	END OpenCursor;

	(** Fade cursor if visible. *)
	PROCEDURE FadeCursor*(VAR c: Cursor);
	BEGIN
		IF c.on THEN c.marker.Fade(c.X, c.Y); c.on := FALSE END
	END FadeCursor;

	(** Draw cursor c using marker m at position X, Y. *)
	PROCEDURE DrawCursor*(VAR c: Cursor; VAR m: Marker; X, Y: INTEGER);
	BEGIN
		IF c.on & ((X # c.X) OR (Y # c.Y) OR (m.Draw # c.marker.Draw)) THEN
			c.marker.Fade(c.X, c.Y); c.on := FALSE
		END;
		IF m.hCursor # c.marker.hCursor THEN
			Windows.SetCursor(m.hCursor)
		END;
		IF ~c.on THEN
			m.Draw(X, Y); c.marker := m; c.X := X; c.Y := Y; c.on := TRUE
		END
	END DrawCursor;

(* display management *)

	PROCEDURE Control(V: Viewers.Viewer; id: INTEGER; oberon: BOOLEAN);
		VAR DC: Display.ControlMsg; C: ControlMsg;
	BEGIN
		IF oberon THEN
			C.F := NIL; C.id := id;
			IF V # NIL THEN
				Viewers.Send(V, C)
			ELSE
				Display.Broadcast(C)
			END
		ELSE
			DC.F := NIL; DC.id := id;
			IF V # NIL THEN
				Viewers.Send(V, DC)
			ELSE
				Display.Broadcast(DC)
			END
		END
	END Control;

	(** Remove the caret by broadcasting a ControlMsg into the display space. Afterwards, no visual object
		should own either a caret for inserting text or objects. *)
	PROCEDURE Defocus*;
	BEGIN
		Control(NIL, defocus, TRUE)
	END Defocus;

	(** Fade the mouse and pointer cursors if located inside the screen area X, Y, W, H. This is required before
			drawing inside the area X, Y, W, H. *)
	PROCEDURE RemoveMarks*(X, Y, W, H: INTEGER);
	BEGIN
		IF (Mouse.X > X - 16) & (Mouse.X < X + W + 16) & (Mouse.Y > Y - 16) & (Mouse.Y < Y + H + 16) THEN
			FadeCursor(Mouse)
		END;
		IF (Pointer.X > X - 8) & (Pointer.X < X + W + 8) & (Pointer.Y > Y - 8) & (Pointer.Y < Y + H + 8) THEN
			FadeCursor(Pointer)
		END
	END RemoveMarks;

	PROCEDURE *HandleFiller(V: Objects.Object; VAR M: Objects.ObjMsg);
	BEGIN
		WITH V: Viewers.Viewer DO
			IF M IS InputMsg THEN
				WITH M: InputMsg DO
					IF M.id = track THEN DrawCursor(Mouse, Arrow, M.X, M.Y) END
				END
			ELSIF M IS ControlMsg THEN
				WITH M: ControlMsg DO
					IF M.id = mark THEN
						IF (M.X >= V.X) & (M.X < V.X + V.W) & (M.Y >= V.Y) & (M.Y < V.Y + V.H) THEN
							DrawCursor(Pointer, Star, M.X, M.Y)
						END
					END
				END
			ELSIF M IS Display.ControlMsg THEN
				WITH M: Display.ControlMsg DO
					IF (M.id = Display.restore) & (V.W > 0) & (V.H > 0) THEN
						RemoveMarks(V.X, V.Y, V.W, V.H);
						Display.ReplConst(FillerColor, V.X, V.Y, V.W, V.H, 0)
					END
				END
			ELSIF M IS Display.ModifyMsg THEN
				WITH M: Display.ModifyMsg DO
					IF (M.F = V) & (M.id = Display.extend) THEN
						RemoveMarks(V.X, M.Y, V.W, V.Y - M.Y);
						Display.ReplConst(FillerColor, V.X, M.Y, V.W, V.Y - M.Y, 0)
					END
				END
			ELSIF M IS Display.DisplayMsg THEN
				WITH M: Display.DisplayMsg DO
					IF (M.device = Display.screen) & ((M.F = V) OR (M.F = NIL)) THEN
						RemoveMarks(V.X, V.Y, V.W, V.H);
						Display.ReplConst(FillerColor, V.X, V.Y, V.W, V.H, Display.replace)
					END
				END
			ELSIF M IS Objects.AttrMsg THEN
				WITH M: Objects.AttrMsg DO
					IF (M.id = Objects.get) & (M.name = "LockedSize") THEN
						M.class := Objects.Bool; M.b := TRUE; M.res := 0
					END
				END
			END
		END
	END HandleFiller;

	(** Initialize a new display. Normally this procedure is only called once to configure the 
		default layout of the Oberon screen. In PlugIn Oberon this procedure can be called to
		open a (single) window with a tiled viewers system.
		Obern.OpenDisplay displaywidth displayheight userW ... systemW *)
	PROCEDURE OpenDisplay*;
		VAR
			S: Texts.Scanner; A: Objects.AttrMsg;
			Filler: Viewers.Viewer; UserW: INTEGER;
			start: Kernel32.StartupInfo; default: BOOLEAN;
	BEGIN
		IF Viewers.tracksWin = NIL THEN
			start.cb := SIZE(Kernel32.StartupInfo);
			Kernel32.GetStartupInfo(start);
			default := FALSE;
			Texts.OpenScanner(S, Par.text, Par.pos); Texts.Scan(S);
			IF (S.class = Texts.Name) & (CAP(S.s[0]) = "M") THEN
				Display.Width := MAX(INTEGER); Display.Height := MAX(INTEGER);
				Texts.Scan(S)
			ELSIF (S.class = Texts.Int) & (S.i > 0) & (S.i <= Displays.desktop.width) THEN
				Display.Width := SHORT(S.i); Texts.Scan(S);
				IF (S.class = Texts.Int) & (S.i > 0) & (S.i <= Displays.desktop.height) THEN
					Display.Height := SHORT(S.i); Texts.Scan(S)
				ELSE
					default := TRUE
				END
			ELSE
				IF (S.class = Texts.Name) & (CAP(S.s[0]) = "D") THEN
					Texts.Scan(S)
				END;
				default := TRUE
			END;
			NEW(Filler); Filler.kind := Viewers.IsDisplay; Filler.handle := HandleFiller;
			IF default THEN
				IF Kernel32.StartFUseSize IN start.dwFlags THEN
					Display.Width := SHORT(start.dwXSize); Display.Height := SHORT(start.dwYSize)
				ELSE
					Display.Width := 0; Display.Height := 0
				END
			END;
			Windows.OpenViewer(Filler, Display.Width, Display.Height);
			A.id := Objects.set; A.class := Objects.String; A.name := "Title"; A.s := "ETH Oberon";
			Filler.win.handle(Filler.win, A);
			Viewers.tracksWin := Filler.win; Display.SetCurrent(Filler.win);
			Display.Width := SHORT(Viewers.tracksWin.width);
			Display.Height := SHORT(Viewers.tracksWin.height);
			Par.vwr := Filler; Par.frame := Filler; Viewers.Update(Filler, TRUE);
			Display.ReplConst(FillerColor, 0, 0, Display.Width, Display.Height, Display.replace);
			IF (S.class = Texts.Int) & (S.i > 0) & (S.i <= Display.Width) THEN
				UserW := Display.Width;
				WHILE (S.class = Texts.Int) & (S.i > 0) & (UserW > 0) DO
					IF S.i > UserW THEN S.i := UserW END;
					NEW(Filler); Filler.handle := HandleFiller;
					Viewers.InitTrack(SHORT(S.i), Display.Height, Filler);
					UserW := SHORT(UserW-S.i); SystemW := SHORT(S.i);
					Texts.Scan(S)
				END;
				IF UserW > 0 THEN
					NEW(Filler); Filler.handle := HandleFiller;
					Viewers.InitTrack(UserW, Display.Height, Filler);
					SystemW := UserW
				END
			ELSE
				UserW := Display.Width DIV 8 * 5; SystemW := Display.Width-UserW;
				NEW(Filler); Filler.handle := HandleFiller;
				Viewers.InitTrack(UserW, Display.Height, Filler); (*init user track*)
				NEW(Filler); Filler.handle := HandleFiller;
				Viewers.InitTrack(SystemW, Display.Height, Filler) (*init system track*)
			END		
		ELSE
			HALT(99)
		END
	END OpenDisplay;

	(** Returns the width in pixels of the display that contains the X coordinate. *)
	PROCEDURE DisplayWidth*(X: INTEGER): INTEGER;
	BEGIN
		RETURN Display.Width
	END DisplayWidth;

	(** Returns the height in pixels of the display that contains the X coordinate. *)
	PROCEDURE DisplayHeight*(X: INTEGER): INTEGER;
	BEGIN
		RETURN Display.Height
	END DisplayHeight;

	(** Open a new track of width W at X. *)
	PROCEDURE OpenTrack*(X, W: INTEGER);
		VAR Filler: Viewers.Viewer;
	BEGIN
		NEW(Filler); Filler.handle := HandleFiller;
		Viewers.OpenTrack(X, W, Filler)
	END OpenTrack;

	(** Get left margin of user track on display X. *)
	PROCEDURE UserTrack*(X: INTEGER): INTEGER;
	BEGIN
		RETURN 0
	END UserTrack;

	(** Get left margin of the system track on display X. *)
	PROCEDURE SystemTrack*(X: INTEGER): INTEGER;
	BEGIN
		RETURN Display.Width-SystemW
	END SystemTrack;

	PROCEDURE UY(X: INTEGER): INTEGER;
		VAR fil, bot, alt, max: Display.Frame;
	BEGIN
		Viewers.Locate(X, 0, fil, bot, alt, max);
		IF fil.H >= Display.Height DIV 8 THEN RETURN Display.Height END;
		RETURN max.Y + max.H DIV 2
	END UY;

	(** Allocate a new user viewer within the display located at DX. (X, Y) returns the suggested position. *)
	PROCEDURE AllocateUserViewer*(DX: INTEGER; VAR X, Y: INTEGER);
	BEGIN
		IF Pointer.on THEN X := Pointer.X; Y := Pointer.Y
		ELSE X := 0; Y := UY(X)
		END
	END AllocateUserViewer;

	PROCEDURE SY(X: INTEGER): INTEGER;
		VAR
			fil, bot, alt, max: Display.Frame;
			H0, H1, H2, H3: INTEGER;
	BEGIN
		H3 := Display.Height - Display.Height DIV 3; H2 := H3 - H3 DIV 2; H1 := Display.Height DIV 5; H0 := Display.Height DIV 10;
		Viewers.Locate(X, Display.Height, fil, bot, alt, max);
		IF fil.H >= Display.Height DIV 8 THEN RETURN Display.Height END;
		IF max.H >= Display.Height - H0 THEN RETURN max.Y + H3 END;
		IF max.H >= H3 - H0 THEN RETURN max.Y + H2 END;
		IF max.H >= H2 - H0 THEN RETURN max.Y + H1 END;
		IF max # bot THEN RETURN max.Y + max.H DIV 2 END;
		IF bot.H >= H1 THEN RETURN bot.H DIV 2 END;
		RETURN alt.Y + alt.H DIV 2
	END SY;

	(** Allocate a new system viewer within the display located at DX. (X, Y) returns the suggested position. *)
	PROCEDURE AllocateSystemViewer*(DX: INTEGER; VAR X, Y: INTEGER);
	BEGIN
		IF Pointer.on THEN X := Pointer.X; Y := Pointer.Y
		ELSE X := Display.Width-SystemW; Y := SY(X)
		END
	END AllocateSystemViewer;

	(** Returns the star-marked viewer. *)
	PROCEDURE MarkedViewer*(): Viewers.Viewer;
	BEGIN
		IF Viewers.marked # NIL THEN
			RETURN Viewers.marked
		ELSE
			RETURN Viewers.ThisPoint(pointerPos)
		END
	END MarkedViewer;

	(** Returns the star-marked frame. *)
	PROCEDURE MarkedFrame*(): Display.Frame;
		VAR V: Viewers.Viewer; L: Display.LocateMsg;
	BEGIN
		L.loc := NIL; L.X := Pointer.X; L.Y := Pointer.Y; L.F := NIL;
		V := MarkedViewer();
		IF V # NIL THEN Viewers.Send(V, L) END;
		RETURN L.loc
	END MarkedFrame;

	(** Returns the text of the star-marked frame. *)
	PROCEDURE MarkedText*(): Texts.Text;
		VAR
			F, V: Display.Frame;
			L: Objects.LinkMsg;
			T: Texts.Text;
	BEGIN
		T := NIL; F := MarkedFrame();
		IF F = NIL THEN
			V := MarkedViewer();
			IF (V # NIL) & (V.dsc # NIL) THEN
				F := V.dsc.next
			END
		END;
		IF F # NIL THEN
			L.id := Objects.get; L.name := "Model"; L.obj := NIL; L.res := -1;
			F.handle(F, L);
			IF (L.obj # NIL) & (L.obj IS Texts.Text) THEN
				T := L.obj(Texts.Text)
			END
		END;
		RETURN T
	END MarkedText;

(* command interpretation *)

	(** Execute an Oberon command. Name should be a string of the form "M.P", where M is the module and P is the
		procedure of the command. Par is the command parameter record; it will be assigned to Oberon.Par so that the
		command can pick up its parameters. The new flag indicates if the module M should be reloaded from disk
		(obly possible if M is a "top" module, i.e. it has no clients. Res indicates success (res = 0) or failure (res # 0).
		Modules.resMsg contains an explanation of what went wrong when res # 0. *)
	PROCEDURE Call*(name: ARRAY OF CHAR; par: ParList; new: BOOLEAN; VAR res: INTEGER);
		VAR Mod: Modules.Module; P: Modules.Command; f: Objects.Object; i, j: LONGINT; cur: Displays.Display;
	BEGIN
		cur := Display.cur; Displays.FlushCharacterCache(cur);
		res := 1; i := 0; j := 0;
		WHILE name[j] # 0X DO
			IF name[j] = "." THEN i := j END;
			INC(j)
		END;
		IF i = 0 THEN i := j; name[j+1] := 0X END;
		name[i] := 0X;
		IF new THEN Modules.Free(name, FALSE) END;
		Mod := Modules.ThisMod(name);
		IF Modules.res = 0 THEN
			INC(i); j := i;
			WHILE name[j] # 0X DO name[j - i] := name[j]; INC(j) END;
			name[j - i] := 0X;
			P := Modules.ThisCommand(Mod, name);
			IF Modules.res = 0 THEN
				Par := par;
				IF par.frame # NIL THEN
					f := Par.frame;
					WHILE (f # NIL) & ~(f IS Viewers.Viewer) DO
						f := f.dlink
					END;
					IF f # NIL THEN Par.vwr := f(Viewers.Viewer) END
				END;
				IF (Par.vwr = NIL) & (cur # NIL) & (cur IS Viewers.Window) THEN
					Par.vwr := cur(Viewers.Window).viewer
				END;
				IF Par.vwr # NIL THEN
					Display.SetCurrent(Par.vwr.win)
				ELSE
					Display.SetCurrent(Viewers.tracksWin)
				END;
				P(); res := 0
			ELSE res := Modules.res
			END
		ELSE res := Modules.res
		END;
		Display.SetCurrent(cur)
	END Call;

	(** Execute an Oberon command. Note that only Par.text and Par.pos are set,
		Par.vwr and Par.frame are all NIL. Par.obj is set to executor. *)
	PROCEDURE Execute*(executor: Objects.Object; cmd: ARRAY OF CHAR);	(** non-portable *)
		VAR par: ParList; S: Texts.Scanner; res: INTEGER;
	BEGIN
		NEW(par); par.obj := executor;
		NEW(par.text); Texts.Open(par.text, "");
		Texts.WriteString(W, cmd); Texts.WriteLn(W);
		Texts.Append(par.text, W.buf);
		Texts.OpenScanner(S, par.text, 0); Texts.Scan(S);
		par.pos := Texts.Pos(S); Call(S.s, par, FALSE, res)
	END Execute;

	(** Returns the selected stretch [beg, end[ of the current selected text T. Time indicates the time of selection;
		time = -1 indicates that no text is currently selected. *)
	PROCEDURE GetSelection*(VAR text: Texts.Text; VAR beg, end, time: LONGINT);
		VAR M: SelectMsg;
	BEGIN
		M.F := NIL; M.id := get; M.time := -1; M.text := NIL; Display.Broadcast(M);
		text := M.text; beg := M.beg; end := M.end; time := M.time
	END GetSelection;

	PROCEDURE *GC(me: Task);
	BEGIN
		me.time := Time() + GCInterval; Kernel.GC()
	END GC;

	(** Install a background task. The background task is periodically activated by calling its handler when the system has nothing else to do. *)
	PROCEDURE Install* (T: Task);
		VAR t: Task;
	BEGIN
		t := NextTask;
		WHILE (t.next # NextTask) & (t.next # T) DO t := t.next END;
		IF t.next # T THEN T.next := t.next; t.next := T END
	END Install;

	(** Remove a background task. *)
	PROCEDURE Remove* (T: Task);
		VAR t: Task;
	BEGIN
		t := NextTask;
		WHILE (t.next # NextTask) & (t.next # T) DO t := t.next END;
		IF t.next = T THEN
			t.next := T.next;
			IF NextTask = T THEN NextTask := NextTask.next END
		END
	END Remove;

	(** Request a garbage collection to be done. The GC will take place immediately. *)
	PROCEDURE Collect*;
	BEGIN
		Kernel.GC()	(* independent of GC task *)
	END Collect;

	(** Set the default font used when typing characters. *)
	PROCEDURE SetFont* (fnt: Fonts.Font);
	BEGIN
		CurFnt := fnt;
		IF CurFnt = NIL THEN CurFnt := Fonts.Default END
	END SetFont;

	(** Set the color of typed characters. *)
	PROCEDURE SetColor*(col: SHORTINT);
	BEGIN
		CurCol := col
	END SetColor;

	(** Set the vertical offset of typed characters. *)
	PROCEDURE SetOffset*(voff: SHORTINT);
	BEGIN
		CurOff := voff
	END SetOffset;

	PROCEDURE *WriteEntry(key, value: ARRAY OF CHAR);
		VAR i: LONGINT; ch: CHAR; simple: BOOLEAN;
	BEGIN
		IF key # "" THEN
			Texts.Write(W, 022X); Texts.WriteString(W, key); Texts.Write(W, 022X); 
			Texts.WriteString(W, "=")
		END;
		ch := value[0];
		IF (CAP(ch) >= "A") & (CAP(ch) <= "Z") OR (ch = ".") OR (ch = "/") THEN (* Texts.Name *)
			INC(i); ch := value[i];
			WHILE (ch # 0X) & Texts.nameChars[ORD(ch)] DO
				INC(i); ch := value[i]
			END;
			simple := ch = 0X
		ELSIF ch = '"' THEN (* Texts.String *)
			simple := TRUE
		ELSIF (ch = "-") OR (ch = "+") OR ((ch >= "0") & (ch <= "9")) THEN (* Texts.Int *)
			INC(i); ch := value[i];
			WHILE (ch # 0X) & (((ch >= "0") & (ch <= "9")) OR ((CAP(ch) >= "A") & (CAP(ch) <= "F"))) DO
				INC(i); ch := value[i]
			END;
			simple := (ch = 0X) OR (CAP(ch) = "H")
		ELSE
			simple := FALSE
		END;
		IF ~simple THEN
			Texts.Write(W, "{"); Texts.WriteString(W, value); Texts.Write(W, "}")
		ELSE
			Texts.WriteString(W, value)
		END;
		Texts.WriteLn(W)
	END WriteEntry;

	PROCEDURE *Export(path: ARRAY OF CHAR);
		VAR bakPath: ARRAY 256 OF CHAR;
	BEGIN
		COPY(curPath, bakPath);
		Texts.WriteString(W, path); Texts.WriteString(W, "={"); Texts.WriteLn(W);
		Strings.AppendCh(curPath, "\"); Strings.Append(curPath, path);
		Registry.EnumerateKeyValue(Registry.CurrentUser, curPath, WriteEntry);
		Registry.EnumeratePath(Registry.CurrentUser, curPath, Export);
		Texts.Write(W, "}"); Texts.WriteLn(W);
		COPY(bakPath, curPath)
	END Export;

	PROCEDURE ExpandCommandLine();
		VAR name, value: FileDir.FileName; adr: Kernel32.ADDRESS; i: LONGINT; ch, end: CHAR; val: BOOLEAN;
		PROCEDURE WriteNameValue();
		BEGIN
			IF i > 0 THEN
				value[i] := 0X;
				IF name = "" THEN COPY(value, name); value := "" END;
				Texts.Write(W, '"'); Texts.WriteString(W, name);
				IF (value # "") OR val THEN
					Texts.WriteString(W, '"="');
					Texts.WriteString(W, value)
				END;
				Texts.Write(W, '"'); 
				Texts.WriteLn(W)
			END;
			i := 0; name := ""; val := FALSE
		END WriteNameValue;
	BEGIN
		(* cmd  { ( "-" name [ "=" value ] ) | name } *)
		Texts.WriteString(W, "CommandLine={"); Texts.WriteLn(W);
		i := 0; name := ""; val := FALSE;
		adr := Kernel32.GetCommandLine();
		REPEAT
			SYSTEM.GET(adr, ch); INC(adr);
			IF ch = "-" THEN
				WriteNameValue(); val := TRUE
			ELSIF ch = "=" THEN 
				value[i] := 0X; i := 0; COPY(value, name)
			ELSIF ch > " " THEN
				IF (ch = '"') OR (ch = "'") THEN
					end := ch;
					SYSTEM.GET(adr, ch); INC(adr);
					WHILE (ch # 0X) & (ch # end) DO
						value[i] := ch; INC(i);
						SYSTEM.GET(adr, ch); INC(adr)
					END
				ELSE
					value[i] := ch; INC(i)
				END
			ELSE
				WriteNameValue()		
			END
		UNTIL ch = 0X;
		WriteNameValue();
		Texts.Write(W, "}"); Texts.WriteLn(W)
	END ExpandCommandLine;

	(* Configuration = Group.
		Group = { Entry }.
		Entry = [ Name "=" ] Value.
		Value = Token | "{" Group "}".
		Token = any token from Texts.Scanner, where "{" and "}" must occur pairwise.
		A named group is refered to as a "section". *)

	PROCEDURE SkipGroup (VAR S: Texts.Scanner);
		VAR openBraces: LONGINT;
	BEGIN
		openBraces := 1;
		REPEAT Texts.Scan(S);
		IF S.class = Texts.Char THEN
			IF S.c = "{" THEN INC(openBraces) ELSIF S.c = "}" THEN DEC(openBraces) END
		END
		UNTIL S.eot OR (openBraces = 0)
	END SkipGroup;

	(** Open a scanner at a specific section of the Oberon Text. Scans the first symbol in the section.
		Returns S.class = Texts.Inval on error.
		Command line parameters to oberon.exe are copied to section "CommandLine". *)
	PROCEDURE OpenScanner*(VAR S: Texts.Scanner; name: ARRAY OF CHAR);
		VAR i, j: LONGINT; done, eos: BOOLEAN; part: ARRAY 32 OF CHAR;
	BEGIN
		IF (conftext = NIL) OR (confStamp # Registry.stamp) THEN
			NEW(conftext); Texts.Open(conftext, "");
			COPY(Registry.oberonRoot, curPath); confStamp := Registry.stamp;
			Registry.EnumeratePath(Registry.CurrentUser, Registry.oberonRoot, Export);
			ExpandCommandLine();
			Texts.Append(conftext, W.buf)
		END;
		Texts.OpenScanner(S, conftext, 0); Texts.Scan(S); done := TRUE;
		WHILE (name[0] # 0X) & ~S.eot & done DO
			(* extract name part *) i := 0;
			WHILE (name[i] # 0X) & (name[i] # ".") DO part[i] := name[i]; INC(i) END;
			part[i] := 0X;
			IF name[i] = "." THEN INC(i) END;
			j := 0;
			WHILE name[i] # 0X DO name[j] := name[i]; INC(i); INC(j) END;
			name[j] := 0X;
			done := FALSE; eos := FALSE;
			REPEAT
				IF (S.class IN {Texts.Name, Texts.String}) THEN Texts.Scan(S);
					IF ~S.eot THEN
						IF (S.class = Texts.Char) & (S.c = "=") THEN (* named entry *)
							IF S.s = part THEN Texts.Scan(S);
							IF ~S.eot & (S.class = Texts.Char) & (S.c = "{") THEN Texts.Scan(S) END;
							done := TRUE
						ELSE Texts.Scan(S);
							IF ~S.eot & (S.class = Texts.Char) & (S.c = "{") THEN SkipGroup(S) END;
							IF ~S.eot THEN Texts.Scan(S) ELSE eos := TRUE END
						END
					ELSE eos := TRUE
					END
					ELSE eos := TRUE
					END
				ELSIF (S.class = Texts.Char) & (S.c = "{") THEN (* unnamed entry *) SkipGroup(S);
					IF ~S.eot THEN Texts.Scan(S) ELSE eos := TRUE END
				ELSE eos := TRUE
				END
			UNTIL done OR eos
		END;
		IF ~done OR (conftext.len = 0) THEN S.class := Texts.Inval END
	END OpenScanner;

	(** Lock Viewers.Broadcast. *)
	PROCEDURE LockOberon*();	(** non-portable *)
	BEGIN
		Threads.Lock(Viewers.moduleCS)
	END LockOberon;

	(** Unlock Viewers.Broadcast. *)
	PROCEDURE UnlockOberon*();	(** non-portable *)
	BEGIN
		Threads.Unlock(Viewers.moduleCS)
	END UnlockOberon;

	(** Broadcast ControlMsg with id=neutralize to all viewers (ESC/F2). *)
	PROCEDURE Neutralize*;	(** non-portable *)
	BEGIN
		Control(NIL, neutralize, TRUE); FadeCursor(Pointer)
	END Neutralize;

	(** Send ControlMsg with id=mark to viewer unter the mouse pointer (F1, F7). *)
	PROCEDURE Setup*;	(** non-portable *)
		VAR V: Viewers.Viewer; N: ControlMsg; keys: SET;
	BEGIN
		User32.GetCursorPos(pointerPos);
		V := Viewers.ThisPoint(pointerPos);
		IF V # NIL THEN
			Display.SetCurrent(V.win);
			N.F := V; N.id := mark;
			Input.Mouse(keys, N.X, N.Y);
			Viewers.Send(V, N)
		END
	END Setup;

	(** Redraw viewer unter the mouse pointer (F9). *)
	PROCEDURE RefreshDisplay*;	(** non-portable *)
		VAR V: Viewers.Viewer;
	BEGIN
		IF (Display.cur # NIL) & (Display.cur IS Viewers.Window) THEN
			V := Display.cur(Viewers.Window).viewer;
			Viewers.Update(V, FALSE)
		END
	END RefreshDisplay;

	(** Invert the value of Oberon.New (F8). *)
	PROCEDURE FlipNew*;
	BEGIN
		New := ~New; RefreshDisplay()
	END FlipNew;

	PROCEDURE Init();
		VAR
			S: Texts.Scanner; diff: ARRAY 8 OF CHAR;
			pename: FileDir.FileName;
			i, j: LONGINT; done: BOOLEAN;
	BEGIN
		NEW(Par); NEW(Par.vwr); Par.vwr.kind := -1; NEW(Par.frame); Par.vwr.dsc := Par.frame;
		OptionChar := "\"; OpenText := NIL; CurTask := NIL;
		Arrow.Fade := FlipArrow; Arrow.Draw := FlipArrow;
		Arrow.hCursor := User32.LoadCursor(Kernel.hInstance, "Arrow");
		Star.Fade := FadeStar; Star.Draw := DrawStar; Star.hCursor := Kernel32.NULL;
		Mouse.marker := Arrow; Mouse.on := TRUE;
		OpenCursor(Mouse); OpenCursor(Pointer);
		SystemW := Display.Width-(Display.Width DIV 8 * 5);
		NEW(GCTask); GCTask.handle := GC; GCTask.safe := TRUE;
		NextTask := GCTask; NextTask.next := NextTask; Collect(); nextTaskTime := Time();
		CurFnt := Fonts.Default; CurCol := Display.FG; CurOff := 0;
		conftext := NIL; Texts.OpenWriter(W); NEW(Log); Texts.Open(Log, "");
		OpenScanner(S, "System.MouseButtons"); New := (S.class = Texts.Int) & (S.i = 2);
		OpenScanner(S, "System.TimeDiff"); Dates.TimeDiff := 0;
		IF S.class = Texts.String THEN
			COPY(S.s, diff);
			i := 0; j := 1;
			IF diff[i] = "+" THEN
				INC(i)
			ELSIF diff[i] = "-" THEN
				INC(i); j := -1
			END;
			WHILE (diff[i] >= "0") & (diff[i] <= "9") DO
				Dates.TimeDiff := 10*Dates.TimeDiff+ORD(diff[i])-ORD("0");
				INC(i)
			END;
			Dates.TimeDiff := (Dates.TimeDiff DIV 100)*60 + (Dates.TimeDiff MOD 100);
			Dates.TimeDiff := j*Dates.TimeDiff
		END;
		updateObj := NIL; execute := NIL; install := NIL; print := NIL;
		IF Kernel.isEXE THEN
			Kernel32.GetModuleFileName(Kernel.hInstance, pename, LEN(pename));
			Strings.GetSuffix(pename, diff); Strings.Upper(diff, diff);
			IF diff = "SCR" THEN
				done := Displays.PutCmd(NIL, NIL, "System.StartDLL", 0);
				done := Displays.PutCmd(NIL, NIL, "ScreenSaver.Init", 0)
			ELSE (* EXE *)
				done := Displays.PutCmd(NIL, NIL, "System.StartEXE", 0)
			END
		ELSE
			done := Displays.PutCmd(NIL, NIL, "System.StartDLL", 0)
		END
	END Init;

	PROCEDURE *OLoop();
		VAR
			V: Viewers.Viewer;
			M: InputMsg;
			lastTask: Task;
			time: LONGINT;
	BEGIN
		IF conftext = NIL THEN
			Init()
		ELSIF (CurTask # NIL) & ~CurTask.safe THEN
			Remove(CurTask)
		END;
		IF (event # NIL) & (event.done # NIL) THEN
			Threads.Set(event.done)
		END;
		Display.SetCurrent(NIL);
		Windows.SetCursor(Arrow.hCursor);
		LOOP
			event := Displays.GetEvent();
			WHILE event # NIL DO
				V := NIL;
				IF event.disp # NIL THEN
					event.disp.updateDC := TRUE;
					IF event.disp IS Viewers.Window THEN
						V := event.disp(Viewers.Window).viewer
					END
				END;
				Display.SetCurrent(event.disp);
				CASE event.id OF
					|Displays.create: IF V = NIL THEN
													install(event.disp(Viewers.Window))
												ELSIF ~Viewers.IsInstalled(V) & (V.kind IN {Viewers.IsDocument, Viewers.IsControl}) THEN																Viewers.Install(V)
												ELSE
													Control(V, Display.restore, FALSE)
												END
					|Displays.remove: Viewers.Close(V)
					|Displays.restore: IF V.state < 0 THEN V.state := -V.state END;
													Control(V, Display.restore, FALSE)
					|Displays.suspend: Control(V, Display.suspend, FALSE);
													IF V.state > 0 THEN V.state := -V.state END
					|Displays.redraw: Viewers.Update(V, FALSE)
					|Displays.resize: Viewers.Update(V, TRUE)
					|Displays.print: print(V, event(Displays.PrintEvent))
					|Displays.focus:
					|Displays.defocus: Control(V, defocus, TRUE)
					|Displays.consume: M.F := NIL; M.id := consume;
													M.ch := event(Displays.InputEvent).ch;
													M.fnt := CurFnt; M.col := CurCol; M.voff := CurOff;
													Display.Broadcast(M)
					|Displays.track: M.F := NIL; M.id := track;
												WITH event: Displays.InputEvent DO
													M.keys := event.keys; M.X := SHORT(event.X); M.Y := SHORT(event.Y)
												END;
												IF ~(V.kind IN {Viewers.IsDocument, Viewers.IsControl}) THEN
													V := Viewers.This(M.X, M.Y)
												END;
												Viewers.Send(V, M)
					|Displays.wheel: M.F := NIL; M.id := consume;
												M.fnt := CurFnt; M.col := CurCol; M.voff := CurOff;
												WITH event: Displays.InputEvent DO
													IF event.zDelta > 0 THEN
														M.ch := 0C1X (* Up *)
													ELSIF event.zDelta < 0 THEN
														M.ch := 0C2X (* Down *)
													ELSE
														M.ch := 0X
													END;
													IF M.ch # 0X THEN Display.Broadcast(M) END
												END
					|Displays.update: updateObj(event(Displays.UpdateEvent).obj)
					|Displays.execute: WITH event: Displays.CommandEvent DO
														IF (event.executor # NIL) & (event.executor IS Display.Frame) & (execute # NIL) THEN
															execute(event.executor(Display.Frame), event.cmd)
														ELSE
															Execute(event.executor, event.cmd)
														END
													END
					|Displays.quit: Threads.oberonLoop := NIL; Threads.Set(event.done);
						Kernel32.Str("Oberon.Loop: quit"); Kernel32.Ln(); Display.SetCurrent(NIL); RETURN
					|Displays.ping:
				END;
				IF event.done # NIL THEN Threads.Set(event.done) END;
				event := Displays.GetEvent()
			END;
			event := NIL; lastTask := NextTask;
			WHILE Displays.eventObj.nEvents <= 0 DO
				CurTask := NextTask; NextTask := CurTask.next;
				time := Kernel32.GetTickCount();
				IF (CurTask.time-time) <= 0 THEN
					CurTask.handle(CurTask);
					nextTaskTime := CurTask.time; lastTask := CurTask
				ELSIF (CurTask.time-nextTaskTime) < 0 THEN
					nextTaskTime := CurTask.time
				ELSIF NextTask = lastTask THEN
					IF (time-nextTaskTime) < 0 THEN
						Kernel32.WaitForSingleObject(Displays.eventObj.handle, nextTaskTime-time)
					ELSE
						nextTaskTime := CurTask.time
					END
				END
			END;
			CurTask := NIL
		END
	END OLoop;

(* Main Oberon task dispatcher. Reads the mouse position and characters typed, informing the viewer of the display space of events using the Display.InputMsg. The loop broadcasts a ControlMsg (id = mark) when the marker is set. Pressing the neutralise key results in a ControlMsg (id = neutralize) to be broadcast. All frames receiving the neutralize message should remove selections and the caret. The Loop periodically activates background tasks and the garbage collector, if no mouse or keyboard events are arriving.*)
	PROCEDURE *Loop;
		VAR done: BOOLEAN;
	BEGIN
		LockOberon();
		ASSERT(Threads.oberonLoop = NIL);
		NEW(Threads.oberonLoop);
		Threads.oberonLoop.name := "Oberon.Loop";
		Threads.oberonLoop.safe := TRUE;
		Threads.Start(Threads.oberonLoop, OLoop, 0);
		UnlockOberon();
		IF Kernel.isEXE THEN (* await loop to start *)
			done := Displays.PutEvent(NIL, NIL, Displays.ping, Threads.Infinite)
		END
	END Loop;

(** Display the contents of the Oberon registry entries in Oberon.Text format.
	Note: to change enteries you must use either the command System.Set, IniEdit.Panel
	or the windows utility regedit . *)
	PROCEDURE Text*;
		VAR S: Texts.Scanner;
	BEGIN
		OpenScanner(S, "");
		OpenText("Oberon.Text", conftext, 512, 512)
	END Text;

BEGIN
	conftext := NIL; Loop()
END Oberon.

(** Remarks:

1. Command execution
Execution of commands is the task of modules Module. Oberon.Call provides an abstraction for this mechanism and also a way to pass parameters in the form of a text to the executed command. After command execution, the global variable Oberon.Par is a pointer to a parameter record specifying a parameter text, a position in that text, and details what objects are involved in the commands. The vwr field of the ParRec points to the viewer in which the command was executed. The frame field of the ParRec points to the direct child of the vwr (the menu or the main frame) from which the command was executed. This semantics is compatible with older Oberon applications and is seldomly used today. The obj field of the ParRec points to the object (normally a frame) that executed the command. The Oberon.Par pointer is initialized before command execution to the parameter record passed to Oberon.Call.

2. Cursors and Markers
Markers are a way to draw and undraw a shape on the display. Typically, draw and undraw can be realized by an invert display operation. Cursors keep track of the current position and state (visible or not) of a marker. The Mouse cursor is the standard mouse arrow, and the Pointer cursor is the star marker placed with the Setup key. Repeatedly calling Oberon.DrawCursor with different coordinates move a cursor (and marker) across the display. Before drawing in a certain area of the display, cursors should be removed with Oberon.RemoveMarks or Oberon.FadeCursor (failingly to do so may result in the cursor leaving garbage on the display when drawing operations are performed in its vicinity). Note that on some Oberon host platforms (Windows, Mac, Unix) the mouse cursor is under control of the host windowing system, and is automatically faded when required. It is recommended to fade the cursor on these platforms anyway, as your Oberon programs will then also work on native Oberon systems.

3. The InputMsg
The InputMsg informs the frames of the display space of the current mouse position and character typed. It is repeatedly broadcast into the display space by the Oberon.Loop for each input event. An InputMsg id of Oberon.consume indicates a key was pressed. The ASCII keycode is contained in the ch field of the message (check the description of module Input for special keycodes). The fields fnt, col and voff give information about the requested font, color (index), and verticall offset (in pixels). These values are copied from hidden variables in the Oberon module, which are set with the procedures SetFont, SetColor, and SetOffset. Note that the TextGadgets ignore these fields when typing. Instead the font, color and vertical offset of the character immediately before the caret is used (thus the fields have fallen out of use). A frame should only process the consume message if it has the caret set. Afterwards the message should be invalidated by setting the message res field to 0 or a positive value. This prevents the character being consumed by other frames in the display space and also terminates the message broadcast.
 An InputMsg id of track indicates a mouse event. The display space normally only forwards this message to the frame located at position X, Y on the display. Field X, Y indicates the absolute mouse position (cursor hotspot) and keys the mouse button state (which mouse buttons are pressed). The mouse buttons are numbered 0, 1, 2 for right, middle, and left respectively. It is typical for a frame receiving a track message with keys # {} to temporarily taking control of the mouse by polling (using module Input). As soon as all mouse buttons are released, control must be passed back to the Oberon loop. A frame should invalidate the track message if it took action on a mouse event; otherwise the enclosing frame might think that the message could not be handled. In some cases a child frame takes no action on an event even though a mouse buttton is pressed and the mouse is located inside the frame. This is an indication that the child frame cannot or is unwilling to process the event, and the parent (and forwarder of the message in the display space) should take a default action. Note that during a tracking operation, no background tasks can be serviced.

4. The ControlMsg
The control message manages display space wide events like removing the (one and only) caret, pressing Neutralise (for removing the caret and the selections), and setting the star marker with the Setup key. The id field of the control message is set to defocus, neutralize, and mark respectively. Note that the mark variant need not be handled by own frames; it is already processed by the Viewers. Messages of this type must never be invalidated during their travels through the display space.

5. The CaretMsg
The CaretMsg controls the removing (id = reset), retrieving (id = get) and setting (id = set) of the caret on a frame to frame basis. All text editor-like frames should respond to this message. The car field of the message defines the editor frame involved for reset and set, and returns the editor frame that has the caret for get. The text field specifies which text is meant (or which text is returned for get). In the reset and set cases this field is mostly redundant but is checked for correctness ANYWAY. The pos field must be valid position in the text. The CaretMsg is always broadcast.

6. The SelectMsg
In a similar way as the CaretMsg, the SelectMsg controls the removing (id = reset), retrieving (id = get) and setting (id = set) of the selection. In this case, the sel field indicates the destination frame or returned frame, in a similar manner as the car field of the CaretMsg. The SelectMsg is extended with fields for specifying/retrieving the selection time, starting and ending position. The SelectMsg is always broadcast.

7. Background tasks
The handle procedure variable of installed background tasks are periodically called by the Oberon loop when the Oberon system is idle (no mouse or keyboard events). The task handlers have to be programmed in an inverted manner and should return as quickly as possible to the loop, otherwise the user will notice delays (typically when elapsed time is greater than 100-200ms). As tasks are activated periodically, a badly written task can cause a cascade of traps, one for each invocation. By default, the loop removes such a task that does not return from the task list (the safe flag prevents the loop from such an action). The garbage collector is realized as a task. A task can request to be invoked only at a specified time by setting the time field in the task descriptor. The time is measured according to Oberon.Time() at tick frequency specified by Input.TimeUnit. After each handler invocation, the task is expected to advance the time field to the next earliest event, overwise it will never be invoked in future. It is highly recommended to use this feature by specifying for tasks that are only invoked every few ms. This will save network traffic when using Oberon in an X-Window environment.

8. The Oberon Loop
The Oberon loop is called when the Oberon system starts, and never returns until the Oberon system is left. After a trap occurs, the run-time stack is reset, and the Oberon loop is started afresh. The Oberon loop polls the mouse and keyboard for events that it broadcasts to the display space using messages. When no events are happening, background tasks are activated periodically in a round-robin fashion.

*)
BIER  :   ~        <       g 
     C  Syntax10.Scn.Fnt 01.09.2004  07:21:37  "         d      d
     C  "         d      d
     C  TimeStamps.New TextGadgets.NewStyleProc  